#include "mbed.h"

#define SAMPLES 128

I2C i2c(PB_9, PB_8);                //Initialize i2c master, where PB_9 is SDA, PB_8 is SCL
Serial pc(SERIAL_TX, SERIAL_RX);    //Init serial connection to PC
Timer t;                            //Timing and stuff

const int addr = 0x90;
float C[SAMPLES];

////////////////////////////////////////////////////////////////////////////////
//                              FUNCTIONS                                     //
////////////////////////////////////////////////////////////////////////////////

//Initialize the AD7746
void capInit(){             //Initialize EXC and CAP
    char cmd[2];
    
    cmd[0] = 0x09;          //EXC setup
    cmd[1] = 0b00001011;
    i2c.write(addr,cmd,2);
    
    cmd[0] = 0x07;          //CAP setup
    
    //De-comment the integration time desired
    //cmd[1] = 0b10000000;  //11.0ms (90.9 Hz)
    //cmd[1] = 0b10000001;  //11.9ms (83.8 Hz)
    cmd[1] = 0b10000010;  //20.0ms (50.0 Hz)
    //cmd[1] = 0b10000011;  //38.0ms (26.3 Hz)
    //cmd[1] = 0b10000100;  //62.0ms (16.1 Hz)
    //cmd[1] = 0b10000101;  //77.0ms (13.0 Hz)
    //cmd[1] = 0b10000110;  //92.0ms (10.9 Hz)
    //cmd[1] = 0b10000111;  //109.6ms (9.1 Hz)
    i2c.write(addr,cmd,2);
}

//Initialize the CAPDAC
void capdac(){
    char cmd[2];
    
    cmd[0] = 0x0B;
    cmd[1] = 0x80 | 30;
    i2c.write(addr,cmd,2);
}

//Check if the AD7746 is ready to do another measurement
bool checkRdy(){
    uint8_t donebit;
    
    i2c.start();            //Read register 0x00 (status), mask & shift RDY bit
    i2c.write(addr & 0xFE);
    i2c.write(0x00);
    i2c.start();
    i2c.write(addr | 0x01);
    donebit = i2c.read(0);
    i2c.stop();
    donebit = (bool)((donebit & 0b00000100) >> 2);
    return donebit;
}

//Take a single capacitance measurement
float capRead(){
    char cmd[2];
    uint8_t lb1, lb2, lb3;
    uint32_t lb;
    float result;
    
    cmd[0] = 0x0A;          //Set up a single measurement
    cmd[1] = 0b00000010;
    i2c.write(addr,cmd,2);
    
    while(checkRdy()){      //Check status register if measurement is done
        ;
    }
    
    i2c.start();            //Read 3 bytes
    i2c.write(addr & 0xFE);
    i2c.write(0x01);
    i2c.start();
    i2c.write(addr | 0x01);
    lb1 = i2c.read(1);
    lb2 = i2c.read(1);
    lb3 = i2c.read(0);
    i2c.stop();
    
    lb = lb1*65536+lb2*256+lb3;     //Calculate capacitance
    result = (lb-8388608)*4.096/8388608;
    
    //pc.printf("lb1 = %d, lb2 = %d, lb3 = %d\n",lb1,lb2,lb3);
    //pc.printf("lb = %d\n",lb);
    //pc.printf("C = %f\n",result);
    
    return result;
}

//Print over serial
void printCap(){
    for(uint16_t i = 0; i < SAMPLES; i++){
        if(i == SAMPLES-1){
            pc.printf("%f\n",C[i]);
        }
        else{
            pc.printf("%f,",C[i]);
        }
    }
}

////////////////////////////////////////////////////////////////////////////////
//                                   MAIN                                     //
////////////////////////////////////////////////////////////////////////////////

int main(){
    pc.baud(115200);
    capInit();
    capdac();
    
    while(1){        
        for(uint16_t i = 0; i < SAMPLES; i++){
            C[i] = capRead();
        }
        printCap();
    }
}