// Project: Nokia5110 - Controlling a NK5110 display from an NXP LPC1768
// File: NOKIA_5110.h
// Author: Chris Yan
// Created: January, 2012
// Revised: January, 2014
//  Desc: Commands, fonts, and class for using a
//      Nokia 5110 LCD via the Phillips 8554 LCD driver.
// 
//  Typical Usage: User must fill the LcdPins struct with the pinout used to control the LCD and
//      instantiate the NokiaLcd class - passing the created LcdPins struct to the constructor.
//      The class function NokiaLcd::InitLcd may then be called to reset and start the LCD driver.
//      A simple 6x6 font (6x8 in LCD space and ~5x5 character space) is included to facilitate 
//      the NokiaLcd::DrawChar( char character ) function, which will copy the character 8 bits 
//      at a time for 6 clock cycles.
//                 Commands may be sent to the LCD via the NokiaLcd::SendFunction(char cmd) 
//      function, but be aware that certain commands require the Function Set register's H-value
//      to be either 1 or 0, depending on the command. This class does not check to see whether
//      the H-value is of proper status. The Function Set register /may/ be changed via the 
//      NokiaLcd::SendFunction(char cmd), but the code uses this internally and expects that
//      most function registers have not been changed by the user.
//
//      Example:
//          #include "mbed.h"
//          #include "NOKIA_5110.h"
//
//          int main() {
//              LcdPins myLcdPins = { p11, NC, p13, p10, p8, p9 };
//              NokiaLcd myLcd( myLcdPins );    // SPI is started here (8-bits, mode 1)
//              myLcd.InitLcd();                // LCD is reset and DDRAM is cleared
//              myLcd.TestLcd( 0xAA );          // Draws a vertical pattern where every other pixel is on 
//              wait(10);                       
//              myLcd.ShutdownLcd();            // Clears the LCD's DDRAM and powers it down via CMD_FS_POWER_DOWN_MODE, H=0
//              while(1)
//              {   };
//          }

// Command Instructions
//       H = 0
#ifndef __NOKIA_5110_H__
#define __NOKIA_5110_H__

// Command Instructions
//       H = 0
#define CMD_DC_CLEAR_DISPLAY   0x08
#define CMD_DC_NORMAL_MODE     0x0C
#define CMD_DC_FILL_DISPLAY    0x09
#define CMD_DC_INVERT_VIDEO    0x0D
#define CMD_FS_HORIZONTAL_MODE 0x00
#define CMD_FS_VERTICAL_MODE   0x02
#define CMD_FS_BASIC_MODE      0x00
#define CMD_FS_EXTENDED_MODE   0x01
#define CMD_FS_ACTIVE_MODE     0x00
#define CMD_FS_POWER_DOWN_MODE 0x04
//       H = 1
#define CMD_TC_TEMP_0          0x04
#define CMD_TC_TEMP_1          0x05
#define CMD_TC_TEMP_2          0x06
#define CMD_TC_TEMP_3          0x07
#define CMD_BI_MUX_24          0x15
#define CMD_BI_MUX_48          0x13
#define CMD_BI_MUX_100         0x10
#define CMD_VOP_6V06           0xB2
#define CMD_VOP_7V38           0xC8

// LCD Characteristics
#define LCD_FREQ 1000000
#define LCD_SPI_MODE 0x01
#define LCD_SPI_BITS 0x08
#define LCD_X_MAX 84
#define LCD_Y_MAX 48

#define PIN_RST  0x00
#define PIN_SCE  0x01
#define PIN_DC   0x02

#include "mbed.h"

struct LcdPins
{
    PinName mosi;
    PinName miso;
    PinName sclk;
    PinName dc;
    PinName sce;
    PinName rst;
};

struct LcdFunctionSet
{
    char PD;
    char V;
    char H;
};

typedef char LcdFunctionChar;
typedef char LcdTempControl;
typedef char LcdDispControl;
typedef char LcdBiasChar;
typedef char LcdVopChar;

class NokiaLcd
{
    public:
        NokiaLcd(LcdPins lcd_pinout);
        ~NokiaLcd();
        
    public:
        void InitLcd();
        void ClearLcdMem();
        void ShutdownLcd();
        void SendFunction(char cmd);
        void TestLcd(char test_pattern);
        void SendDrawData(char data);
        
    public:
        void DrawString(char* str);
        void DrawChar(char character);
        void SetXY(char x, char y);
        void DrawFrameChar(char character);
        void DrawNegFrameChar(char character);
        char* NumToStr(int num);
        
    private:
        char CreateFunctionChar();
        void ResetLcd();
        
    private:
        LcdFunctionChar FunctionChar;
        LcdTempControl  TempControlChar;
        LcdDispControl  DispControlChar;
        LcdFunctionSet  FunctionSet;
        LcdBiasChar     BiasChar;
        LcdVopChar      VopChar;
        DigitalOut**    Pins;
        SPI*            LcdSpi;
        
};

const char FONT_6x6[570] = //should be 564 total char
{
 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // SPACE   1   
 0x00, 0x06, 0x2F, 0x06, 0x00, 0x00, // !   2
 0x00, 0x06, 0x00, 0x06, 0x00, 0x00, // "   3
 0x14, 0x3E, 0x14, 0x3E, 0x14, 0x00, // #   4
 0x2E, 0x2A, 0x3F, 0x2A, 0x3A, 0x00, // $   5
 0x26, 0x16, 0x08, 0x34, 0x32, 0x00, // %   6
 0x34, 0x2A, 0x3C, 0x18, 0x28, 0x00, // &   7
 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, // '   8
 0x00, 0x00, 0x1C, 0x36, 0x22, 0x00, // (   9
 0x22, 0x36, 0x1C, 0x00, 0x00, 0x00, // )   10
 0x24, 0x18, 0x0E, 0x18, 0x24, 0x00, // *   11
 0x08, 0x08, 0x3E, 0x08, 0x08, 0x00, // +   12
 0x20, 0x30, 0x00, 0x00, 0x00, 0x00, // ,   13
 0x08, 0x08, 0x08, 0x08, 0x08, 0x00, // -   14
 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, // .   15
 0x30, 0x18, 0x0C, 0x06, 0x00, 0x00, // /   16
 0x00, 0x1C, 0x22, 0x22, 0x1C, 0x00, // 0   17
 0x00, 0x24, 0x3E, 0x20, 0x00, 0x00, // 1   18
 0x3A, 0x2A, 0x2A, 0x2A, 0x2E, 0x00, // 2   19
 0x22, 0x2A, 0x2A, 0x2A, 0x3E, 0x00, // 3   20
 0x0E, 0x08, 0x08, 0x3E, 0x08, 0x00, // 4   21
 0x2E, 0x2A, 0x2A, 0x2A, 0x3A, 0x00, // 5   22
 0x3E, 0x2A, 0x2A, 0x2A, 0x3A, 0x00, // 6   23
 0x22, 0x12, 0x0A, 0x06, 0x02, 0x00, // 7   24
 0x3E, 0x2A, 0x2A, 0x2A, 0x3E, 0x00, // 8   25
 0x00, 0x2E, 0x2A, 0x2A, 0x3E, 0x00, // 9   26
 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, // :   27
 0x00, 0x20, 0x14, 0x00, 0x00, 0x00, // ;   28
 0x00, 0x00, 0x08, 0x14, 0x22, 0x00, // <   29
 0x14, 0x14, 0x14, 0x14, 0x14, 0x00, // =   30
 0x22, 0x14, 0x08, 0x00, 0x00, 0x00, // >   31
 0x06, 0x01, 0x2D, 0x06, 0x00, 0x00, // ?   32
 0x1E, 0x23, 0x19, 0x35, 0x3E, 0x00, // @   33
 0x3C, 0x0A, 0x0A, 0x0A, 0x3C, 0x00, // A   34
 0x3E, 0x2A, 0x2A, 0x2A, 0x1C, 0x00, // B   35
 0x1C, 0x22, 0x22, 0x22, 0x22, 0x00, // C   36
 0x3E, 0x22, 0x22, 0x22, 0x1C, 0x00, // D   37
 0x3E, 0x2A, 0x2A, 0x2A, 0x22, 0x00, // E   38
 0x3E, 0x0A, 0x0A, 0x0A, 0x02, 0x00, // F   39
 0x1C, 0x22, 0x2A, 0x2A, 0x18, 0x00, // G   40
 0x3E, 0x08, 0x08, 0x08, 0x3E, 0x00, // H
 0x22, 0x22, 0x3E, 0x22, 0x22, 0x00, // I
 0x10, 0x22, 0x22, 0x1E, 0x02, 0x00, // J
 0x3E, 0x08, 0x14, 0x22, 0x00, 0x00, // K
 0x00, 0x3E, 0x20, 0x20, 0x20, 0x00, // L   45
 0x3E, 0x04, 0x08, 0x04, 0x3E, 0x00, // M
 0x3C, 0x02, 0x02, 0x02, 0x3C, 0x00, // N
 0x1C, 0x22, 0x22, 0x22, 0x1C, 0x00, // O
 0x3E, 0x0A, 0x0A, 0x04, 0x00, 0x00, // P
 0x1C, 0x22, 0x32, 0x3C, 0x20, 0x00, // Q   50
 0x3E, 0x0A, 0x0A, 0x1A, 0x24, 0x00, // R
 0x24, 0x2A, 0x2A, 0x2A, 0x12, 0x00, // S
 0x02, 0x02, 0x3E, 0x02, 0x02, 0x00, // T
 0x1E, 0x20, 0x20, 0x20, 0x1E, 0x00, // U
 0x06, 0x18, 0x20, 0x18, 0x06, 0x00, // V   55
 0x0E, 0x30, 0x18, 0x30, 0x0E, 0x00, // W
 0x22, 0x14, 0x08, 0x14, 0x22, 0x00, // X
 0x02, 0x04, 0x38, 0x04, 0x02, 0x00, // Y
 0x22, 0x32, 0x2A, 0x26, 0x22, 0x00, // Z
 0x00, 0x00, 0x00, 0x3E, 0x22, 0x00, // [   60
 0x06, 0x0C, 0x18, 0x30, 0x00, 0x00, // backslash
 0x22, 0x3E, 0x00, 0x00, 0x00, 0x00, // ]
 0x00, 0x04, 0x02, 0x02, 0x04, 0x00, // ^
 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, // _
 0x00, 0x00, 0x04, 0x08, 0x00, 0x00, // `   65
 0x18, 0x24, 0x14, 0x38, 0x00, 0x00, // a
 0x1E, 0x28, 0x28, 0x10, 0x00, 0x00, // b
 0x18, 0x24, 0x24, 0x00, 0x00, 0x00, // c
 0x10, 0x28, 0x28, 0x1E, 0x00, 0x00, // d
 0x18, 0x2C, 0x2C, 0x08, 0x00, 0x00, // e   70
 0x00, 0x3C, 0x12, 0x04, 0x00, 0x00, // f
 0x24, 0x2A, 0x1E, 0x00, 0x00, 0x00, // g
 0x3E, 0x08, 0x30, 0x00, 0x00, 0x00, // h
 0x00, 0x3A, 0x00, 0x00, 0x00, 0x00, // i
 0x10, 0x20, 0x1A, 0x00, 0x00, 0x00, // j   75
 0x3E, 0x10, 0x2C, 0x20, 0x00, 0x00, // k
 0x00, 0x3E, 0x00, 0x00, 0x00, 0x00, // l
 0x38, 0x08, 0x18, 0x08, 0x30, 0x00, // m
 0x30, 0x08, 0x08, 0x30, 0x00, 0x00, // n
 0x10, 0x28, 0x28, 0x10, 0x00, 0x00, // o   80
 0x38, 0x14, 0x14, 0x08, 0x00, 0x00, // p
 0x08, 0x14, 0x14, 0x38, 0x00, 0x00, // q
 0x3C, 0x08, 0x04, 0x00, 0x00, 0x00, // r
 0x2C, 0x34, 0x00, 0x00, 0x00, 0x00, // s
 0x08, 0x3C, 0x28, 0x00, 0x00, 0x00, // t   85
 0x18, 0x20, 0x20, 0x18, 0x00, 0x00, // u
 0x08, 0x10, 0x20, 0x10, 0x08, 0x00, // v
 0x18, 0x20, 0x10, 0x20, 0x18, 0x00, // w
 0x28, 0x10, 0x28, 0x00, 0x00, 0x00, // x
 0x2C, 0x30, 0x1C, 0x00, 0x00, 0x00, // y   90
 0x24, 0x34, 0x2C, 0x24, 0x00, 0x00, // z
 0x00, 0x00, 0x08, 0x3E, 0x22, 0x00, // {
 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x00, // |
 0x22, 0x3E, 0x08, 0x00, 0x00, 0x00, // }
 0x10, 0x08, 0x18, 0x10, 0x08, 0x00, // ~   95
};

#endif
