#include "mbed.h"
#include "Communication.hpp"    // 通信ライブラリ

/* 定数設定 */
#define SERIAL_BAUDRATE 115200

#define THRUSTER_HH 1950
#define THRUSTER_HL 1600
#define THRUSTER_NN 100
#define THRUSTER_LH 1400
#define THRUSTER_LL 1300
#define SERVO_H 2500
#define SERVO_N 1500
#define SERVO_L 550

#define SERVO_ROTATE_SPEED 50
#define BLDC_ACCELE_SPEED 50

/* ピン初期設定 */
Serial pc(USBTX, USBRX);
DigitalOut myled(LED1);

PinName rs485_txd = PA_9;
PinName rs485_rxd = PA_10;
PinName rs485_nre = PA_12;
PinName rs485_de = PA_11;

AnalogIn vr1_thruster1_power(PA_4); // A3
AnalogIn vr2_thruster2_power(PA_5); // A4
AnalogIn vr3_thruster3_power(PA_6); // A5
AnalogIn vr4_thruster3_direction(PA_7); // A6

DigitalOut led1_is_working(PB_7);   //D4
DigitalOut led2r_bad_connection(PB_6);   //D5
DigitalOut led2g_good_connection(PB_1);   //D6
DigitalOut led3_is_outputting(PF_0);  // D7

DigitalIn sw1_is_connecting(PF_1);  // D8
DigitalIn sw2_is_outputting(PA_8);  // D9

Communication communication(rs485_txd,rs485_rxd,rs485_de,rs485_nre);

/* プロトタイプ宣言 */
void ioInit();
void waitSwitchInitialPosition();
void serialInit();
void serialReceive();
void updateCommunication();

/* 変数宣言 */
bool is_start = false;
int u[6];
int operation_command;  // 大事な命令があるときにそのCommnadが代入
int voltage_int;        // バッテリー電圧整数値
int current_int;        // バッテリー電流整数値
int power_supply;       // 強電供給状態
int rotating;           // 回転中？
int accelerating;       // 加速中？

/* メインループ */
int main()
{
    ioInit();
    serialInit();
    //waitSwitchInitialPosition();
    //while(is_start){
    //    pc.printf("please press u to start\n");
    //    wait(1.0);
    //}
    //led2r_bad_connecion = 1;
    //led2g_good_connection = 1;
    communication.init(Communication::ROLL_LAND);
    u[0] = THRUSTER_NN;
    u[1] = THRUSTER_NN;
    u[2] = THRUSTER_NN;
    u[3] = SERVO_N;
    u[4] = SERVO_N;
    u[5] = SERVO_N;
       
    while(1) {
        if (rotating != 0){
            if ((u[3]>SERVO_L) && (u[4]>SERVO_L) && (u[5]>SERVO_L) && (u[3]<SERVO_H) && (u[4]<SERVO_H) && (u[5]<SERVO_H)) {
                u[3] += rotating*SERVO_ROTATE_SPEED;
                u[4] += rotating*SERVO_ROTATE_SPEED;
                u[5] += rotating*SERVO_ROTATE_SPEED;
            }else{
                rotating = 0;
            }
        }
        if (accelerating != 0){
            if ((u[0]>THRUSTER_NN) && (u[1]>THRUSTER_NN) && (u[2]>THRUSTER_NN) && (u[0]<THRUSTER_HL) && (u[1]<THRUSTER_HL) && (u[2]<THRUSTER_HL)) {
                u[0] += accelerating*BLDC_ACCELE_SPEED;
                u[1] += accelerating*BLDC_ACCELE_SPEED;
                u[2] += accelerating*BLDC_ACCELE_SPEED;
            }else{
                accelerating = 0;
            }
        }
        communication.freshSendNumDat();
        if (operation_command == 0) {    // 特に大事な命令はない
            communication.send_command_dat = Command::SIGNAL_LAND_TO_MAIN;  // 入力命令
            for(int i=0; i<6; i++){
                communication.send_num_dat.push(u[i]);
            }
        }else{
            communication.send_command_dat = operation_command; // 大事な命令を送信
            operation_command = 0;
        }
        communication.encode();
        communication.sendDat();
        //for (int t = 0; t<20; t++){
        for (int t = 0; t<10; t++){
            updateCommunication();  // 通信受信処理
            //wait_ms(20);
            wait_ms(10);
        }
        pc.printf(", power : %4d us, ", u[0]);
    }
    
    return 0;
}

/* IOピンまわり */
void ioInit(void) {
    led1_is_working = 1;
    led2r_bad_connection = 0;
    led2g_good_connection = 0;
    led3_is_outputting = 0;
    
    sw1_is_connecting.mode(PullUp);
    sw2_is_outputting.mode(PullUp);
}

void waitSwitchInitialPosition(void) {
    while (sw1_is_connecting == 0 || sw2_is_outputting == 0) {  // 2つのスイッチはオフになってないと次に進ませない
        pc.printf("[WARNING] Please Off the switches !! \n");
        led2r_bad_connection =~ led2r_bad_connection;
        wait(1);
    }
}

/* シリアルまわり */
void serialInit(void) {
    pc.baud(SERIAL_BAUDRATE);
    pc.attach(serialReceive, Serial::RxIrq);
    pc.printf("Land Operator\n Now Set up...\n");
}

void initControlInput(void){
    u[0] = THRUSTER_NN;
    u[1] = THRUSTER_NN;
    u[2] = THRUSTER_NN;
    u[3] = SERVO_N;
    u[4] = SERVO_N;
    u[5] = SERVO_N;
    rotating = 0;
    accelerating = 0;
}

void serialReceive(void) {
    char c = pc.getc();
    
    switch(c) {
        
        case 8:    // BackSpace
        operation_command = Command::POWER_SUPPLY_STOP;
        break;
        case 13:    // ENTER        
        operation_command = Command::POWER_SUPPLY_START;
        break;
        
        case 'm':
        u[0] = 1480;
        break;
        
        case 'v':       // vertical 方向に回転
        rotating = 1;
        break;
        case 'h':
        rotating = -1;  // horizon 方向に回転
        break;
        
        case 'u':       // 加速
        //accelerating = 1;
        u[0] = u[0] + 50;
        break;
        case 'd':       // 減速
        //accelerating = -1;
        u[0] = u[0] - 50;
        break;
        case '0':
        //initControlInput();
        u[0] = THRUSTER_LH;
        break;
        case '1':
        initControlInput();
        u[1] = THRUSTER_HL;
        break;
        case '2':
        initControlInput();
        u[2] = THRUSTER_HL;
        break;
        case '3':
        initControlInput();
        u[3] = SERVO_H;
        rotating = 0;
        break;
        case '4':
        initControlInput();
        u[4] = SERVO_H;
        rotating = 0;
        break;
        case '5':
        initControlInput();
        u[5] = SERVO_H;
        rotating = 0;
        break;
        
        case '6':
        initControlInput();
        u[3] = SERVO_L;
        u[4] = SERVO_H;
        u[5] = SERVO_H;
        rotating = 0;
        break;
        
        case '7':
        initControlInput();
        u[3] = SERVO_L;
        u[4] = SERVO_H;
        u[5] = SERVO_N;
        rotating = 0;
        break;
        
        case '8':
        initControlInput();
        u[5] = SERVO_H;
        rotating = 0;
        break;
        
        case '9':
        //initControlInput();
        u[0] = THRUSTER_HH;
        rotating = 0;
        break;
        
        default:
        initControlInput();
        break;
    }
}

/* rs485まわり */
void updateCommunication(void) {
    if (communication.flag_rdat_check){
        communication.decode();
        communication.flag_rdat_check = false;
        switch (communication.receive_command_dat){
            case Command::SIGNAL_BATT_TO_LAND:
                voltage_int = communication.receive_num_dat.front();
                communication.receive_num_dat.pop();
                current_int = communication.receive_num_dat.front();
                communication.receive_num_dat.pop();
                power_supply = communication.receive_num_dat.front();
                communication.receive_num_dat.pop();
                pc.printf("BATT:%2.1f V,%2.1f A, POWER:%2d\n", (float)(voltage_int)/10.0f, (float)(current_int)/10.0f, power_supply);
                break;
            default:
                break;
        }
    }
}