 /*
 *  PCF2129 mbed library
 *
 *  @author     king33jp , imAkichi
 *  @version    0.1
 *  @date       13-Jul-2015
 *
 *  datasheet:
 *  http://www.jp.nxp.com/products/interface_and_connectivity/real_time_clocks/rtcs_with_temp_compensation/PCF2129T.html
 *
 */

#ifndef        MBED_PCF2129
#define        MBED_PCF2129

#include    "mbed.h"

/** PCF2129 class
 *
 *  PCF2129AT is RTC IC.
 *  datasheet is followed.
 *  http://www.jp.nxp.com/products/interface_and_connectivity/real_time_clocks/rtcs_with_temp_compensation/PCF2129T.html
 *  Example:
 *  @code
 *  #include "mbed.h"
 *  #include "PCF2129.h"
 *  
 *  PCF2129 rtc( PTE0, PTE1 );  // sda,scl 
 *  
 *  int main()
 *   struct tm dt,*dtp;
 *   dtp = &dt;
 *   dt.tm_sec     = 0;
 *   dt.tm_min     = 0;
 *   dt.tm_hour    = 23;
 *   dt.tm_mday    = 25;
 *   dt.tm_mon     = 12 - 1;     // Janualy=0
 *   dt.tm_year    = 115;        // 1900 is start. PCF2129 is start 2000
 *    
 *   t   = mktime( &dt );
 *   dtp = localtime(&t);
 *
 *  {
 *      if ( rtc.is_init_required() ) {
 *          rtc.set_time( dtp );
 *      }
 *  
 *      while ( 1 ) {
 *          t   = rtc.time( NULL );
 *          printf( "%s\r", ctime( &t ) );
 *          wait( 0.25 );
 *      }
 *  }
 *  @endcode
 */

class PCF2129
{
public:
    /** Error code */
    typedef enum {
        NO_ERROR                = 0,
        CLOCK_INTEGRITY_FAIL    = 1,
        I2C_ACCESS_FAIL         = 2,
        TIME_FUNC_ERROR         = ((time_t)-1)
    }
    ErrorNum;

    /** Create a PCF2129 instance connected to specified I2C pins with specified address
     *
     *  @param I2C_sda      I2C-bus SDA pin
     *  @param I2C_scl      I2C-bus SCL pin
     */
    PCF2129( PinName sda, PinName scl );

    /** Destractor
     */
    ~PCF2129();


    /** Clock integrity check
     *
     *  @return     non-zero value if the clock was stopped (means need to set the time)
     */
    int     is_init_required( void );

    /** Set the time
     *
     *  @param dtp  Pointer to struct tm
     *  @return     Error code (NO_ERROR==0)
     */
    int     set_time( struct tm *dtp );

    /** Set the time
     *
     *  @param tp   pointer to time_t
     *  @return     Error code (NO_ERROR==0)
     */
    int     set_time( time_t *tp );

    /** Get time of day
     *
     *  This function works similar to "time()" in standard-C-library
     *
     *  @param tp   Pointer to time_t
     *  @return     Error code (NO_ERROR==0)
     */
    time_t  get_time( time_t *tp );

    char    Dec2RtcData(char tmp);
    char    RtcData2Dec(char tmp);

private:
    int     init( char vControl_1, char vControl_2, char vControl_3, char vCLKOUT_ctl );
    int     set_register( char reg_addr, char reg_data );
    int     read_register( char reg_addr );
   
    I2C     m_i2c;
}
;
#endif  // end of "#ifndef MBED_PCF2129"
