#include <cmath>
#include "math.h"
#define M_PI 3.141592653589             //Used to avoid typing the number pi over and over again
#include "QEI.h"
#include <algorithm>
#include "BiQuad.h"


DigitalOut led(LED_RED);
DigitalOut led_blue(LED_BLUE);

//Kinematic Variables____________________________________________________________________________________________
float L = 350;                              //Arm length
float Velocity_x = 0,Velocity_y = 0;        //Linear desired velocities
float q1 = 0, q1c = 0,q2 = 0;               //Desired angles, where q1c is 16 times q1 because a 1/16 gearbox is used
float w1 = 0, w2 = 0;                       //Desired angular velocity
float sample_time = 0.00125;                //sample time for the tickers used
float determinant = 0;                      //variable used to store the determinant
float q1_min = -50,q1_max = 50;             //Max and minimum q1
float q2_min = -50,q2_max = 50;             //Max and min q2
float q1_fixed = 0.0, q1_singularity = 1.0; //Calculation of the determinand when q1=q2
float speed = 3;                            //Motorspeed

//PID Variables__________________________________________________________________________________________________
float error1 = 0, error1_prev = 0, error1_der = 0, error1_int = 0;  //PID error for motor1
float error2 = 0,error2_prev = 0, error2_der = 0, error2_int = 0;   //PID error for motor2
float cur_angle1 = 0, cur_angle2 = 0;                               //Current angle of the motor
float countsm1 = 0, countsm2 = 0;                                   //Counts of the rotary encoder
float kp1 = 0.05;                                                   //PID controller values motor1
float kp2 = 0.07, kd2 = 0.0015;                                     //PID controller values motor2


//Configuration Variables________________________________________________________________________________________
bool button_left_prev= true;                //Switching boolean
double MEANminimum_left=0;                  //Configuration minimum value for left arm
double MEANmaximum_left=0;                  //Configuration maximum value for left arm
double MEANminimum_right=0;                 //Configuration minimum value for right arm
double MEANmaximum_right=0;                 //Configuration minimum value for right arm

int counter_min_left = 1;                   //Counter for minimum value configuration left arm
int counter_max_left = 1;                   //Counter for Maximum value configuration left arm
int counter_min_right = 1;                  //Counter for minimum value configuration right arm
int counter_max_right = 1;                  //Counter for maximum value configuration left arm

double Moving_average_left[1000];           //Storeage for moving average filter left arm
double Moving_average_right[1000];          //Storeage for moving average filter right arm
double results_left = 0;                    //Temperory storage before averaging the MOVAG
double results2_left = 0;                   //"                                         "
double results_right = 0;                   //"                                         "
double results2_right = 0;                  //"                                         "
double MOVAG_left=0;                        //Moving average left
double MOVAG_right=0;                       //"             " right                 
int cnt=1;                                  //Counter for states


//Tickers_______________________________________________________________________________________________________
Ticker EMG;             //Filter and calibrate on incomign EMG signal
Ticker Angle;           //Calculate desired angle
Ticker Encoder;         //Get current motor angles
Ticker PID;             //Calculate motorspeed out of errors using PID

//Input and output______________________________________________________________________________________________
QEI Encoderm1(D12,D13,NC,32);           //Motor1 encoder
DigitalOut motor1_direction(D4);        //Motor1 direction (true or false)
PwmOut motor1_speed(D5);                //Motor1 speed (pwm value between 0 and 1)

QEI Encoderm2(D10,D11,NC,32);           //Motor2 encoder
DigitalOut motor2_direction(D7);        //Motor2 direction (true or false)
PwmOut motor2_speed(D6);                //Motor2 speed (pwm value between 0 and 1)

DigitalIn button_right(SW2);
DigitalIn button_left(SW3);

AnalogIn emg0(A0);                      //used to read out the sensors on the left biceps
AnalogIn emg1(A1);                      //used to read out the sensors on the right biceps
    
BiQuad bq1(0.9800945861100181,-1.9601891722200362,0.9800945861100181,-1.9597054353584475,0.9606729090816247);//BiQuad class that represents a Highpass biquad filter with a cut-off of 5Hz and a sample freq of 500Hz.  leftbiceps                 values of the filters are determined with the site 
BiQuad bq2(0.9800945861100181,-1.9601891722200362,0.9800945861100181,-1.9597054353584475,0.9606729090816247);//BiQuad class that represents a Highpass biquad filter with a cut-off of 5Hz and a sample freq of 500Hz.  rightbiceps
BiQuadChain bqc; //BiQuad chain that represents a cascade of biquad filters 


float min(float a, float b){  
/*This function returns the lowest value (float) of the 2 intput values (float) */
 if(b<a){ return b;} else {return a;}     
}


 
void calculate_angle(float Omega1, float Omega2){
/*This function calculates the new desired angle, by numirical integrating the input arguments omega 1 and 2. It also limits the angles to the max and min values*/  
     
    q1 = q1 + sample_time*Omega1;       // increment angle of the lower arm 
    q2 = q2 + sample_time*Omega2;       // increment he angle of the upper arm, which is the same as the angle of the motor
    q1c = q1c + sample_time*Omega1*16;  // increment the angle of the motor which controls the lower arm, this ahas a amplification of 16 due to the gearbox
        
    /* create upper en lower limits for the angles */
        
    if(q1<=q1_min){                     //If next desired angle is smaller than the desired minimum
        q1 = q1_min;                    //Angle is set to minimum
        q2 = q2 - sample_time*Omega2;   //Other angle is kept at sameplace by counteracting the integration done above
    }                                   //Same works for maximum and other angles
    else if(q1 >= q1_max){
        q1 = q1_max;
        q2 = q2 - sample_time*Omega2;
    }
    
    if(q1c<=q1_min*16){
        q1c= q1_min*16;
    }  
    else if(q1c>= q1_max*16){
        q1c= q1_max*16;
    }
        
    if(q2 <= q2_min){
        q2 = q2_min;
        q1 = q1- sample_time*Omega1;
        q1c = q1c - sample_time*Omega1*16;
    } 
    else if(q2 >= q2_max){
        q2 = q2_max;
        q1 = q1 - sample_time*Omega1;
        q1c = q1c- sample_time*Omega1*16;
    }
        
}

void calculate_omega(){
/*This function calculates the desired angular velocity of q1 and q2 which are needed to move in a straigt line in the x or y direction depending on velocity x and y*/
    
    if (q1 ==q2)           // if the values are equel an very high determinand will be calculated, this is the safeguard to prevent that from happening
        {
        q1_fixed = q1 + q1_singularity;    // This line is to fix the singularity at q1 = q2 where the determinant becomes zero
        determinant = pow(L, 2) * cos(M_PI / 180 * q1_fixed) * sin(M_PI / 180 * q2) - pow(L, 2) * sin(M_PI / 180 * q1_fixed) * cos(M_PI / 180 * q2);    // calculation of determinant
        }    else {
        determinant = pow(L, 2) * cos(M_PI / 180 * q1) * sin(M_PI / 180 * q2) - pow(L, 2) * sin(M_PI / 180 * q1) * cos(M_PI / 180 * q2);    // calculation of determinant
    }
    
    w1 = (Velocity_x / determinant) * -L * sin(M_PI / 180 * q2) + (Velocity_y / determinant) * L * cos(M_PI / 180 * q2);    // calculation of angular velocity 1
    w2 = (Velocity_x / determinant) * L * sin(M_PI / 180 * q1) + (Velocity_y / determinant) * -L * cos(M_PI / 180 * q1);    // calculation of angular velocity 2
    w1 = w1*(180/M_PI);     // convert velocity to degrees
    w2 = w2*(180/M_PI);     // convert velocity to degrees
    
    // if angular speed is behond safety limits, set it to a lower value, this is just a safety procaution
    if(w1>=7){
        w1 = 1;
    }              
    else if(w1<= -7){ 
        w1 = -1;
    }
    
    if(w2>=7){
        w2 = 1;
    }
    else if(w2<=-7){
        w2 = -1;
    }
    
    calculate_angle(w1,w2);   //Call calculate angle with the new desired omega 1 and 2
    }
    
void PID_control(){
/*This function calculates the correct duty cycle of the PWM powered motors with use of PID and the error between the desired and current angle*/

    error1 = q1c - cur_angle1;      //Calculate current error of motor1
         
    error2 = q2 - cur_angle2;       //Calculate current error of motor2
    error2_der = min((error2 - error2_prev)/sample_time,250);   //Calculate numirical derivative of current angle
    error2_prev = error2;           //Move current error to previous error for next calculation
    
    //If the error is negative or positive, direction is set to true or false and the new motor PWM value is calculated and set
    if(error1 < 0){                 
        motor1_direction = true;
        motor1_speed = min(-(error1*kp1),1);
    } 
    else if(error1 > 0) {
        motor1_direction = false;
        motor1_speed = min((error1*kp1),1);
    }    
    
    if(error2< 0){
        motor2_direction = false;
        motor2_speed = min(-(error2*kp2 + error2_der*kd2),1);
    } 
    else if(error2 > 0){
        motor2_direction = true;
        motor2_speed = min((error2*kp2 + error2_der*kd2),1);
    }  
    
}

    
void encoder(){
/*This function gets the pulses out of the encoder and transforms this to the current angle of the motor*/
    //Motor 1
    countsm1 = Encoderm1.getPulses();                     
    cur_angle1 = (countsm1*90) / 1048;
    //Motor 2
    countsm2 = Encoderm2.getPulses();
    cur_angle2 = (countsm2*90) / 1048;
    
}



void emg(){
/*
This function first filters the EMG signal and calibrates the system for the user. 
The maximum and minimum left and right are stored and used to calculate the treshold values.
When calibrating is done, it filters the EMG and sets the x and y velocities with these filtered signals. 
*/
    
    led_blue=1; //Reference for the user it is indeed measuring the EMG

    // apply a highpass filter and a rectifier to the left biceps signal
    double bicepsleft = emg0.read();                                //read the analog input of emg0 (bicepsleft)
    double highpassfilter_bicepsleft = bq1.step(bicepsleft);        //apply a highpass filter to the emg signal that is read
    double rectify_bicepsleft = fabs(highpassfilter_bicepsleft);    //rectify the highpass filtered emg signal
                             
    //Apply a moving average filter on the high-pass filtered and rectified signal for the left biceps
    MOVAG_left=0;                                           //Reset variable to 0   
    for(int i = 999;i>0;i--){                               //Loop trough all moving average left values
      MOVAG_left = MOVAG_left + Moving_average_left[i];     //Sum al up into MOVAG left
      Moving_average_left[i] = Moving_average_left[i-1];    //Move al values one place higher
    }   
    Moving_average_left[0]=rectify_bicepsleft;              //Fill moving average left 0 with new value
    MOVAG_left = MOVAG_left + Moving_average_left[0];       //Sum this one too to MOVAG
    MOVAG_left = MOVAG_left/1000;                           //Devide by amount of samples to create an average
      

    // apply a highpass filter and a rectifier to the right biceps signal
    double bicepsright = emg1.read(); //read the analog input of emg1(bicepsright)
    double highpassfilter_bicepsright = bq2.step(bicepsright); //apply a highpass filter to the emg signal that is read
    double rectify_bicepsright= fabs(highpassfilter_bicepsright); //rectify the highpass filtered emg signal

    //Same is done as with left arm above
    MOVAG_right = 0;
    for(int i = 999;i>0;i--){
    MOVAG_right = MOVAG_right + Moving_average_right[i];
    Moving_average_right[i] = Moving_average_right[i-1];
    }
    Moving_average_right[0]=rectify_bicepsright;
    MOVAG_right = MOVAG_right + Moving_average_right[0];
    MOVAG_right = MOVAG_right/1000;
    
    //Case switched when previous action was the left button pressed which means measuring calibrate value and now the button is pressed.
    if(cnt<=4 && button_left==0 && button_left_prev == true)
    { 
        cnt++;
        button_left_prev=false;
    }    
            
    // calibration
    /*
    Case 1: Minimum value left arm
    Case 2: Maximum value left arm
    Case 3: Minimum value right arm
    Case 4: Macimum value right arm
    Case default: Calculate desired velocity x and y
    */
        
    switch(cnt)
    {   
        case 1: 
            Velocity_y=0; //Calibrating thus velocities are zero                                                
            Velocity_x=0;
            if(counter_min_left <= 3000 && button_right == 0){  //For 3000 samples = 3 seconds
                results_left = results_left + MOVAG_left;       //Sum the moving average values calculated
                counter_min_left++;
                led=1;                                          //User interface, when on keep (de)stressing the arm depending on the case
            } 
            else if(button_right == 1 && counter_min_left != 1){//When 3 seconds are done and user wants to go to next case (pressing right button) 
                MEANminimum_left = results_left/3000;           //Calculate mean
                counter_min_left = 1;
                led=0;                                          //Sign for user to stop                                         
                results_left=0;
                button_left_prev= true; 
            }
            else if(counter_min_left >3000) { // done with calibrating
                led = !led;                   //Sign for user to stop       
                wait(0.5f);
            }
            break;
            
             
   
        //Same as case 1 now maximum for left arm
        case 2:
            Velocity_y=0; 
            Velocity_x=0;
            if(counter_max_left <= 3000 && button_right==0){
                results2_left = results2_left + MOVAG_left; 
                counter_max_left++;
                led=1;
            } 
            else if(button_right == 1 && counter_max_left != 1){
                counter_max_left = 1;
                MEANmaximum_left = results2_left/3000;
                led=0;
                results2_left=0;
                button_left_prev=true;
            }
            else if(counter_max_left >3000) {
                led = !led;
                wait(0.5f);
            }
            break;
             
        //Same as case 1 now minimum for right arm
        case 3:
            Velocity_y=0; 
            Velocity_x=0;
            if(counter_min_right <= 3000 && button_right == 0){
                results_right = results_right + MOVAG_right; 
                counter_min_right++;
                led=1;
            } 
            else if(button_right == 1 && counter_min_right != 1){
                MEANminimum_right = results_right/3000;
                counter_min_right = 1;
                led=0; 
                results_right=0;
                button_left_prev=true;
            }
            else if(counter_min_right >3000) {
                led = !led;
                wait(0.5f);
            }
            break;
             
             
  
        case 4:  
            Velocity_y=0; 
            Velocity_x=0;
            if(counter_max_right <= 3000 && button_right==0){
                results2_right = results2_right + MOVAG_right; // if the function sample is called this is increased.
                counter_max_right++;
                led=1;
            } 
            else if(button_right == 1 && counter_max_right != 1){
                counter_max_right = 1;
                MEANmaximum_right = results2_right/3000;
                led=0;
                results2_right=0;
                button_left_prev=true;
            }
            else if(counter_max_right >3000) {
                led = !led;
                wait(0.5f);
            }
            break;
             
        default:
        led_blue=0; //Calibrating done so blue led is turned off and red is turned on.
        led=1;
            //Put current stress level in a range of the calibrated max and min values
            
            if ( MOVAG_left >= MEANmaximum_left - ((MEANmaximum_left - MEANminimum_left)/3)){
                Velocity_y =speed; // muscles stressed gives a positive velocity
            }
            else if ( MOVAG_left >= MEANminimum_left + ((MEANmaximum_left - MEANminimum_left)/3) && MOVAG_left <= MEANmaximum_left - ((MEANmaximum_left - MEANminimum_left)/3)){
                Velocity_y =-1*speed; //muscles stressed slightly gives a negative velocity
            }
            else if(MOVAG_left <= MEANminimum_left +(MEANmaximum_left - MEANminimum_left)/3) {
                Velocity_y = 0;   // muscles relaxed gives a stable position (velocity 0)   
            }   
    

            if ( MOVAG_right >= MEANmaximum_right - ((MEANmaximum_right - MEANminimum_right)/3)){
                Velocity_x =speed;
            }
            else if ( MOVAG_right >= MEANminimum_right + ((MEANmaximum_right - MEANminimum_right)/3) && MOVAG_right <= MEANmaximum_right - ((MEANmaximum_right - MEANminimum_right)/3)){
                Velocity_x =-1*speed;
            }
            else if(MOVAG_right <= MEANminimum_right +(MEANmaximum_right - MEANminimum_right)/3) {
                Velocity_x = 0;      
            }   
        
    }  
}





    
int main()
{
    EMG.attach(&emg, 0.003);                    //Call the EMG function every 0.003 seconds (this is the maximum frequency the board was able to run)
    Angle.attach(&calculate_omega, sample_time);//Call angle calculation with sample time (has to be te same time as omega calculation and angle encoder measuring)
    PID.attach(&PID_control, sample_time);      //Call the controller every sample_time seconds
    Encoder.attach(&encoder,sample_time);       //Call the encoder every sample_time seconds
       
     while (true) { //Loop forever
     
    }
}

 
 
 




