#include "mbed.h"
#include "TextLCD.h"
// included in libmbed rev14
#include "CAN.h"

//enable additional serial debug messages
//#define DEBUG_CAN_TX 1
#define DEBUG_IR_RAW 1

AnalogIn ain(p20);
DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);
TextLCD lcd(p14, p15, p16, p17, p18, p19, p21, 8);

Ticker TickerSerialTx;
Ticker TickerLcdUpdate;
Ticker TickerCanTx;

// conversion from IR sensor input voltage to nominal gas value
#define CAN_GAS_MAX  4000
#define CAN_GAS_MIN  700
#define IR_MIN  0.28
#define IR_MAX  0.76
// 0.12 - 0.13   idle   update: seen 0.02 ..
// 0.76          max

// CAN defines
#define CAN_ID_EGAS_CTRL   0x700
#define CAN_DLC_EGAS_CTRL  1
#define CAN_ID_EGAS_NOMINAL_VALUE   0x701
#define CAN_DLC_EGAS_NOMINAL_VALUE  2
#define CAN_BYTE_HIGH_EGAS_NOMINAL_VALUE  1
#define CAN_BYTE_LOW_EGAS_NOMINAL_VALUE   0

// CAN_RS pin at Philips PCA82C250 can bus controller.
// activate transceiver by pulling this pin to GND.
// (Rise and fall slope controlled by resistor R_s)
// (+5V result in tranceiver standby mode)
// For further information see datasheet page 4
DigitalOut can_Pca82c250SlopePin(p28);
// We use can on mbed pins 29(CAN_TXD) and 30(CAN_RXD).
CAN can2(p30, p29);

// value which will be sent via can
unsigned short u16_CanGasNominal = CAN_GAS_MIN;
// IR sensor value
float f_IrDistance = 0.0;
// 2 can objects for TX
CANMessage can_MsgTx_EgasCtrl;
CANMessage can_MsgTx_EgasNominalValue;


void CanSendEgas() {
    // idle
    if ( (f_IrDistance > IR_MAX) || (f_IrDistance < IR_MIN) ) {
        u16_CanGasNominal = CAN_GAS_MIN;
    }
    // valid input range
    else {
        u16_CanGasNominal = (unsigned short)( (((CAN_GAS_MAX-CAN_GAS_MIN)*(f_IrDistance-IR_MIN))/(IR_MAX-IR_MIN))+CAN_GAS_MIN );
    }
    // debug
    //printf("u16_CanGasNominal: %d\n\r", u16_CanGasNominal);

    if (can2.write(can_MsgTx_EgasCtrl)) {
#ifdef DEBUG_CAN_TX
        printf("CanTx--> id: 0x%x  dlc: %d  data: ", can_MsgTx_EgasCtrl.id, can_MsgTx_EgasCtrl.len);
        for (char i=0; i<can_MsgTx_EgasCtrl.len; i++) {
            printf("%x ", can_MsgTx_EgasCtrl.data[i]);
        }
        printf("\n\r");
#endif
    }

    can_MsgTx_EgasNominalValue.data[CAN_BYTE_LOW_EGAS_NOMINAL_VALUE] = (unsigned char)(u16_CanGasNominal);
    can_MsgTx_EgasNominalValue.data[CAN_BYTE_HIGH_EGAS_NOMINAL_VALUE] = (unsigned char)(u16_CanGasNominal>>8);
    if (can2.write(can_MsgTx_EgasNominalValue)) {
#ifdef DEBUG_CAN_TX
        printf("CanTx--> id: 0x%x  dlc: %d  data: ", can_MsgTx_EgasNominalValue.id, can_MsgTx_EgasNominalValue.len);
        for (char i=0; i<can_MsgTx_EgasNominalValue.len; i++) {
            printf("%x ", can_MsgTx_EgasNominalValue.data[i]);
        }
        printf("\n\r");
#endif
    }

}

void SerialSendIrDistance() {
    printf("f_IrDistance: %f\r\n", f_IrDistance);
}

void LcdPrintIrDistance() {
    lcd.printf("IrDist: %f", f_IrDistance);
}

int main() {
    // 500kbit/s
    can2.frequency(500000);
    // activate external can transceiver
    can_Pca82c250SlopePin = 0;
    // init everything
    can_MsgTx_EgasCtrl.id = CAN_ID_EGAS_CTRL;
    can_MsgTx_EgasCtrl.len = CAN_DLC_EGAS_CTRL;
    can_MsgTx_EgasCtrl.data[0] = 1;
    can_MsgTx_EgasNominalValue.data[CAN_BYTE_LOW_EGAS_NOMINAL_VALUE] = (unsigned char)(u16_CanGasNominal);
    can_MsgTx_EgasNominalValue.data[CAN_BYTE_HIGH_EGAS_NOMINAL_VALUE] = (unsigned char)(u16_CanGasNominal>>8);
    can_MsgTx_EgasNominalValue.id = CAN_ID_EGAS_NOMINAL_VALUE;
    can_MsgTx_EgasNominalValue.len = CAN_DLC_EGAS_NOMINAL_VALUE;

#ifdef DEBUG_IR_RAW
    TickerSerialTx.attach(&SerialSendIrDistance, 0.35);
#endif
    TickerLcdUpdate.attach(&LcdPrintIrDistance, 0.75);
    TickerCanTx.attach(&CanSendEgas, 0.1);

    while (1) {
        // get new distance from sensor
        f_IrDistance = ain.read();

        // show distance with the 4 leds
        led1 = (f_IrDistance > 0.11) ? 1 : 0;
        led2 = (f_IrDistance > 0.30) ? 1 : 0;
        led3 = (f_IrDistance > 0.47) ? 1 : 0;
        led4 = (f_IrDistance > 0.74) ? 1 : 0;
    }
}

// TODO: no global variables