/*!
  @file main.cpp

  The main vcdMaker demo file.

  @par Full Description
  The vcdMaker demo. See http://vcdmaker.org for details.

  @if REVISION_HISTORY_INCLUDED
  @par Edit History
  @li [0]    wojciech.rynczuk@wp.pl    04-MAR-2016    Initial file revision.
  @li [1]    wojciech.rynczuk@wp.pl    22-APR-2016    Sync line added.
  @endif

  @ingroup vcdMakerDemo

  The MIT License (MIT)
  Copyright (c) 2016 Wojciech Rynczuk

*/

#include "mbed.h"
#include "serialLogger.hpp"
#include "magnetometer.hpp"
#include "accelerometer.hpp"
#include "button.hpp"
#include "sync.hpp"
#include "COMPASS_DISCO_L476VG.h"

#define NO_OF_LINES      ((uint32_t)75)
#define NO_OF_CHARACTERS ((uint32_t)100)

// Periodic timers
Ticker timer1;
Ticker timer2;

// Time reference
uint32_t time_ref = 0;

// Logger
SerialLogger  slogger(NO_OF_LINES,NO_OF_CHARACTERS);
Logger*       uniLogger;

// Compass
COMPASS_DISCO_L476VG compass;

// Signals
Magnetometer  magnetometer("Disco.Sensors.Magnetometer.", compass, slogger);
Accelerometer accelerometer("Disco.Sensors.Accelerometer.", compass, slogger);

// Buttons
Button        button_center("Disco.Buttons.Center", slogger);
Button        button_left("Disco.Buttons.Left", slogger);
Button        button_up("Disco.Buttons.Up", slogger);
Button        button_right("Disco.Buttons.Right", slogger);
Button        button_down("Disco.Buttons.Down", slogger);

//Synchronization
Sync          sync_signal("Disco.Sync", slogger);

// Joystick button
InterruptIn center(JOYSTICK_CENTER);
InterruptIn left(JOYSTICK_LEFT);
InterruptIn right(JOYSTICK_RIGHT);
InterruptIn up(JOYSTICK_UP);
InterruptIn down(JOYSTICK_DOWN);
InterruptIn sync(PE_11);

// Interrupt handlers
void center_released() {
    button_center = 0;
    button_center.PrintState();
    if ((uniLogger->GetTime() - time_ref) > 3000000)
    {
        if (uniLogger->IsRecording())
        {
            uniLogger->StopRecording();
        }
        else
        {
            uniLogger->StartRecording();
        }
    }
}

void center_pressed() {
    button_center = 1;
    button_center.PrintState();
    time_ref = uniLogger->GetTime();
}

void left_released() {
    button_left = 0;
    button_left.PrintState(); 
}

void left_pressed() {
    button_left = 1;
    button_left.PrintState(); 
}

void right_released() {
    button_right = 0;
    button_right.PrintState();  
}

void right_pressed() {
    button_right = 1;
    button_right.PrintState();   
}

void up_released() {
    button_up = 0;
    button_up.PrintState();   
}

void up_pressed() {
    button_up = 1;
    button_up.PrintState();  
}

void down_released() {
    button_down = 0;
    button_down.PrintState(); 
}

void down_pressed() {
    button_down = 1;
    button_down.PrintState();  
}

void sync_released() {
    sync_signal = 0;
    sync_signal.PrintState(); 
}

void sync_pressed() {
    sync_signal = 1;
    sync_signal.PrintState();  
}

void accelerometer_tick()
{
    accelerometer.PrintState();
}

void magnetometer_tick()
{
    magnetometer.PrintState();
}

int main()
{    
    uniLogger = &slogger;
    
    // Both rise and fall edges generate an interrupt
    center.fall(&center_released);
    center.rise(&center_pressed);
    left.fall(&left_released);
    left.rise(&left_pressed);
    right.fall(&right_released);
    right.rise(&right_pressed);
    up.fall(&up_released);
    up.rise(&up_pressed);
    down.fall(&down_released);
    down.rise(&down_pressed);
    sync.fall(&sync_pressed);
    sync.rise(&sync_released);

    // Add pull-down on these pins
    // Warning: must be done AFTER edges setting
    left.mode(PullDown);
    right.mode(PullDown);
    up.mode(PullDown);
    down.mode(PullDown);
    sync.mode(PullUp);
        
    timer1.attach_us(&magnetometer_tick, 50000);
    timer2.attach_us(&accelerometer_tick, 40000);
        
    printf("Press joystick pressed for more than 3 seconds to start/stop logging.\n\r");
  
    while(true) {
        uniLogger->Print();
    };
}
