/*!
  @file main.cpp

  The main vcdMaker demo file.

  @par Full Description
  The vcdMaker demo. See http://vcdmaker.org for details.

  @if REVISION_HISTORY_INCLUDED
  @par Edit History
  @li [0]    wojciech.rynczuk@wp.pl    20-JAN-2015    Initial file revision.
  @li [1]    wojciech.rynczuk@wp.pl    22-APR-2016    Sync line added.
  @endif

  @ingroup vcdMakerDemo

  The MIT License (MIT)
  Copyright (c) 2016 Wojciech Rynczuk

*/

#include "mbed.h"
#include "serialLogger.hpp"
#include "button.hpp"
#include "light.hpp"
#include "slider.hpp"
#include "magnetometer.hpp"
#include "accelerometer.hpp"
#include "sync.hpp"

#define NO_OF_LINES      ((uint32_t)75)
#define NO_OF_CHARACTERS ((uint32_t)100)

// Periodic timers
Ticker timer1;
Ticker timer2;

// Button interrupts
InterruptIn sw1_event(PTC3);
InterruptIn sw3_event(PTC12);
InterruptIn sync_event(PTA13);

// Logger
SerialLogger  slogger(NO_OF_LINES,NO_OF_CHARACTERS);
Logger*       uniLogger;

// Signals
Sync          sync("FRDM.Sync",PTA13,slogger);
Button        sw1("FRDM.Buttons.SW1",PTC3,slogger);
Button        sw3("FRDM.Buttons.SW3",PTC12,slogger);
Light         light("FRDM.Sensors.Light",PTE22,slogger);
Slider        slider("FRDM.Sensors.Slider", slogger);
Magnetometer  magnetometer("FRDM.Sensors.Magnetometer.", slogger);
Accelerometer accelerometer("FRDM.Sensors.Accelerometer.", slogger);

// Time reference
uint32_t time_ref = 0;

void log_slider_accelerometer()
{
    slider.PrintState();
    accelerometer.PrintState();
}

void log_light_magnetometer()
{
    light.PrintState();
    magnetometer.PrintState();
}

void sw1_irq()
{
    sw1.PrintState();
    if (0 == sw1)
    {
        time_ref = uniLogger->GetTime();
    }
    else
    {
        if ((uniLogger->GetTime() - time_ref) > 3000000)
        {
            if (uniLogger->IsRecording())
            {
                uniLogger->StopRecording();
            }
            else
            {
                uniLogger->StartRecording();
            }
        }
    }
}

void sw3_irq()
{
    sw3.PrintState();
}

void sync_irq()
{
    sync.PrintState();
}


int main()
{
    uniLogger = &slogger;
    
    sw1_event.rise(&sw1_irq);
    sw3_event.rise(&sw3_irq);
    sw1_event.fall(&sw1_irq);
    sw3_event.fall(&sw3_irq);
    sync_event.fall(&sync_irq);
    sync_event.rise(&sync_irq);
    timer1.attach_us(&log_light_magnetometer, 50000);
    timer2.attach_us(&log_slider_accelerometer, 40000);
    
    while (true)
    {        
        uniLogger->Print();
    };
}