/*!
  @file logger.cpp

  The implementation of the logger base class.

  @par Full Description
  The implementation of the logger base class.

  @if REVISION_HISTORY_INCLUDED
  @par Edit History
  @li [0]    wojciech.rynczuk@wp.pl    20-JAN-2015    Initial file revision.
  @endif

  @ingroup Logger

  The MIT License (MIT)
  Copyright (c) 2016 Wojciech Rynczuk

*/

#include "logger.hpp"

#include <new>
#include <cstring>

Logger::Logger(uint32_t nLines, uint32_t nCharacters) : m_Lines(nLines), m_Characters(nCharacters)
{
    m_ReadMarker = 0;
    m_WriteMarker = 0;
    m_isRecording = false;
    m_pLogData = new(std::nothrow) char [sizeof(char) * m_Lines * m_Characters];
    m_Timer.start();
}

Logger::~Logger()
{
    delete m_pLogData;
}

uint32_t Logger::Log(const char* line)
{
    int32_t was_masked;
    
    if (m_pLogData != NULL)
    {
        if (true == m_isRecording)
        {
            //Disable interrupts
            was_masked = __disable_irq();
    
            if ((m_ReadMarker == (m_WriteMarker + 1)) || \
               ((m_WriteMarker == (m_Lines - 1)) && (m_ReadMarker == 0)))
            {
                AlarmFull();
            } 
            else
            {
                (void)std::strncpy(m_pLogData + (m_WriteMarker * m_Characters), line, m_Characters);
                m_WriteMarker++;
                if (m_WriteMarker == m_Lines)
                {
                    m_WriteMarker = 0;
                }
            }
    
            //Enable interrupts (or not)
            if (!was_masked)
            {
                __enable_irq();
            }
        } 
    }
    else
    {
        Printf("Not enough memory for logging.\n");
    }
    return 0;
}

uint32_t Logger::Print()
{
    uint32_t counter = 0;
    if (m_pLogData != NULL)
    {
        while(m_ReadMarker != m_WriteMarker)
        {
            Printf(m_pLogData + (m_ReadMarker++ * m_Characters));
            if (m_ReadMarker == m_Lines)
            {
                m_ReadMarker = 0;
            }
            counter++;
        }
    }
    return counter;
}

int32_t Logger::GetTime()
{
    return m_Timer.read_us();
}

void Logger::StartRecording()
{
    if (0 == StartAction())
    {
        m_isRecording = true;
    }
}

void Logger::StopRecording()
{
    m_isRecording = false;
    StopAction();
}

bool Logger::IsRecording()
{
    return m_isRecording;
}
