/*
 * mbed library program
 *  Wake-Up function only at Sleep mode
 *      Limitation:
 *       1) only for nRF51822
 *       2) RTC1 is used in mbed
 *
 * Copyright (c) 2016 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created:    March     24th, 2016
 *      Revised:    June      11th, 2016
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef NRF51_WAKEUP
#define NRF51_WAKEUP

#include "mbed.h"
#include "nrf_soc.h"
#include "nrf_delay.h"

#ifndef TARGET_NRF51822
#error "This function is only for nRF51822."
#endif

/** Set deepsleep duration time and wakeup as same as RESET state
 *
 * @code
 * #include "mbed.h"
 * #include "nRF51_WakeUp.h"
 *
 * DigitalOut   myled(LED1);
 * DigitalIn    sw(P0_1);
 * // RTC1 is 24bit counter, you cannot wait over 582 hours (125ms * 2^24 = 0.125 * 2097152 sec)
 * nRF51_WakeUp wakeup(LED2, P0_0); // Trigger & Interrupt (MUST tied up together!!)
 *
 * int main() {
 *   while(true){
 *      myled = 1;
 *      wait(1.0);
 *      myled = 0;
 *      wait(1.0);
 *      if (sw == 0){
 *          wakeup.set_and_wait(120);   // 2 minutes
 *          deepsleep();                // just in case
 *      }
 *   }
 * }
 * @endcode
 */

class nRF51_WakeUp
{
public:
    /** Configure control pin
      * @param output(trigger) pin & Interrupt input pin
      * @Hardware: Trigger pin and Interrupt pin are must connected!!
      */
    nRF51_WakeUp(PinName p_trigger, PinName p_interrupt);

    /** Set wake-up time then goto Deep Sleep mode
      * @param wake-up time (in minutes)
      * @none
      */
    void set_and_wait(uint32_t t_sec);

private:
    void        action4restart(void);

    DigitalOut  _pin0;
    InterruptIn _pin1;
    DigitalIn   _pin2;
    PinName     p_name_trgr;
};

#endif      // NRF51_WAKEUP
