/*
 * mbed library program
 *  Read nRF51822 VDD volatage and return 100 to 0% charging level
 *      https://www.nordicsemi.com/eng/Products/Bluetooth-Smart-Bluetooth-low-energy/nRF51822
 *
 * Copyright (c) 2016,'18 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  https://os.mbed.com/users/kenjiArai/
 *      Created:    January   23rd, 2016
 *      Revised:    April     14th, 2018
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef NRF51_VDD
#define NRF51_VDD

#include "mbed.h"

#define  ONLY4VDD   1
#define  USE_ADC    0

/** Read nRF51822 VDD volatage and return 0 to 100% charging level
 *
 * @code
 * #include "mbed.h"
 * #include "nRF51_Vdd.h"
 *
 * nRF51_Vdd vdd(3.6f, 1.8f);
 * // or follows if you don't use ADC function (more low current)
 * // nRF51_Vdd vdd(3.6f, 1.8f, ONLY4VDD);
 *
 * int main() {
 * uint8_t batteryLevel;
 * float batt;
 *
 *   while(true){
 *      wait(1.0);
 *      batteryLevel = vdd.read();
 *      batt = vdd.read_real_value();
 *   }
 * }
 * @endcode
 */

class nRF51_Vdd
{
public:
    /** Configure data pin
      * @param 100% voltage reference value
      * @param 0% voltage reference value
      * @param LOW POWER MODE (none zero) Use PWR_DWN or USE_ADC
      */
    nRF51_Vdd(float vdd100percent, float vddzeropercent, uint8_t adc_use);

    /** Configure data pin
      * @param 100% voltage reference value
      * @param 0% voltage reference value
      */
    nRF51_Vdd(float vdd100percent, float vddzeropercent);

    /** Configure data pin (with other devices on I2C line)
      * @param none
      */
    nRF51_Vdd(void);

    /** Read Voltage value (percentage)
      * @param none
      * @return 0 to 100%
      */
    uint8_t read(void);

    /** Read Voltage value (real voltage)
      * @param none
      * @return real voltage (example 3.30V)
      */
    float read_real_value(void);

private:
    float       wrk_vdd, v100p, v0p;    // working buffer
    uint32_t    reg0,reg1,reg2;         // Save current Configuration data
    uint32_t    wrk;                    // save ADC(Vdd) data 
    uint8_t     use_adc_for_others;     // save PWR_DWN or USE_ADC  
};

#endif      // NRF51_VDD
