/*
 * mbed library program
 *  Read nRF51822 VDD volatage and return 100 to 0% charging level
 *      https://www.nordicsemi.com/eng/Products/Bluetooth-Smart-Bluetooth-low-energy/nRF51822
 *
 * Copyright (c) 2016,'18 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  https://os.mbed.com/users/kenjiArai/
 *      Created:    January   23rd, 2016
 *      Revised:    April     14th, 2018
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "nRF51_Vdd.h"
#include "nrf_delay.h"

nRF51_Vdd::nRF51_Vdd(float vdd100percent, float vddzeropercent, uint8_t adc_use){
    v100p = vdd100percent;
    v0p = vddzeropercent;
    use_adc_for_others = adc_use;
}

nRF51_Vdd::nRF51_Vdd(float vdd100percent, float vddzeropercent){
    v100p = vdd100percent;
    v0p = vddzeropercent;
    use_adc_for_others = USE_ADC;
}

nRF51_Vdd::nRF51_Vdd(void){
    v100p = 3.3f;
    v0p = 1.8f;
    use_adc_for_others = USE_ADC;
}

float nRF51_Vdd::read_real_value(void){
    NRF_ADC->POWER       = 1;
    nrf_delay_us(5);
    NRF_ADC->EVENTS_END  = 0;
    NRF_ADC->TASKS_STOP  = 1;
    reg0 = NRF_ADC->ENABLE;     // save register value
    reg1 = NRF_ADC->CONFIG;     // save register value
    reg2 = NRF_ADC->RESULT;     // save register value
    NRF_ADC->ENABLE = ADC_ENABLE_ENABLE_Enabled;
    NRF_ADC->CONFIG = (ADC_CONFIG_RES_10bit << ADC_CONFIG_RES_Pos) |
                      (ADC_CONFIG_INPSEL_SupplyOneThirdPrescaling << ADC_CONFIG_INPSEL_Pos) |
                      (ADC_CONFIG_REFSEL_VBG << ADC_CONFIG_REFSEL_Pos) |
                      (ADC_CONFIG_PSEL_Disabled << ADC_CONFIG_PSEL_Pos) |
                      (ADC_CONFIG_EXTREFSEL_None << ADC_CONFIG_EXTREFSEL_Pos);
    NRF_ADC->EVENTS_END  = 0;
    NRF_ADC->TASKS_START = 1;
    while (!NRF_ADC->EVENTS_END) {;}
    wrk = NRF_ADC->RESULT;          // 10 bit result
    NRF_ADC->EVENTS_END  = 0;
    NRF_ADC->TASKS_STOP = 1;
    if (use_adc_for_others == USE_ADC){
        NRF_ADC->ENABLE = reg0;     // recover register value
        NRF_ADC->CONFIG = reg1;     // recover register value
        NRF_ADC->RESULT = reg2;     // recover register value
        NRF_ADC->EVENTS_END  = 0;
    } else {
        NRF_ADC->CONFIG      = 0x18;
        NRF_ADC->EVENTS_END  = 1;
        NRF_ADC->TASKS_STOP  = 1;
        NRF_ADC->ENABLE      = 0;
        nrf_delay_us(5);
        NRF_ADC->POWER       = 0;
        nrf_delay_us(5);
    }
    return ((float)wrk / 1024.0f * 1.2f * 3.0f);            
}

uint8_t nRF51_Vdd::read(void){
    wrk_vdd = read_real_value();
    if (wrk_vdd <= v0p){
        return 0;
    } else if (wrk_vdd >= v100p){
        return 100;
    }
    wrk_vdd = (wrk_vdd - v0p) / (v100p - v0p);
    return (uint8_t)(wrk_vdd * 100); 
}
