//-------------------------------------------------------------------------------------------------------------------
// Monitor program Ver.1
//      (c)2010  Kenji Arai / JH1PJL 
//      http://www.page.sannet.ne.jp/kenjia/index.html
//      http://mbed.org/users/kenjiArai/   
//          May 9th,2010  Started
//          May 15th,2010
//--------------------------------------------------------------------------------------------------------------------
// Reference 
//  http://www.nxp.com/#/pip/pip=[pip=LPC1769_68_67_66_65_64_4]|pp=[t=pip,i=LPC1769_68_67_66_65_64_4]
//--------------------------------------------------------------------------------------------------------------------
// Function
//      Show Memory contents, Digital port, Analog input port, CPU clock, RTC/Time and others in the mbed board
// Connection
//      Analog input    PIN 15,16,17
//      Digital input   PIN 24,25,26
//      LCD             PIN 5,6,7,21,22,23
//      RTC             PIN 3 needs to connect 3V Battery
//       -> Please refer my program " RTC_w_COM" for time adjustment
//          at " http://mbed.org/users/kenjiArai/programs/RTC_w_COM/5yi9a/ "
//---------------------------------------------------------------------------------------------------------------------

#include "mbed.h"

// Define clocks
#define XTAL        (12000000UL)        /* Oscillator frequency               */
#define OSC_CLK     (      XTAL)        /* Main oscillator frequency          */
#define RTC_CLK     (   32000UL)        /* RTC oscillator frequency           */
#define IRC_OSC     ( 4000000UL)        /* Internal RC oscillator frequency   */
// Range check status
#define ERR_NOTHING         0
#define ERR_MODIFY_SIZ      1
#define ERR_OUT_OF_RANGE    2

typedef struct{
  unsigned long mstr;
  unsigned long msiz;
  unsigned long mtmp;
  unsigned long mold;
  unsigned char mflg;
} MEMO;

Serial pc(USBTX, USBRX);
AnalogIn ana0(p15);
AnalogIn ana1(p16);
AnalogIn ana2(p17);
DigitalIn sw0(p24);
DigitalIn sw1(p25);
DigitalIn sw2(p26);

char linebuf[256];

//  Memory management
static MEMO mem;
unsigned long SystemFrequency;

//  Memory range data
const uint32_t mem_range[][2] = {         // Memory access range
        { 0x00000000, 0x0007ffff },            // On-chip non-volatile memory    //512KB Flash memory
        { 0x10000000, 0x10007fff },            // On-chip SRAM                    //32KB local RAM
        { 0x1fff0000, 0x1fff1fff },            // Boot ROM                        //8KB Boot ROM
        { 0x2007c000, 0x2007ffff },            // On-chip SRAM                    //16KB AHB SRAM
        { 0x20080000, 0x20083fff }             // On-chip SRAM                    //16KB AHB SRAM
};

//  Put \r\n
void put_cr ( void ){
    pc.putc('\r');
    pc.putc('\n');
}

//  Put \r
void put_r ( void ){
    pc.putc('\r');
}

//  Help Massage
void msg_hlp ( void ){
    pc.printf("Monitor for mbed LPC1768 Ver. on "__DATE__" (" __TIME__ ")");
    put_cr();
    pc.printf("a  - Show ADC data ");
    put_cr();
    pc.printf("m  - Entry Memory Mode");
    put_cr();
    pc.printf("m>? -> After entry the Memory Mode, adiitinal functions can see by ?");  
    put_cr();      
    pc.printf("p  - Show Port status ");
    put_cr();
    pc.printf("sf - System Clock");
    put_cr();
    pc.printf("sc - System / CPU information");
    put_cr();
    pc.printf("t    [<year> <mon> <mday> <hour> <min> <sec>] ");
    put_cr();
    pc.printf("/    [a],[p],[t] commands every 1 sec  Exit =hit any key (not ENTER key) ");
    put_cr();
    pc.printf("q  - Quit (back to called routine)");
    put_cr();
}

//  Check Switch port status
void check_port ( void ){
    put_r();
    pc.printf("Input Port Status");
    put_cr();
    pc.printf("mbed P26 (LPC1768 p2.0) = ");
    if ( sw2.read() ){
        pc.printf("ON  ");
    } else {
        pc.printf("OFF ");
    }
    put_cr();
    pc.printf("mbed P25 (LPC1768 p2.1) = ");
    if ( sw1.read() ){
        pc.printf("ON  ");
    } else {
        pc.printf("OFF ");
    }
    put_cr();
    pc.printf("mbed P24 (LPC1768 p2.2) = ");
    if ( sw0.read() ){
        pc.printf("ON  ");
    } else {
        pc.printf("OFF ");
    }
    put_cr();
}

void cpu_inf ( void ){
    unsigned long m1, m2;
    
    m1 = SCB->CPUID;
    m2 = ( m1 >> 24 );
    if ( m2 == 0x41 ){
        put_r();    
        pc.printf("CPU = ARM ");
    } else {
        put_r();
        pc.printf("CPU = NOT ARM ");
    }
    m2 = ( m1 >> 4 ) & 0xfff;
    if ( m2 == 0xc23 ){
        pc.printf("Cortex-M3");
        put_cr();
    } else {
        pc.printf("NOT Cortex-M3");
        put_cr();
    }
    m2 = ( m1 >> 20 ) & 0x0f;
    pc.printf("Variant:%x", m2);
    put_cr();
    m2 = m1 & 0x7;
    pc.printf("Revision:%x", m2);
    put_cr();
}

//  Calculate CPU System Clock Frequency
void get_freq ( int pr ){
    if(pr){
        put_r();
        pc.printf("System Clock = %dHz",SystemFrequency );
        put_cr();
    }
    if (((LPC_SC->PLL0STAT >> 24) & 3) == 3) {/* If PLL0 enabled and connected      */
        if(pr){
            pc.printf("PLL0 enabled");
            put_cr();
        }
        switch (LPC_SC->CLKSRCSEL & 0x03) {
            case 0:                /* Internal RC oscillator => PLL0     */
            case 3:                /* Reserved, default to Internal RC   */
                if(pr){
                    pc.printf("Internal RC Oscillator");
                    put_cr();
                }
                SystemFrequency = (IRC_OSC *
                                    (((2 * ((LPC_SC->PLL0STAT & 0x7FFF) + 1))) /
                                    (((LPC_SC->PLL0STAT >> 16) & 0xFF) + 1))   /
                                    ((LPC_SC->CCLKCFG & 0xFF)+ 1));
                break;
            case 1:                /* Main oscillator => PLL0            */
                if(pr){
                    pc.printf("Xtal Osc Clock = %dHz",XTAL );
                    put_cr();
                }
                SystemFrequency = (OSC_CLK *
                                    (((2 * ((LPC_SC->PLL0STAT & 0x7FFF) + 1))) /
                                    (((LPC_SC->PLL0STAT >> 16) & 0xFF) + 1))   /
                                    ((LPC_SC->CCLKCFG & 0xFF)+ 1));
                break;
            case 2:                /* RTC oscillator => PLL0             */
                if(pr){
                    pc.printf("RTC Xtal Oscillator f = %dHz", RTC_CLK );
                    put_cr();
                }
                SystemFrequency = (RTC_CLK *
                                    (((2 * ((LPC_SC->PLL0STAT & 0x7FFF) + 1))) /
                                    (((LPC_SC->PLL0STAT >> 16) & 0xFF) + 1))   /
                                    ((LPC_SC->CCLKCFG & 0xFF)+ 1));
                break;
        }
    } else {
        if(pr){
            pc.printf("PLL0 disabled");
        }
        switch (LPC_SC->CLKSRCSEL & 0x03) {
            case 0:                /* Internal RC oscillator => PLL0     */
            case 3:                /* Reserved, default to Internal RC   */
                if(pr){
                    pc.printf("Internal RC Oscillator");
                    put_cr();
                }
                SystemFrequency = IRC_OSC / ((LPC_SC->CCLKCFG & 0xFF)+ 1);
                break;
            case 1:                /* Main oscillator => PLL0            */
                if(pr){
                    pc.printf("Xtal Osc Clock = %dHz",XTAL );
                    put_cr();
                }
                SystemFrequency = OSC_CLK / ((LPC_SC->CCLKCFG & 0xFF)+ 1);
                break;
            case 2:                /* RTC oscillator => PLL0             */
                if(pr){
                    pc.printf("RTC Xtal Oscillator f = %dHz", RTC_CLK );
                    put_cr();
                }
                SystemFrequency = RTC_CLK / ((LPC_SC->CCLKCFG & 0xFF)+ 1);
                break;
        }
    }
}

//  Range check for Memory dump 
static void check_range( MEMO * mem ){
    uint8_t    i;
    uint32_t m;

    mem->mflg = ERR_NOTHING;
    for ( i = 0 ; i < 5 ; i++ ){
        if ( mem->mstr >= mem_range[i][0]){
            if ( mem->mstr < mem_range[i][1] ){
                m = mem->mstr + mem->msiz;
                if ( m < mem_range[i][1]){
                    return;            // no modification
                } else {
                    m = mem_range[i][1];
                    mem->msiz = m - mem->mstr + 1;
                    mem->mflg = ERR_MODIFY_SIZ;
                    return;            // modified size
                }
            }
        }
    }
    mem->mflg = ERR_OUT_OF_RANGE;
    mem->mstr = 0;
    mem->msiz = 0;
    return ;
}

//  Memory dump error massage
void error_print ( unsigned char flg ){
    switch (flg) {
        case ERR_MODIFY_SIZ :
            put_r();
            pc.printf("Reach to out of range ");
            put_cr();
            break;
        case ERR_OUT_OF_RANGE :
            put_r();
            pc.printf("Not in a memory area ");
            put_cr();
            break;
        case ERR_NOTHING :
        default :
            ;
    }
}

//  Change string -> integer
int xatoi (char **str, unsigned long *res){
    unsigned long val;
    unsigned char c, radix, s = 0;

    while ((c = **str) == ' ') (*str)++;
    if (c == '-') {
        s = 1;
        c = *(++(*str));
    }
    if (c == '0') {
        c = *(++(*str));
        if (c <= ' ') {
            *res = 0; return 1;
        }
        if (c == 'x') {
            radix = 16;
            c = *(++(*str));
        } else {
            if (c == 'b') {
                radix = 2;
                c = *(++(*str));
            } else {
                if ((c >= '0')&&(c <= '9'))
                    radix = 8;
                else
                    return 0;
            }
        }
    } else {
        if ((c < '1')||(c > '9'))
            return 0;
        radix = 10;
    }
    val = 0;
    while (c > ' ') {
        if (c >= 'a') c -= 0x20;
        c -= '0';
        if (c >= 17) {
            c -= 7;
            if (c <= 9) return 0;
        }
        if (c >= radix) return 0;
        val = val * radix + c;
        c = *(++(*str));
    }
    if (s) val = -val;
    *res = val;
    return 1;
}

//  Print memory contents
void put_dump (const unsigned char *buff, unsigned long ofs, int cnt){
    int n;

#if 1
    pc.printf("%08lX ", ofs);
    for(n = 0; n < cnt; n++){
        pc.printf(" %02X", buff[n]);
    }
    pc.putc(' ');
    for(n = 0; n < cnt; n++) {
        if ((buff[n] < 0x20)||(buff[n] >= 0x7F)){
            pc.putc('.');
        } else {
            pc.putc(buff[n]);
        }
    }
    put_cr();
#endif
    
#if 0
    printf("%08lX ", ofs);
    for(n = 0; n < cnt; n++){
        printf(" %02X", buff[n]);
    }
    pc.putc(' ');
    for(n = 0; n < cnt; n++) {
        if ((buff[n] < 0x20)||(buff[n] >= 0x7F)){
            pc.putc('.');
        } else {
            pc.putc(buff[n]);
        }
    }
    pc.puts("\r\n");    
#endif
}

//  Get key input data
void get_line (char *buff, int len){
    char c;
    int idx = 0;

    for (;;) {
        c = pc.getc();
        //    Added by Kenji Arai / JH1PJL   May 9th, 2010
        if (c == '\r'){
            buff[idx++] = c;
            break;
        }
        if ((c == '\b') && idx) {
            idx--; pc.putc(c); pc.putc(' '); pc.putc(c);
        }
        if (((uint8_t)c >= ' ') && (idx < len - 1)) {
            buff[idx++] = c; pc.putc(c);
        }
    }
    buff[idx] = 0;
    pc.putc('\n');
}

//  Monitor Program
int mon() {
    char *ptr;
    unsigned long p1;
    char c;
    time_t seconds;
    struct tm rtc;

    pc.baud(115200);
    put_cr();
    put_cr();
    pc.printf("Monitor for mbed LPC1768 Ver. on "__DATE__" (" __TIME__ ") [Help:'?' key]");
    put_cr();
    get_freq(0);
    for (;;) {
        put_r();
        pc.putc('>');
        ptr = linebuf;
        get_line(ptr, sizeof(linebuf));
        switch (*ptr++) {
        //--------------------------------------------------------------------------------------
        //    Memory
        //--------------------------------------------------------------------------------------
        case 'm' :
            put_r();
            pc.printf("Entry Memory Mode  1) d <address> [<count>], 2) s, 3) <ret>, 4) <esc>");
            put_cr();
            mem.mstr = 0;
            mem.msiz =256;
            mem.mold = 0;
            mem.mtmp = 0;
            mem.mflg = 0;
            for (;mem.mflg != 0xff;){
                pc.printf("m>");
                c = pc.getc();
                pc.putc(c);
                switch (c) {
                    case 'd' :    /* d <address> [<count>] - Dump memory */
                        ptr = linebuf;
                        get_line(ptr, sizeof(linebuf));
                        mem.mtmp = mem.mstr;
                        if (!xatoi(&ptr, &mem.mstr)){
                            mem.mstr = mem.mtmp;
                        }
                        if (!xatoi(&ptr, &mem.msiz)){
                            mem.msiz = 256;
                        }
                        mem.mtmp = mem.msiz;
                        check_range (&mem);
                        for (ptr=(char*)mem.mstr; mem.msiz >= 16; ptr += 16, mem.msiz -= 16){
                            put_r();
                            put_dump((unsigned char*)ptr, (unsigned int)ptr, 16);
                        }
                        if (mem.msiz){
                            put_dump((unsigned char*)ptr, (unsigned int)ptr, mem.msiz);
                        }
                        error_print(mem.mflg);
                        mem.mold = mem.mstr;
                        mem.mstr += mem.mtmp;
                        break;
                    case 0x0d :        // CR
                        put_cr();
                        mem.msiz = 256;
                        mem.mtmp = mem.msiz;
                        check_range (&mem);
                        for (ptr=(char*)mem.mstr; mem.msiz >= 16; ptr += 16, mem.msiz -= 16){
                            put_r();
                            put_dump((unsigned char*)ptr, (unsigned int)ptr, 16);
                        }
                        error_print(mem.mflg);
                        mem.mold = mem.mstr;
                        mem.mstr += 256;
                        break;
                    case 0x1b :        // ESC
                        mem.mflg = 0xff;
                        break;
                    case 0x08 :        // Back Space
                        put_cr();
                        mem.mstr = mem.mold;
                        mem.msiz = 256;
                        mem.mtmp = mem.msiz;
                        check_range (&mem);
                        for (ptr=(char*)mem.mstr; mem.msiz >= 16; ptr += 16, mem.msiz -= 16){
                            put_r();
                            put_dump((unsigned char*)ptr, (unsigned int)ptr, 16);
                        }
                        error_print(mem.mflg);
                        mem.mstr += 256;
                        break;
                    case 's' :
                        put_cr();
                        pc.printf("Memory Configuration");
                        put_cr();
                        pc.printf("FLASH    0x%08lx to 0x%08lx ", mem_range[0][0], mem_range[0][1]);
                        put_cr();
                        pc.printf("S-RAM    0x%08lx to 0x%08lx ", mem_range[1][0], mem_range[1][1]);
                        put_cr();
                        pc.printf("Boot-ROM 0x%08lx to 0x%08lx ", mem_range[2][0], mem_range[2][1]);
                        put_cr();
                        pc.printf("AHB-RAM0 0x%08lx to 0x%08lx ", mem_range[3][0], mem_range[3][1]);
                        put_cr();
                        pc.printf("AHB-RAM1 0x%08lx to 0x%08lx \r\n", mem_range[4][0], mem_range[4][1]);
                        break;
                    case '?' :
                        put_cr();
                        pc.printf("d <address> [<count>] - Dump memory");
                        put_cr();
                        pc.printf("s     - Show memory structure ");
                        put_cr();
                        pc.printf("<RET> - Dump memory / next 128 bytes");
                        put_cr();
                        pc.printf("<BS>  - Dump memory / same as before 128 bytes");
                        put_cr();
                        pc.printf("<ESC> - Exit memory mode");
                        put_cr();
                        break;
                    default:
                        put_cr();
                }
            }
            pc.putc('>');
            put_cr();
            break;
        //--------------------------------------------------------------------------------------
        //    Port
        //--------------------------------------------------------------------------------------
        case 'p' :    /* p  - Show Port status */
                check_port();
                break;
        //--------------------------------------------------------------------------------------
        //    ADC
        //--------------------------------------------------------------------------------------
        case 'a' :    /* a  - Show ADC data */
                put_r();
                pc.printf("anao = %f, ana1 = %f, ana2 = %f", ana0.read(), ana1.read(), ana2.read());
                put_cr();
                break;
        //--------------------------------------------------------------------------------------
        //    Time
        //--------------------------------------------------------------------------------------
        case 't' :    /* t [<year> <mon> <mday> <hour> <min> <sec>] */
                if (xatoi(&ptr, &p1)) {
                    rtc.tm_year = (unsigned short)p1;
                    xatoi(&ptr, &p1); rtc.tm_mon = (unsigned char)p1;
                    xatoi(&ptr, &p1); rtc.tm_mday = (unsigned char)p1;
                    xatoi(&ptr, &p1); rtc.tm_hour = (unsigned char)p1;
                    xatoi(&ptr, &p1); rtc.tm_min = (unsigned char)p1;
                    if (!xatoi(&ptr, &p1)) break;
                    rtc.tm_sec = (unsigned char)p1;
                    seconds = mktime(&rtc);
                    set_time(seconds);
                }
                while ( seconds == time(NULL)) ;
                seconds = time(NULL);
                //                 13:24:00 PM (2010/03/27)
                strftime(linebuf,40, "%I:%M:%S %p (%Y/%m/%d)", localtime(&seconds));
                put_r();
                pc.printf("Time = %s", linebuf);
                put_cr();
                break;
        //--------------------------------------------------------------------------------------
        //    System
        //--------------------------------------------------------------------------------------
        case 's' :    /* System related information */
                switch (*ptr++) {
                    case 'f' :    /* sc - show system clock frequency */
                        get_freq(1);
                        break;
                    case 'c' :    /* sc - show system CPU information*/
                        cpu_inf();
                        break;
                    case '?' :
                        pc.printf("sc - System CPU information");
                        put_cr();
                        pc.printf("sf - System Clock");
                        put_cr();
                        break;
                }
                break;
        //--------------------------------------------------------------------------------------
        //    Repeat Command every second
        //--------------------------------------------------------------------------------------
        case '/' :
                switch (*ptr++) {
                    case 'a' :
                        while(1){
                            put_r();
                            pc.printf("anao = %f, ana1 = %f, ana2 = %f", ana0.read(), ana1.read(), ana2.read());
                            put_cr();
                            if ( pc.readable() ){
                                break;
                            }
                            wait(1.0);    // Wait 1sec
                        }
                        break;
                    case 'p' :
                        while(1){
                            check_port();
                            if ( pc.readable() ){
                                break;
                            }
                            wait(1.0);    // Wait 1sec
                        }
                        break;
                    case 't' :
                        while(1){
                            while ( seconds == time(NULL)) ;
                            seconds = time(NULL);
                            //                 13:24:00 PM (2010/03/27)
                            strftime(linebuf,40, "%I:%M:%S %p (%Y/%m/%d)", localtime(&seconds));
                            put_r();
                            pc.printf("Time = %s", linebuf);
                            put_cr();
                            if ( pc.readable() ){
                                break;
                            }
                            wait(1.0);    // Wait 1sec
                        }
                        break;
                }
                break;
        //--------------------------------------------------------------------------------------
        //    Help
        //--------------------------------------------------------------------------------------
        case '?' :
                msg_hlp();
                break;
        //--------------------------------------------------------------------------------------
        //    Return to main routine
        //--------------------------------------------------------------------------------------
        case 'q' :        // Quit
                pc.printf("\rReturn to main but back to mon()\r\n");
                return 0;
        }
    }
}
