/*
 * mbed Application program
 *      Check "System clock" frequency
 *
 * Copyright (c) 2015 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: may       18th, 2015
 *      Revised: may       19th, 2015
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "mbed.h"

DigitalOut myled(LED1);

void put_rn(void);
void detail_info_if_available(void);

int main()
{
    put_rn();
    printf("Your mbed SysClock is %u Hz.", SystemCoreClock);
    put_rn();
    put_rn();
    detail_info_if_available();
    while(true) {
        myled = !myled;
        wait(1.0);
    }
}

/* measured by JH1PJL on May 18th & 19th, 2015

 Check old revision
 Case L152 (C-01 old ver with internal RC clock)
    before and rev.81 20140319  -> 16000000Hz
    rev.82 and after  20140407  -> 32000000Hz
 Case L152 (C-02 new ver)
    before and rev.81 20140319  -> 16000000Hz
    rev.82 and after  20140407  -> 24000000Hz
 Case F401
    all rev.(?)                 -> 84000000Hz
 Case F411
    befoe and rev.97  20150414  -> 100000000Hz
    rev.98& 99 (as of 20150518) -> 96000000Hz
 Case LPC1768
    rev.99 -> 96000000Hz
 Case GR-PEACH
    error "SystemCoreClock" is undefined
 Case LPC1114FN28
    rev.99 -> 48000000Hz
 Case FRDM-K64F
    rev.99 -> 120000000Hz
 */

/* ------ History L152RE Clock value ----------------
    -------------------------------------------------
    Ver C-02 (NEW) mbed (Rev.99)
    -------------------------------------------------
    Your mbed SysClock is 24000000 Hz.
    Use PLL with Mul=6, Div=2
    Use HSE(not Xtal but External Clock)=8000000Hz
    PLL freq=24000000Hz
    SYSCLK clock freq. =24000000Hz
    HCLK   clock freq. =24000000Hz
    PCLK1  clock freq. =24000000Hz
    PCLK2  clock freq. =24000000Hz
    RTC/LCD Clock
    No clock
    -------------------------------------------------
    Ver C-01 (OLD) mbed (Rev.99)
    -------------------------------------------------
    Your mbed SysClock is 32000000 Hz.
    Use PLL with Mul=4, Div=2
    Use HSI(internal RC/High speed), RC=16000000Hz
    PLL freq=32000000Hz
    SYSCLK clock freq. =32000000Hz
    HCLK   clock freq. =32000000Hz
    PCLK1  clock freq. =32000000Hz
    PCLK2  clock freq. =32000000Hz
    RTC/LCD Clock
    No clock
    -------------------------------------------------
    Ver C-02 (NEW) mbed (Rev.81)
    -------------------------------------------------
    Your mbed SysClock is 16000000 Hz.
    Use HSI(internal RC/High speed), freq=16000000Hz
    -------------------------------------------------
    Ver C-01 (OLD) mbed (Rev.81)
    -------------------------------------------------
    Your mbed SysClock is 16000000 Hz.
    Use HSI(internal RC/High speed), freq=16000000Hz
 */

///////////////////////////////////////////////////////////////////////////////////////////////////

#if \
 defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F411RE) || defined(TARGET_NUCLEO_L152RE)

// USB Frequency
#define USB_FREQ_H          48100000
#define USB_FREQ_L          47900000

char *const cmsg0 = "Use MSI(internal RC)";
char *const cmsg1 = "freq=";
char *const cmsg2 = "Use HSI(internal RC/High speed)";
char *const cmsg3 = "Use HSE(External Xtal)";
char *const cmsg4 = "Use PLL with";
char *const cmsg5 = "??? following infromation is not valid !";
char *const cmsg6 = "clock freq. =";
char *const cmsg7 = "No clock";
char *const cmsg8 = "Use LSE(external Xtal)=32768Hz";
char *const cmsg9 = "Use LSI(internal RC/Low speed), RC=";
char *const cmsg10= "Use HSE(external Xtal & prescaler)";
//char *const cmsg11= "Power Control";

#endif

#if defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F411RE)

void detail_info_if_available(void)
{
    uint32_t m1 = 0, m2 = 0, m3 = 0, m4 = 0, m5 = 0;

    m1 = RCC->CFGR & RCC_CFGR_SWS;  /* Get SYSCLK source */
    switch (m1) {
        case 0x00:  // HSI used as system clock
            printf( "%s, %s%dHz", cmsg2, cmsg1,HSI_VALUE );
            m2 = HSI_VALUE;
            break;
        case 0x04:  // HSE used as system clock
            printf( "%s, %s%dHz", cmsg3, cmsg1, HSE_VALUE );
            m2 = HSE_VALUE;
            break;
        case 0x08:  // PLL used as system clock
            printf("fVCO = fPLL-in x (PLLN/PLLM), fPLL-out = fVCO/PLLP, fUSB-out = fVCO/PLLQ");
            put_rn();
            m5 = (RCC->PLLCFGR >> 6) & 0x1ff;   // PLLN
            m1 = RCC->PLLCFGR & 0x3f;           // PLLM
            printf( "%s PLLN=%d, PLLM=%d", cmsg4, m5, m1 );
            put_rn();
            m3 = (RCC->PLLCFGR >> 22) & 0x1;    // Clock source
            if (m3 == 0) {
                // HSI oscillator clock selected as PLL clock source
                m2 = (HSI_VALUE * (m5 / m1));
                printf( "%s, RC=%dHz", cmsg2, HSI_VALUE );
            } else {
                // HSE selected
                m2 = (((HSE_VALUE) * m5) / m1);
                if ((RCC->CR >> 18) & 0x01) {  // check HSEBYP bit
                    // HSE(not Xtal) selected as PLL clock source
                    printf( "Use HSE(not Xtal but External Clock)=%dHz", HSE_VALUE );
                } else {
                    // HSE(Xtal) selected as PLL clock source
                    printf( "%s, Xtal=%dHz", cmsg3, HSE_VALUE );
                }
            }
            put_rn();
            printf("PLL/Base   %s%dHz", cmsg1, m2);
            put_rn();
            m3 = (RCC->PLLCFGR >> 16) & 0x03;  // PLLP
            switch (m3) {
                case 0:
                    m3 = 2;
                    break;
                case 1:
                    m3 = 4;
                    break;
                case 2:
                    m3 = 6;
                    break;
                case 3:
                    m3 = 8;
                    break;
            }
            m4 = (RCC->PLLCFGR >> 24) & 0x0f;  // PLLQ
            printf("%s PLLP=%d, PLLQ=%d", cmsg4, m3, m4);
            put_rn();
            printf("PLL/System %s%dHz", cmsg1, m2/m3);
            put_rn();
            printf("PLL/USB    %s%dHz", cmsg1, m2/m4);
            m2 = m2/m4;
            if ((m2 > USB_FREQ_H) || (m2 <USB_FREQ_L)) {
                printf(" -> USB Freq. is out of range!");
            }
            put_rn();
            break;
        default:    // Not come here
            printf( cmsg5 );
            break;
    }
    put_rn();
    printf( "SYSCLK %s%dHz", cmsg6, HAL_RCC_GetSysClockFreq());
    put_rn();
    printf( "HCLK   %s%dHz", cmsg6, HAL_RCC_GetHCLKFreq());
    put_rn();
    printf( "PCLK1  %s%dHz", cmsg6, HAL_RCC_GetPCLK1Freq());
    put_rn();
    printf( "PCLK2  %s%dHz", cmsg6, HAL_RCC_GetPCLK2Freq());
    put_rn();
    put_rn();
    // Check RTC Clock
    printf("RTC Clock");
    put_rn();
    m1 = (RCC->BDCR >> 8) & 0x03;
    switch (m1) {
        case 0: // no clock
            printf(cmsg7);
            break;
        case 1: // LSE
            printf(cmsg8);
            break;
        case 2: // LSI
            printf("%s 17 to 47, typ.32KHz", cmsg9);
            break;
        case 3: // HSE
            printf( cmsg10 );
            m2 = (RCC->PLLCFGR >> 16) & 0x1f;   // RTCPRE
            m3 = HSE_VALUE / m2;
            printf("%s%dHz", cmsg6, m3);
            put_rn();
            break;
        default:    // Not come here
            printf(cmsg5);
            break;
    }
    put_rn();
    put_rn();
}

#elif defined(TARGET_NUCLEO_L152RE)

static __I uint8_t PLLMulTable[9] = {3, 4, 6, 8, 12, 16, 24, 32, 48};

void detail_info_if_available(void)
{
    uint32_t m1, m2, m3, m4, m5;

    m1 = RCC->CFGR & RCC_CFGR_SWS;  /* Get SYSCLK source */
    switch (m1) {
        case 0x00:  // MSI used as system clock
            m4 = ( RCC->ICSCR & RCC_ICSCR_MSIRANGE ) >> 13;
            m2 = (32768 * (1 << (m4 + 1)));
            printf( "%s, %s%dHz", cmsg0, cmsg1, m2);
            break;
        case 0x04:  // HSI used as system clock
            printf( "%s, %s%dHz", cmsg2, cmsg1,HSI_VALUE );
            m2 = HSI_VALUE;
            break;
        case 0x08:  // HSE used as system clock
            printf( "%s, %s%dHz", cmsg3, cmsg1, HSE_VALUE );
            m2 = HSE_VALUE;
            break;
        case 0x0C:  // PLL used as system clock
            // Get PLL clock source and multiplication factor
            m5 = RCC->CFGR & RCC_CFGR_PLLMUL;
            m1 = RCC->CFGR & RCC_CFGR_PLLDIV;
            m5 = PLLMulTable[(m5 >> 18)];
            m1 = (m1 >> 22) + 1;
            printf( "%s Mul=%d, Div=%d", cmsg4, m5, m1 );
            put_rn();
            m3 = RCC->CFGR & RCC_CFGR_PLLSRC;
            if ( m3 == 0x00 ) {
                // HSI oscillator clock selected as PLL clock source
                m2 = (((HSI_VALUE) * m5) / m1);
                printf( "%s, RC=%dHz", cmsg2, HSI_VALUE );
            } else {
                // HSE selected
                m2 = (((HSE_VALUE) * m5) / m1);
                if ((RCC->CR >> 18) & 0x01) {  // check HSEBYP bit
                    // HSE(not Xtal) selected as PLL clock source
                    printf( "Use HSE(not Xtal but External Clock)=%dHz", HSE_VALUE );
                } else {
                    // HSE(Xtal) selected as PLL clock source
                    printf( "%s, Xtal=%dHz", cmsg3, HSE_VALUE );
                }
            }
            put_rn();
            printf( "PLL %s%dHz", cmsg1, m2 );
            put_rn();
            break;
        default:    // Not come here
            printf( cmsg5 );
            break;
    }
    put_rn();
#if 1   // Nucleo L152RE does NOT support folloing function call (Please set #if 0)
    printf( "SYSCLK %s%dHz", cmsg6, HAL_RCC_GetSysClockFreq() );
    put_rn();
    printf( "HCLK   %s%dHz", cmsg6, HAL_RCC_GetHCLKFreq() );
    put_rn();
    printf( "PCLK1  %s%dHz", cmsg6, HAL_RCC_GetPCLK1Freq() );
    put_rn();
    printf( "PCLK2  %s%dHz", cmsg6, HAL_RCC_GetPCLK2Freq() );
#endif
    put_rn();
    put_rn();
    m1 = RCC->CSR & RCC_CSR_RTCSEL;
    // Check RTC & LCD Clock
    printf("RTC/LCD Clock");
    put_rn();
    switch (m1) {
        case RCC_CSR_RTCSEL_NOCLOCK:
            printf( cmsg7 );
            break;
        case RCC_CSR_RTCSEL_LSE:
            printf( cmsg8 );
            break;
        case RCC_CSR_RTCSEL_LSI:
            printf("%s 26 to 56, typ.38KHz", cmsg9);
            break;
        case RCC_CSR_RTCSEL_HSE:
            printf( cmsg10 );
            break;
        default:    // Not come here
            printf( cmsg5 );
            break;
    }
    put_rn();
    put_rn();
}

#elif defined(TARGET_LPC1768)

#define XTAL                (12000000UL)        // Oscillator frequency
#define OSC_CLK             (      XTAL)        // Main oscillator frequency
#define RTC_CLK             (   32000UL)        // RTC oscillator frequency
#define IRC_OSC             ( 4000000UL)        // Internal RC oscillator frequency

void detail_info_if_available(void)
{
    uint32_t sys_freq = 0;

    if (((LPC_SC->PLL0STAT >> 24) & 3) == 3) {/* If PLL0 enabled and connected      */
        printf("PLL0 enabled");
        put_rn();
        switch (LPC_SC->CLKSRCSEL & 0x03) {
            case 0:                /* Internal RC oscillator => PLL0     */
            case 3:                /* Reserved, default to Internal RC   */
                printf("Internal RC Oscillator");
                put_rn();
                sys_freq = (IRC_OSC *
                                   (((2 * ((LPC_SC->PLL0STAT & 0x7FFF) + 1))) /
                                    (((LPC_SC->PLL0STAT >> 16) & 0xFF) + 1))   /
                                   ((LPC_SC->CCLKCFG & 0xFF)+ 1));
                break;
            case 1:                /* Main oscillator => PLL0            */
                printf("Xtal Osc Clock = %dHz",XTAL );
                put_rn();
                sys_freq = (OSC_CLK *
                                   (((2 * ((LPC_SC->PLL0STAT & 0x7FFF) + 1))) /
                                    (((LPC_SC->PLL0STAT >> 16) & 0xFF) + 1))   /
                                   ((LPC_SC->CCLKCFG & 0xFF)+ 1));
                break;
            case 2:                /* RTC oscillator => PLL0             */
                printf("RTC Xtal Oscillator f = %dHz", RTC_CLK );
                put_rn();
                sys_freq = (RTC_CLK *
                                   (((2 * ((LPC_SC->PLL0STAT & 0x7FFF) + 1))) /
                                    (((LPC_SC->PLL0STAT >> 16) & 0xFF) + 1))   /
                                   ((LPC_SC->CCLKCFG & 0xFF)+ 1));
                break;
        }
    } else {
        printf("PLL0 disabled");
        switch (LPC_SC->CLKSRCSEL & 0x03) {
            case 0:                /* Internal RC oscillator => PLL0     */
            case 3:                /* Reserved, default to Internal RC   */
                printf("Internal RC Oscillator");
                put_rn();
                sys_freq = IRC_OSC / ((LPC_SC->CCLKCFG & 0xFF)+ 1);
                break;
            case 1:                /* Main oscillator => PLL0            */
                printf("Xtal Osc Clock = %dHz",XTAL );
                put_rn();
                sys_freq = OSC_CLK / ((LPC_SC->CCLKCFG & 0xFF)+ 1);
                break;
            case 2:                /* RTC oscillator => PLL0             */
                printf("RTC Xtal Oscillator f = %dHz", RTC_CLK );
                put_rn();
                sys_freq = RTC_CLK / ((LPC_SC->CCLKCFG & 0xFF)+ 1);
                break;
        }
    }
    printf("System Clock = %dHz", sys_freq);
}

#else

void detail_info_if_available(void)
{
    printf("Detail information -> No support");
    put_rn();
    put_rn();
}

#endif

void put_rn(void)
{
    printf("\r\n");
}
