/*
 * Copyright (c) 2018, Arm Limited. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 *
 */

#ifndef __TFM_PLAT_BOOT_SEED_H__
#define __TFM_PLAT_BOOT_SEED_H__
/**
 * \file tfm_plat_boot_seed.h
 *
 * Boot seed is used by a validating entity to ensure multiple reports were
 * generated in the same boot session. Boot seed is a random number, generated
 * only once during a boot cycle and its value is constant in the same cycle.
 * Size recommendation is 256-bit to meet the statistically improbable property.
 * Boot seed can be generated by secure boot loader an included to the measured
 * boot state or can be generated by PRoT SW.
 */

/**
 * \note The interfaces defined in this file must be implemented for each
 *       SoC.
 */

#include <stdint.h>
#include "tfm_plat_defs.h"

#ifdef __cplusplus
extern "C" {
#endif

/*!
 * \def BOOT_SEED_SIZE
 *
 * \brief Size of boot seed in bytes.
 */
#define BOOT_SEED_SIZE (32u)

/**
 * \brief Gets the boot seed, which is a constant random number during a boot
 *        cycle.
 *
 * \param[in]  size The required size of boot seed in bytes
 * \param[out] buf  Pointer to the buffer to store boot seed
 *
 * \return  TFM_PLAT_ERR_SUCCESS if the value is generated correctly. Otherwise,
 *          it returns TFM_PLAT_ERR_SYSTEM_ERR.
 */
enum tfm_plat_err_t tfm_plat_get_boot_seed(uint32_t size, uint8_t *buf);

#ifdef __cplusplus
}
#endif

#endif /* __TFM_PLAT_BOOT_SEED_H__ */
