/*************************************************************************************************/
/*!
 *  \brief  LL initialization for controller configuration.
 *
 *  Copyright (c) 2013-2018 Arm Ltd. All Rights Reserved.
 *  Arm Ltd. confidential and proprietary.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
/*************************************************************************************************/

#include "ll_init_api.h"
#include "lhci_api.h"
#include "chci_api.h"

/**************************************************************************************************
  Functions
**************************************************************************************************/

/*! \brief      Extended VS command decoder. */
extern bool_t lhciVsExtDecodeCmdPkt(LhciHdr_t *pHdr, uint8_t *pBuf);

/*************************************************************************************************/
/*!
 *  \brief  Initialize controller LHCI handler.
 *
 *  \return None.
 */
/*************************************************************************************************/
void LlInitLhciHandler(void)
{
  wsfHandlerId_t handlerId;

#if (BT_VER >= LL_VER_BT_CORE_SPEC_MILAN)
  handlerId = WsfOsSetNextHandler(LhciIsoHandler);
  LhciIsoHandlerInit(handlerId);
#else
  handlerId = WsfOsSetNextHandler(LhciHandler);
  LhciHandlerInit(handlerId);
#endif
}

/*************************************************************************************************/
/*!
 *  \brief  Initialize controller HCI transport.
 *
 *  \return None.
 */
/*************************************************************************************************/
void LlInitChciTrInit(void)
{
  wsfHandlerId_t handlerId = WsfOsSetNextHandler(ChciTrHandler);
  ChciTrHandlerInit(handlerId);
}

/*************************************************************************************************/
/*!
 *  \brief  Initialize LL HCI.
 *
 *  \return None.
 */
/*************************************************************************************************/
void LlInitLhciInit(void)
{
  #ifdef INIT_CENTRAL
    LhciScanMasterInit();
    #ifdef INIT_ENCRYPTED
      LhciEncMasterInit();
    #endif
    LhciConnMasterInit();
  #else
    #ifdef INIT_OBSERVER
      LhciScanMasterInit();
    #endif
  #endif

  #ifdef INIT_PERIPHERAL
    LhciAdvSlaveInit();
    #ifdef INIT_ENCRYPTED
      LhciEncSlaveInit();
    #endif
    LhciConnInit();
  #else
    #ifdef INIT_BROADCASTER
      LhciAdvSlaveInit();
    #endif
  #endif

  #ifdef INIT_ENCRYPTED
    LhciPrivAdvInit();
    LhciPrivConnInit();
    LhciScInit();
  #endif

  LhciVsExtInit(lhciVsExtDecodeCmdPkt);

  #if (BT_VER >= LL_VER_BT_CORE_SPEC_5_0)
    #ifdef INIT_CENTRAL
      LhciExtScanMasterInit();
      LhciExtConnMasterInit();
      #if (BT_VER >= LL_VER_BT_CORE_SPEC_MILAN)
        LhciCisMasterInit();
      #endif
    #else
      #ifdef INIT_OBSERVER
        LhciExtScanMasterInit();
      #endif
    #endif

    #ifdef INIT_BROADCASTER
      LhciExtAdvSlaveInit();
    #endif

    #ifdef INIT_PERIPHERAL
      #if (BT_VER >= LL_VER_BT_CORE_SPEC_MILAN)
        LhciCisSlaveInit();
      #endif
    #endif

    #if defined(INIT_PERIPHERAL) || defined(INIT_CENTRAL)
      LhciChannelSelection2Init();
      LhciPhyInit();
      #if defined(INIT_OBSERVER) || defined(INIT_BROADCASTER)
        #if (BT_VER >= LL_VER_BT_CORE_SPEC_5_1)
          LhciPastInit();
        #endif
      #endif
    #endif
  #endif

  #if (BT_VER >= LL_VER_BT_CORE_SPEC_MILAN)
    LhciIsoInit();
  #endif
}

/*************************************************************************************************/
/*!
 *  \brief  Initialize controller configuration.
 *
 *  \param  pCfg    Runtime configuration.
 *
 *  \return Memory used.
 */
/*************************************************************************************************/
uint32_t LlInitControllerInit(LlInitRtCfg_t *pCfg)
{
  uint32_t totalMemUsed;

  totalMemUsed = LlInit(pCfg);

  LlInitChciTrInit();
  LlInitLhciInit();
  LlInitLhciHandler();

  return totalMemUsed;
}
