/*
 * mbed Application program for the mbed LPC1114FN28
 * Monitor program Ver.3 for only for LPC1114FN28
 *
 *  Copyright (c) 2010-2014 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created:  May       15th, 2010
 *          release as "monitor_01" http://mbed.org/users/kenjiArai/code/monitor_01/
 *      Spareted: June      25th, 2014      mon() & mon_hw()
 *      Revised:  Nobember   2nd, 2014
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#if defined(TARGET_LPC1114)

//  Include ---------------------------------------------------------------------------------------
#include "mbed.h"
#include "mon_hw_config.h"
#include "mon_hw_common.h"

//  Object ----------------------------------------------------------------------------------------

//  Definition ------------------------------------------------------------------------------------
// Define clocks
#define __XTAL              (12000000UL)    // Oscillator frequency
#define __SYS_OSC_CLK       (    __XTAL)    // Main oscillator frequency
#define __IRC_OSC_CLK       (12000000UL)    // Internal RC oscillator frequency

//  RAM -------------------------------------------------------------------------------------------
uint32_t SystemFrequency;

//  ROM / Constant data ---------------------------------------------------------------------------
#if USE_MEM
//  Memory range data
const uint32_t mem_range[][2] = {          // Memory access range
    { 0x00000000, 0x00007fff },            // On-chip non-volatile memory     //32KB Flash memory
    { 0x10000000, 0x10000fff },            // On-chip SRAM                    //4KB local RAM
    { 0x1fff0000, 0x1fff3fff },            // Boot ROM                        //16KB Boot ROM
    { 0x40000000, 0x4007ffff },            // IO area
    { 0x50000000, 0x501fffff }             // IO area
};
#endif  // USE_MEM

char *const mon_msg = "HW monitor only for mbed LPC1114FN28 created on "__DATE__","__TIME__"";

char *const xmsg0 = "Not implimented yet";

char *const hmsg0 = "m  - Entry Memory Mode";
char *const hmsg1 = "m>? -> Help ";
char *const hmsg2 = "r  - Show PORT,I2C,SPI,UART & other Reg.";
char *const hmsg3 = "r>? -> Help";
char *const hmsg4 = "sf - System Clock";
char *const hmsg5 = "sc - System / CPU information";
char *const hmsg6 = "x  - Special command for Debug";
char *const hmsg7 = "q  - Quit (back to called routine)";

#if (USE_PORT==0)||(USE_UART==0)||(USE_SPI==0)||(USE_I2C==0)
char *const io_port_name0 = "PIO0_";
char *const io_port_name1 = "PIO1_";
char *const iomsg0 = "Func->select ";
char *const iomsg1 = "IO";
char *const iomsg2 = "Reserved";
char *const iomsg30 = "B0_MAT";
char *const iomsg31 = "B1_MAT";
char *const iomsg4 = "Std/F-md I2C";
char *const iomsg5 = "func. R";
char *const iomsg6 = "D-Mode";
char *const iomsg7 = "A-Mode";
#endif

//  Calculate CPU System Clock Frequency /refrence: system_LPCxx.c
char *const fmsg0 = "Internal RC Oscillator";
char *const fmsg1 = "Xtal Osc Clock";
char *const fmsg2 = "Watch dog Osc Clock";
char *const fmsg3 = "with PLL";
char *const fmsg4 = "System Clock =";
char *const fmsg5 = "PLL Post divider ratio =";
char *const fmsg6 = "feedback devider =";
char *const fmsg7 = "NO Clock ?!";

#if (USE_UART==1) || (USE_SPI==1) || (USE_I2C == 1)
char *const imsg2 = "-->Control Reg.";
char *const imsg3 = "-->Status Reg.";
char *const imsg4 = "-->Data Reg.";
char *const imsg5 = "-->Clock control Reg.";
#endif

//  Function prototypes ---------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------------
//  Control Program
//-------------------------------------------------------------------------------------------------
// No function
void not_yet_impliment( void )
{
    PRINTF(xmsg0);
    put_rn();
}

// No function
#if (USE_MEM==0)||(USE_PORT==0)||(USE_UART==0)||(USE_SPI==0)||(USE_I2C==0)||(USE_SYS==0)
void not_select( void )
{
    PRINTF("Not select the function (refer mon_hw_config.h)");
    put_rn();
}
#endif

//  Help Massage
void msg_hlp_hw (void)
{
    PRINTF(mon_msg);
    put_rn();
    PRINTF(hmsg0);
    put_rn();
    PRINTF(hmsg1);
    put_rn();
    PRINTF(hmsg2);
    put_rn();
    PRINTF(hmsg3);
    put_rn();
    PRINTF(hmsg4);
    put_rn();
    PRINTF(hmsg5);
    put_rn();
    PRINTF(hmsg6);
    put_rn();
    PRINTF(hmsg7);
    put_rn();
}

#if USE_MEM
char *const rmsg0 = "FLASH     ";
char *const rmsg1 = "SRAM      ";
char *const rmsg2 = "Boot ROM  ";
char *const rmsg3 = "IO Area0  ";
char *const rmsg4 = "IO Area1  ";

#include "mon_hw_mem.h"
#endif   // USE_MEM

//  Show 16bit register contents
void reg_print(uint16_t size, uint16_t reg)
{
    uint16_t i, j, k, n;

    if (size == 8) {
        PRINTF(" 7, 6, 5, 4, 3, 2, 1, 0");
        put_rn();
        i = 8;
        n = 0x80;
    } else if (size == 16) {
        PRINTF( "15,14,13,12,11,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0" );
        put_rn();
        i = 16;
        n = 0x8000;
    } else {
        PRINTF("0x%08x", reg);
        return;
    }
    PUTC(' ');
    for (; i>0; i--) {
        k = n >> (size-i);
        j = reg & k;
        if (j) {
            PUTC('1');
        } else {
            PUTC('0');
        }
        PUTC(' ');
        PUTC(' ');
    }
    PRINTF("  (0x%04x)", reg);
}

#if USE_PORT
void io_mode(uint32_t reg)
{
    PRINTF("MODE->");
    switch (reg) {
        case 0:
            PRINTF("Inactive");
            break;
        case 1:
            PRINTF("P-DWN");
            break;
        case 2:
            PRINTF("P-UP");
            break;
        case 3:
            PRINTF("Repeater");
            break;
    }
}

void io_hys(uint32_t reg)
{
    PRINTF("HIS->");
    switch (reg) {
        case 0:
            PRINTF("Dis");
            break;
        case 1:
            PRINTF("Ena");
            break;
    }
}

void io_od(uint32_t reg)
{
    PRINTF("OD->");
    switch (reg) {
        case 0:
            PRINTF("no OD");
            break;
        case 1:
            PRINTF("OD");
            break;
    }
}

void io_mode_hys_od(uint32_t reg)
{
    io_mode ((reg >>  3) & 0x3);
    put_lin();
    io_hys (( reg >>  5) & 0x1);
    put_lin();
    io_od (( reg >> 10) & 0x1);
}

// I/O Config IO0_x
void io_config0(void)
{
    uint32_t r0;

    // P0_0
    r0 = LPC_IOCON->RESET_PIO0_0;
    PRINTF("RESET_%s0(dp23)", io_port_name0);
    put_spc(1);
    reg_print( SIZE_X, r0 );
    put_spc(2);
    PRINTF( iomsg0 );
    if ((r0 & 0x7) == 0) {
        PRINTF("RESET");
    } else {
        PRINTF( iomsg1 );
    }
    put_lin();
    io_mode_hys_od( r0 );
    put_rn();
    // P0_1
    r0 = LPC_IOCON->PIO0_1;
    PRINTF("%s1(dp24)", io_port_name0);
    put_spc(3);
    reg_print( SIZE_X, r0 );
    put_spc(2);
    PRINTF( iomsg0 );
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("CLKOUT");
            break;
        case 2:
            PRINTF("32%s2", iomsg30);
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_2
    r0 = LPC_IOCON->PIO0_2;
    PRINTF("%s2(dp25)",io_port_name0);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("SSEL0");
            break;
        case 2:
            PRINTF("16B0_CAP0");
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_3
    r0 = LPC_IOCON->PIO0_3;
    PRINTF("%s3(dp26)",io_port_name0);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_4
    r0 = LPC_IOCON->PIO0_4;
    PRINTF("%s4(dp27)",io_port_name0);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("SCL");
            break;
    }
    put_lin();
    switch (( r0 >> 8 ) & 0x3) {
        case 0:
            PRINTF(iomsg4);
            break;
        case 1:
            PRINTF(iomsg1);
            break;
        case 2:
            PRINTF("Fast md");
            break;
        case 3:
            PRINTF(iomsg2);
            break;
    }
    put_rn();
    // P0_5
    r0 = LPC_IOCON->PIO0_5;
    PRINTF("%s5(dp5)",io_port_name0);
    put_spc(4);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("SDA");
            break;
    }
    put_lin();
    switch ( ( r0 >> 8 ) & 0x3 ) {
        case 0:
            PRINTF(iomsg4);
            break;
        case 1:
            PRINTF(iomsg1);
            break;
        case 2:
            PRINTF("Fast md");
            break;
        case 3:
            PRINTF(iomsg2);
            break;
    }
    put_rn();
    // P0_6
    r0 = LPC_IOCON->PIO0_6;
    PRINTF("%s6(dp6)", io_port_name0);
    put_spc(4);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF(iomsg2);
            break;
        case 2:
            PRINTF("SCK0");
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_7
    r0 = LPC_IOCON->PIO0_7;
    PRINTF("%s7(dp28)", io_port_name0);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("CTS");
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_8
    r0 = LPC_IOCON->PIO0_8;
    PRINTF("%s8(dp1)", io_port_name0);
    put_spc(4);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("MISO0");
            break;
        case 2:
            PRINTF("16%s0", iomsg30);
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_9
    r0 = LPC_IOCON->PIO0_9;
    PRINTF("%s9(dp2)", io_port_name0);
    put_spc(4);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("MOSI0");
            break;
        case 2:
            PRINTF("16%s1", iomsg30);
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_10
    r0 = LPC_IOCON->SWCLK_PIO0_10;
    PRINTF("SWCLK_%s10(dp3)", io_port_name0);
    put_spc(1);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF("SWCLK");
            break;
        case 1:
            PRINTF(iomsg1);
            break;
        case 2:
            PRINTF("SCK0");
            break;
        case 3:
            PRINTF("16%s2", iomsg30);
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_11
    r0 = LPC_IOCON->R_PIO0_11;
    PRINTF("R_%s11(dp4)", io_port_name0);
    put_spc(1);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF( iomsg0 );
    switch ( r0 & 0x7 ) {
        case 0:
            PRINTF(iomsg5);
            break;
        case 1:
            PRINTF(iomsg1);
            break;
        case 2:
            PRINTF("AD0");
            break;
        case 3:
            PRINTF("32%s3", iomsg30);
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    if ( r0 & 0x80 ) {
        PRINTF(", %s", iomsg6);
    } else {
        PRINTF(", %s", iomsg7);
    }
    put_rn();
}

// I/O Config IO1_x
void io_config1(void)
{
    uint32_t r0;

    // P1_0
    r0 = LPC_IOCON->R_PIO1_0;
    PRINTF("R_%s0(dp9)", io_port_name1);
    put_spc(2);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF( iomsg0 );
    switch ( r0 & 0x7 ) {
        case 0:
            PRINTF(iomsg5);
            break;
        case 1:
            PRINTF(iomsg1);
            break;
        case 2:
            PRINTF("AD1");
            break;
        case 3:
            PRINTF("32B1_CAP0");
            break;
    }
    io_mode_hys_od(r0);
    if (r0 & 0x80) {
        PRINTF(", %s", iomsg6);
    } else {
        PRINTF(", %s", iomsg7);
    }
    put_rn();
    // P1_1
    r0 = LPC_IOCON->R_PIO1_1;
    PRINTF("R_%s1(dp10)", io_port_name1);
    put_spc(1);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg5);
            break;
        case 1:
            PRINTF(iomsg1);
            break;
        case 2:
            PRINTF("AD2");
            break;
        case 3:
            PRINTF("32%s0", iomsg31);
            break;
    }
    io_mode_hys_od(r0);
    if (r0 & 0x80) {
        PRINTF(", %s", iomsg6);
    } else {
        PRINTF(", %s", iomsg7);
    }
    put_rn();
    // P1_2
    r0 = LPC_IOCON->R_PIO1_2;
    PRINTF("R_%s2(dp11)", io_port_name1);
    put_spc(1);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg5);
            break;
        case 1:
            PRINTF(iomsg1);
            break;
        case 2:
            PRINTF("AD3");
            break;
        case 3:
            PRINTF("32%s1", iomsg31);
            break;
    }
    io_mode_hys_od( r0 );
    if (r0 & 0x80) {
        PRINTF(", %s", iomsg6);
    } else {
        PRINTF(", %s", iomsg7);
    }
    put_rn();
    // P1_3
    r0 = LPC_IOCON->SWDIO_PIO1_3;
    PRINTF("SWDIO_%s3(dp12)",io_port_name1);
    put_spc(1);
    reg_print(SIZE_X, r0);
    put_spc(3);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF("SWDIO");
            break;
        case 1:
            PRINTF(iomsg1);
            break;
        case 2:
            PRINTF("AD4");
            break;
        case 3:
            PRINTF("32%s2", iomsg31);
            break;
    }
    io_mode_hys_od(r0);
    if (r0 & 0x80) {
        PRINTF(", %s", iomsg6);
    } else {
        PRINTF(", %s", iomsg7);
    }
    put_rn();
    // P1_4
    r0 = LPC_IOCON->PIO1_4;
    PRINTF("%s4(dp13)",io_port_name1);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF( iomsg0 );
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("AD5");
            break;
        case 2:
            PRINTF("32%s3", iomsg31);
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P1_5
    r0 = LPC_IOCON->PIO1_5;
    PRINTF("%s5(dp14)",io_port_name1);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("RTS");
            break;
        case 2:
            PRINTF("32B0_CAP0");
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P1_6
    r0 = LPC_IOCON->PIO1_6;
    PRINTF("%s6(dp15)", io_port_name1);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("RXD");
            break;
        case 2:
            PRINTF( "32%s0", iomsg30 );
            break;
    }
    put_lin();
    io_mode_hys_od( r0 );
    put_rn();
    // P1_7
    r0 = LPC_IOCON->PIO1_7;
    PRINTF("%s7(dp16)", io_port_name1);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("TXD");
            break;
        case 2:
            PRINTF("32%s1", iomsg30);
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P1_8
    r0 = LPC_IOCON->PIO1_8;
    PRINTF("%s8(dp17)", io_port_name1);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("32%s1", iomsg30);
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P1_9
    r0 = LPC_IOCON->PIO1_9;
    PRINTF("%s9(dp18)", io_port_name1);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("16%s0", iomsg31);
            break;
        case 2:
            PRINTF("MOSI1");
            break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
}
#endif  // USE_PORT

#if USE_SPI
void spi_reg(int no)
{
    uint32_t n, r0, r1 ,r2, r3, r4, r5, r6, r7, r8;

    if (no == SPI_0) {
        n = 0;
        r0 = LPC_SSP0->CR1;
        r1 = LPC_SSP0->DR;
        r2 = LPC_SSP0->SR;
        r3 = LPC_SSP0->CPSR;
        r4 = LPC_SSP0->IMSC;
        r5 = LPC_SSP0->RIS;
        r6 = LPC_SSP0->MIS;
        r7 = LPC_SSP0->ICR;
        r8 = LPC_SSP0->CR0;
    } else if (no == SPI_1) {
        n = 1;
        r0 = LPC_SSP1->CR1;
        r1 = LPC_SSP1->DR;
        r2 = LPC_SSP1->SR;
        r3 = LPC_SSP1->CPSR;
        r4 = LPC_SSP1->IMSC;
        r5 = LPC_SSP1->RIS;
        r6 = LPC_SSP1->MIS;
        r7 = LPC_SSP1->ICR;
        r8 = LPC_SSP1->CR0;
    } else {
        return;
    }
    PRINTF("Show SSP%1d(SPI%1d) Registers", n, n);
    put_rn();
    PRINTF("CR0");
    PRINTF(imsg2);
    reg_print(SIZE8, r8);
    put_rn();
    PRINTF("CR1");
    PRINTF(imsg2);
    reg_print(SIZE8, r0);
    put_rn();
    PRINTF("DR");
    PRINTF(imsg4);
    reg_print(SIZE8, r1);
    put_rn();
    PRINTF("SR");
    PRINTF(imsg3);
    reg_print(SIZE8, r2);
    put_rn();
    PRINTF("CPSR");
    PRINTF(imsg5);
    reg_print(SIZE8, r3);
    put_rn();
    PRINTF("IMSC");
    PRINTF(imsg2);
    reg_print(SIZE8, r4);
    put_rn();
    PRINTF("RIS");
    PRINTF(imsg3);
    reg_print(SIZE8, r5);
    put_rn();
    PRINTF("MIS");
    PRINTF(imsg3);
    reg_print(SIZE8, r6);
    put_rn();
    PRINTF("ICR");
    PRINTF(imsg2);
    reg_print(SIZE8, r7);
    put_rn();
}
#endif  //USE_SPI

#if USE_UART
void uart_reg(void)
{
    uint32_t r0,r1,r2,r3,r4,r5,r6;

    // clear LCR[DLAB] to read registers
    LPC_UART->LCR &= ~(1 << 7);
    r0 = LPC_UART->RBR;
    r1 = LPC_UART->IER;
    r2 = LPC_UART->IIR;
    r3 = LPC_UART->TER;
    // set LCR[DLAB] to enable writing to divider registers
    LPC_UART->LCR |= (1 << 7);
    r4 = LPC_UART->DLL;
    r5 = LPC_UART->DLM;
    r6 = LPC_UART->FDR;
    // clear LCR[DLAB]
    LPC_UART->LCR &= ~(1 << 7);
    // Print each register
    PRINTF("Show UART Registers");
    put_rn();
    PRINTF("RBR");
    PRINTF(imsg4);
    reg_print(SIZE8, r0);
    put_rn();
    PRINTF("THR--Write only");
    PRINTF(imsg4);
    put_rn();
    PRINTF("DLL");
    PRINTF(imsg2);
    reg_print(SIZE8, r4);
    put_rn();
    PRINTF("DLM");
    PRINTF(imsg2);
    reg_print(SIZE8, r5);
    put_rn();
    PRINTF("IER");
    PRINTF(imsg2);
    reg_print(SIZE8, r1);
    put_rn();
    PRINTF("IIR");
    PRINTF(imsg2);
    reg_print(SIZE8, r2);
    put_rn();
    PRINTF("FCR--Write only");
    PRINTF(imsg2);
    put_rn();
    PRINTF("LCR,MCR,MSR,SCR,ACR--Not support");
    put_rn();
    PRINTF("FDR");
    PRINTF(imsg2);
    reg_print(SIZE8, r6);
    put_rn();
    PRINTF("TER");
    PRINTF(imsg3);
    reg_print(SIZE8, r3);
    put_rn();
    PRINTF("RS485CTRL,485ADRMATCH,485DLY--Not support");
    put_rn();
}
#endif  //USE_UART

//#if USE_SYS
void cpu_inf (void)
{
    unsigned long m1, m2;

    m1 = SCB->CPUID;
    m2 = (m1 >> 24);
    if (m2 == 0x41) {
        put_r();
        PRINTF("CPU = ARM ");
    } else {
        put_r();
        PRINTF("CPU = NOT ARM ");
    }
    m2 = (m1 >> 4) & 0xfff;
    if (m2 == 0xc23) {
        PRINTF("Cortex-M3");
        put_rn();
    } else if (m2 == 0xc20) {
        PRINTF("Cortex-M0");
        put_rn();
    } else {
        PRINTF("NOT Cortex-M3,M0");
        put_rn();
    }
    m2 = (m1 >> 20) & 0x0f;
    PRINTF("Variant:%x", m2);
    put_rn();
    m2 = m1 & 0x7;
    PRINTF("Revision:%x", m2);
    put_rn();
#if 0
    // unique ID
    // http://nemuisan.blog.bai.ne.jp/?eid=192845
    m1 = *(__IO uint32_t *)((uint32_t)0x1FFF7A22);
    PRINTF("Unique device ID(94bits):(1) 0x%08x ", m1);
    m1 = *(__IO uint32_t *)((uint32_t)0x1FFF7A26);
    PRINTF("(2) 0x%08x ", m1);
    m1 = *(__IO uint32_t *)((uint32_t)0x1FFF7A2A);
    PRINTF( "(3) 0x%08x", m1 );
    put_rn();
//    m1 = DBGMCU->IDCODE;
//    PRINTF( "DBG ID: 0x%08x", m1 );
#endif
}

//  Calculate CPU System Clock Frequency
void get_freq(int pr)
{
    uint32_t wdt_osc = 0, pll_fct0 = 0, pll_fct1 = 0;

    // Determine clock frequency according to clock register values
    switch ((LPC_SYSCON->WDTOSCCTRL >> 5) & 0x0F) {
        case 0:
            wdt_osc =  400000;
            break;
        case 1:
            wdt_osc =  500000;
            break;
        case 2:
            wdt_osc =  800000;
            break;
        case 3:
            wdt_osc = 1100000;
            break;
        case 4:
            wdt_osc = 1400000;
            break;
        case 5:
            wdt_osc = 1600000;
            break;
        case 6:
            wdt_osc = 1800000;
            break;
        case 7:
            wdt_osc = 2000000;
            break;
        case 8:
            wdt_osc = 2200000;
            break;
        case 9:
            wdt_osc = 2400000;
            break;
        case 10:
            wdt_osc = 2600000;
            break;
        case 11:
            wdt_osc = 2700000;
            break;
        case 12:
            wdt_osc = 2900000;
            break;
        case 13:
            wdt_osc = 3100000;
            break;
        case 14:
            wdt_osc = 3200000;
            break;
        case 15:
            wdt_osc = 3400000;
            break;
    }
    wdt_osc /= ((LPC_SYSCON->WDTOSCCTRL & 0x1F) << 1) + 2;
    put_r();
    switch (LPC_SYSCON->MAINCLKSEL & 0x03) {
        case 0:                             // Internal RC oscillator
            SystemCoreClock = __IRC_OSC_CLK;
            if(pr) {
                PRINTF("%s = %dHz", fmsg0, SystemCoreClock);
            }
            break;
        case 1:                             // Input Clock to System PLL
            switch (LPC_SYSCON->SYSPLLCLKSEL & 0x03) {
                case 0:                         // Internal RC oscillator
                    SystemCoreClock = __IRC_OSC_CLK;
                    if(pr) {
                        PRINTF("%s = %dHz", fmsg0, SystemCoreClock);
                    }
                    break;
                case 1:                         // System oscillator
                    SystemCoreClock = __SYS_OSC_CLK;
                    if(pr) {
                        PRINTF("%s = %dHz", fmsg1, SystemCoreClock );
                    }
                    break;
                case 2:                         // WDT Oscillator
                    SystemCoreClock = wdt_osc;
                    if(pr) {
                        PRINTF("%s = %dHz", fmsg2, wdt_osc );
                    }
                    break;
                case 3:                         // Reserved
                    SystemCoreClock = 0;
                    if(pr) {
                        PRINTF(fmsg7);
                    }
                    break;
            }
            break;
        case 2:                             // WDT Oscillator
            SystemCoreClock = wdt_osc;
            if(pr) {
                PRINTF("%s = %dHz", fmsg2, wdt_osc );
            }
            break;
        case 3:                             // System PLL Clock Out
            switch (LPC_SYSCON->SYSPLLCLKSEL & 0x03) {
                case 0:                         // Internal RC oscillator
                    if (LPC_SYSCON->SYSPLLCTRL & 0x180) {
                        SystemCoreClock = __IRC_OSC_CLK;
                        if(pr) {
                            PRINTF("%s = %dHz", fmsg0, SystemCoreClock);
                        }
                    } else {
                        pll_fct0 = (LPC_SYSCON->SYSPLLCTRL & 0x01F) + 1;
                        pll_fct1 = (LPC_SYSCON->SYSPLLCTRL & 0x060) >> 5UL;
                        SystemCoreClock = __IRC_OSC_CLK *  pll_fct0;
                        if(pr) {
                            PRINTF("Use Internal RC = %dHz", __IRC_OSC_CLK);
                            put_rn();
                            PRINTF("%s %s = %dHz", fmsg0, fmsg3, SystemCoreClock);
                            put_rn();
                            PRINTF("%s %d, %s %d", fmsg5, pll_fct1, fmsg6, pll_fct0);
                        }
                    }
                    break;
                case 1:                         // System oscillator
                    if (LPC_SYSCON->SYSPLLCTRL & 0x180) {
                        SystemCoreClock = __SYS_OSC_CLK;
                        if(pr) {
                            PRINTF("%s = %dHz", fmsg1, SystemCoreClock );
                        }
                    } else {
                        pll_fct0 = (LPC_SYSCON->SYSPLLCTRL & 0x01F) + 1;
                        pll_fct1 = (LPC_SYSCON->SYSPLLCTRL & 0x060) >> 5UL;
                        SystemCoreClock = __SYS_OSC_CLK * pll_fct0;
                        if(pr) {
                            PRINTF("Use XTAL = %dHz", __XTAL);
                            put_rn();
                            PRINTF("%s %s = %dHz", fmsg1, fmsg3, SystemCoreClock);
                            put_rn();
                            PRINTF("%s %d, %s %d", fmsg5, pll_fct1, fmsg6, pll_fct0);
                        }
                    }
                    break;
                case 2:                         // WDT Oscillator
                    if (LPC_SYSCON->SYSPLLCTRL & 0x180) {
                        SystemCoreClock = wdt_osc;
                        if(pr) {
                            PRINTF("%s = %dHz", fmsg2, wdt_osc );
                        }
                    } else {
                        pll_fct0 = (LPC_SYSCON->SYSPLLCTRL & 0x01F) + 1;
                        pll_fct1 = (LPC_SYSCON->SYSPLLCTRL & 0x060) >> 5UL;
                        SystemCoreClock = wdt_osc *  pll_fct0;
                        if(pr) {
                            PRINTF("Use WDT OSC = %dHz", wdt_osc);
                            put_rn();
                            PRINTF("%s %s = %dHz", fmsg2, fmsg3, SystemCoreClock);
                            put_rn();
                            PRINTF("%s %d, %s %d", fmsg5, pll_fct1, fmsg6, pll_fct0);
                        }
                    }
                    break;
                case 3:                         // Reserved
                    SystemCoreClock = 0;
                    if(pr) {
                        PRINTF("fmsg7");
                    }
                    break;
            }
            break;
    }
    SystemCoreClock /= LPC_SYSCON->SYSAHBCLKDIV;
    if(pr) {
        put_rn();
        PRINTF("%s %dHz", fmsg4, SystemCoreClock);
        put_rn();
    }
}
//#endif  //USE_SYS

#if USE_I2C
void i2c_io_reg (void)
{
    uint32_t r0;

    PRINTF("<Show IO Pin>");
    put_rn();
    // P0_4
    r0 = LPC_IOCON->PIO0_4;
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF("%s4(dp27)",io_port_name0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("SCL");
            break;
    }
    put_lin();
    switch (( r0 >> 8 ) & 0x3) {
        case 0:
            PRINTF(iomsg4);
            break;
        case 1:
            PRINTF(iomsg1);
            break;
        case 2:
            PRINTF("Fast md");
            break;
        case 3:
            PRINTF(iomsg2);
            break;
    }
    put_rn();
    // P0_5
    r0 = LPC_IOCON->PIO0_5;
    reg_print( SIZE_X, r0 );
    put_spc(2);
    PRINTF("%s5(dp5)",io_port_name0);
    put_spc(3);
    PRINTF(iomsg0);
    switch (r0 & 0x7) {
        case 0:
            PRINTF(iomsg1);
            break;
        case 1:
            PRINTF("SDA");
            break;
    }
    put_lin();
    switch ((r0 >> 8 ) & 0x3) {
        case 0:
            PRINTF(iomsg4);
            break;
        case 1:
            PRINTF(iomsg1);
            break;
        case 2:
            PRINTF("Fast md");
            break;
        case 3:
            PRINTF(iomsg2);
            break;
    }
    put_rn();
}

void i2c_reg (void)
{
    uint32_t r0,r1,r2,r3,r4,r5,r6,r7;

    r0 = LPC_I2C->CONSET;
    r1 = LPC_I2C->STAT;
    r2 = LPC_I2C->DAT;
    r3 = LPC_I2C->SCLH;
    r4 = LPC_I2C->SCLL;
    r5 = LPC_I2C->CONCLR;
    r6 = LPC_I2C->MMCTRL;
    r7 = LPC_I2C->DATA_BUFFER;

    PRINTF("<Show I2C Registers>");
    put_rn();
    reg_print(SIZE8, r0);
    put_spc(2);
    PRINTF("CONSET");
    PRINTF(imsg2);
    put_rn();
    reg_print(SIZE8, r1);
    put_spc(2);
    PRINTF("STAT");
    PRINTF(imsg3);
    put_rn();
    reg_print(SIZE8, r2);
    put_spc(2);
    PRINTF("DAT");
    PRINTF(imsg4);
    put_rn();
    put_spc(2);
    PRINTF("ADR0--Not support");
    put_rn();
    reg_print(SIZE8, r3);
    put_spc(2);
    PRINTF("SCLH");
    PRINTF(imsg2);
    put_rn();
    reg_print(SIZE8, r4);
    put_spc(2);
    PRINTF("SCLL");
    PRINTF(imsg2);
    put_rn();
    reg_print(SIZE8, r5);
    put_spc(2);
    PRINTF("CONCLR");
    PRINTF(imsg2);
    put_rn();
    reg_print(SIZE8, r6);
    put_spc(2);
    PRINTF("MMCTRL");
    PRINTF(imsg2);
    put_rn();
    put_spc(2);
    PRINTF("ADR1,2,3--Not support");
    put_rn();
    reg_print(SIZE8, r7);
    put_spc(2);
    PRINTF("DATA_BUFFER");
    PRINTF(imsg5);
    put_rn();
    put_spc(2);
    PRINTF("MASK0,1,2,3--Not support");
    put_rn();
}

void i2c_freq (void)
{
    uint32_t r0,r1;

    r0 = LPC_I2C->SCLH;
    r1 = LPC_I2C->SCLL;
    get_freq(0);
    PRINTF("<I2C Status>");
    put_rn();
    PRINTF("Freq. = I2CPCLK/(SCLH+SCLL) = %d/(%d+%d) = %d Hz",
           SystemCoreClock, r0, r1, SystemCoreClock/(r0+r1));
    put_rn();
    r0 = LPC_I2C->CONSET;
    PRINTF("I2C I/F ");
    if ( r0 & 0x40 ) {
        PRINTF("Enabled");
    } else {
        PRINTF("disabled");
    }
    put_rn();
}
#endif

//-------------------------------------------------------------------------------------------------
//  Monitor Main Program
//-------------------------------------------------------------------------------------------------
int mon_hw (void)
{
    char *ptr;

    put_r();
    PRINTF("%s [Help:'?' key]", mon_msg);
    put_rn();
#if USE_SYS
    get_freq(0);
#endif      //USE_SYS
    for (;;) {
        put_r();
        PUTC('>');
        ptr = linebuf;
        get_line(ptr, buf_size);
        switch (*ptr++) {
                //---------------------------------------------------------------------------------
                //    Memory
                //---------------------------------------------------------------------------------
            case 'm' :
#if USE_MEM
                mem_inf(ptr);
                put_rn();
                break;
#else
                not_select();
#endif   // USE_MEM
                //---------------------------------------------------------------------------------
                //    Register
                //---------------------------------------------------------------------------------
            case 'r' :
                uint8_t  r_flg;
                put_r();
                PRINTF("Reg. Mode p,u,i,s,t,a,d,l,w,c & ?");
                put_rn();
                r_flg = 0;
                for (; r_flg != 0xff;) {
                    PRINTF("r>");
                    ptr = linebuf;
                    get_line(ptr, buf_size);
                    put_r();
                    switch(*ptr++) {
                        case 'p' :
#if USE_PORT
                            put_r();
                            switch(*ptr++) {
                                case '0' :
                                    io_config0();
                                    break;
                                case '1' :
                                    io_config1();
                                    break;
                                case '*' :
                                    io_config0();
                                    put_rn();
                                    io_config1();
                                    break;
                                case '?' :
                                default:
                                    PRINTF("Enter p0,p1 and p* for all");
                                    put_rn();
                            }
#else
                            not_select();
#endif // USE_PORT
                            break;
                        case 'u' :
#if USE_UART
                            uart_reg();
#else
                            not_select();
#endif // USE_UART
                            break;
                        case 'i' :
#if USE_I2C
                            put_r();
                            i2c_io_reg();
                            i2c_reg();
                            i2c_freq();
#else
                            not_select();
#endif // USE_I2C
                            break;
                        case 's' :
#if USE_SPI
                            switch(*ptr++) {
                                case '0' :
                                    spi_reg(SPI_0);
                                    break;
                                case '1' :
                                    spi_reg(SPI_1);
                                    break;
                                case '*' :
                                    spi_reg(SPI_0);
                                    put_rn();
                                    spi_reg( SPI_1 );
                                    break;
                                case '?' :
                                default:
                                    PRINTF("Enter s0,s1 and s* for all");
                                    put_rn();
                            }
#else
                            not_select();
#endif // USE_SPI
                            break;
                        case 't' :      //
                            not_yet_impliment();
                            break;
                        case 'a' :      //
                            not_yet_impliment();
                            break;
                        case 'd' :      //
                            not_yet_impliment();
                            break;
                        case 'w' :      //
                            not_yet_impliment();
                            break;
                        case 'l' :        //
                            not_yet_impliment();
                            break;
                        case 'c' :      //
                            not_yet_impliment();
                            break;
                        case 'x' :      //
                            not_yet_impliment();
                            break;
                        case 'y' :      //
                            not_yet_impliment();
                            break;
                        case 'q' :        // quit
                            r_flg = 0xff;
                            break;
                        case '?' :
                            PRINTF("p - I/O Pin Config");
                            put_rn();
                            PRINTF("u - UART");
                            put_rn();
                            PRINTF("i - I2C");
                            put_rn();
                            PRINTF("s - SPI");
                            put_rn();
                            PRINTF("t - TIMER");
                            put_rn();
                            PRINTF("a - ADC");
                            put_rn();
                            PRINTF("d - DAC");
                            put_rn();
                            PRINTF("l - LDC");
                            put_rn();
                            PRINTF("w - WWDG");
                            put_rn();
                            PRINTF("c - COMP");
                            put_rn();
                            PRINTF("q - Exit mode");
                            put_rn();
                            break;
                        default:
                            PUTC('?');
                            put_rn();
                    }
                }
                PRINTF("Return to All Mode");
                put_rn();
                break;
                //---------------------------------------------------------------------------------
                //    System
                //---------------------------------------------------------------------------------
            case 's' :    // System related information
#if USE_SYS
                switch (*ptr++) {
                    case 'f' :    // sc - show system clock frequency
                        get_freq(1);
                        break;
                    case 'c' :    // sc - show system CPU information
                        cpu_inf();
                        break;
                    case '?' :
                    default:
                        put_r();
                        PRINTF("sc - System CPU information");
                        put_rn();
                        PRINTF("sf - System Clock");
                        put_rn();
                        break;
                }
#else
                not_select();
#endif   // USE_SYS
                break;
                //---------------------------------------------------------------------------------
                //    Help
                //---------------------------------------------------------------------------------
            case '?' :
                put_r();
                msg_hlp_hw();
                break;
                //---------------------------------------------------------------------------------
                //    Return to main routine
                //---------------------------------------------------------------------------------
            case 'q' :        // Quit
                PRINTF("\rReturn to monitor ");
                return 0;
                //---------------------------------------------------------------------------------
                //    Special command for DEBUG
                //---------------------------------------------------------------------------------
            case 'x' :
                not_yet_impliment();
                break;
                //---------------------------------------------------------------------------------
                //    no support
                //---------------------------------------------------------------------------------
            default:
                PUTC('?');
                put_rn();
                break;
        }
    }
}

#endif  // defined(TARGET_LPC1114)
