/*
 * mbed Application program for the mbed LPC1768 Board
 * Monitor program Ver.3 for only for LPC1768
 *
 *	Copyright (c) 2010-2014 Kenji Arai / JH1PJL
 *	http://www.page.sannet.ne.jp/kenjia/index.html
 *	http://mbed.org/users/kenjiArai/
 *      Started: May 		9th, 2010
 *      Created: May       15th, 2010
 *			release as "monitor_01"	http://mbed.org/users/kenjiArai/code/monitor_01/
 *      restart: September 22nd, 2014
 *      Revised: Nobember   2nd, 2014
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#if defined(TARGET_LPC1768)

//  Include ---------------------------------------------------------------------------------------
#include "mbed.h"
#include "mon_hw_config.h"
#include "mon_hw_common.h"

//  Object ----------------------------------------------------------------------------------------

//  Definition ------------------------------------------------------------------------------------
// Define clocks
#define XTAL        		(12000000UL)        // Oscillator frequency
#define OSC_CLK     		(      XTAL)        // Main oscillator frequency
#define RTC_CLK     		(   32000UL)        // RTC oscillator frequency
#define IRC_OSC     		( 4000000UL)        // Internal RC oscillator frequency

//  RAM -------------------------------------------------------------------------------------------
uint32_t SystemFrequency;

//  ROM / Constant data ---------------------------------------------------------------------------
#if USE_MEM
//  Memory range data
const uint32_t mem_range[][2] = {	// Memory access range
    { 0x00000000, 0x0007ffff },     // On-chip non-volatile memory     // 512KB Flash memory
    { 0x10000000, 0x10007fff },     // On-chip SRAM                    // 32KB local RAM
    { 0x1fff0000, 0x1fff1fff },     // Boot ROM                        // 8KB Boot ROM
    { 0x2007c000, 0x2007ffff },     // On-chip SRAM                    // 16KB AHB SRAM
    { 0x20080000, 0x20083fff }      // On-chip SRAM                    // 16KB AHB SRAM
};
#endif // USE_MEM

char *const mon_msg = "HW monitor only for mbed LPC1768 created on "__DATE__","__TIME__"";
char *const xmsg0 = "Not implimented yet";

char *const hmsg0 = "m  - Entry Memory Mode";
char *const hmsg1 = "m>? -> Help ";
char *const hmsg2 = "r  - Show PORT,I2C,SPI,UART & other Reg.";
char *const hmsg3 = "r>? -> Help";
char *const hmsg4 = "sf - System Clock";
char *const hmsg5 = "sc - System / CPU information";
char *const hmsg6 = "x  - Special command for Debug";
char *const hmsg7 = "q  - Quit (back to called routine)";

//  Function prototypes ---------------------------------------------------------------------------

//-------------------------------------------------------------------------------------------------
//  Control Program
//-------------------------------------------------------------------------------------------------
// No function
static void not_yet_impliment( void )
{
    PRINTF(xmsg0);
    put_rn();
}

// No function
#if (USE_MEM==0)||(USE_PORT==0)||(USE_UART==0)||(USE_SPI==0)||(USE_I2C==0)||(USE_SYS==0)
static void not_select( void )
{
    PRINTF("Not select the function (refer mon_hw_config.h)");
    put_rn();
}
#endif

//  Help Massage
void msg_hlp_hw (void)
{
    PRINTF(mon_msg);
    put_rn();
    PRINTF(hmsg0);
    put_rn();
    PRINTF(hmsg1);
    put_rn();
    PRINTF(hmsg2);
    put_rn();
    PRINTF(hmsg3);
    put_rn();
    PRINTF(hmsg4);
    put_rn();
    PRINTF(hmsg5);
    put_rn();
    PRINTF(hmsg6);
    put_rn();
    PRINTF(hmsg7);
    put_rn();
}

#if USE_MEM
char *const rmsg0 = "FLASH     ";
char *const rmsg1 = "SRAM      ";
char *const rmsg2 = "Boot ROM  ";
char *const rmsg3 = "AHB SRAM0 ";
char *const rmsg4 = "AHB SRAM1 ";

#include "mon_hw_mem.h"
#endif   // USE_MEM

#if USE_PORT
void io_condition(uint32_t reg0, uint32_t reg1, uint8_t shift)
{
    PRINTF("IO->");
    if (reg0 & (1UL << shift)) {
        PRINTF("Output=");
    } else {
        PRINTF("Input =");
    }
    if (reg1 & (1UL << shift)) {
        PUTC('1');
    } else {
        PUTC('0');
    }
}

void port_config_left(void)
{
    uint32_t r0,r1;

    // p5(P0.9)
    PRINTF("p 5(P0. 9):");
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 18) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 9);
            break;
        case 1:
            PRINTF("I2STX_SDA");
            break;
        case 2:
            PRINTF("MOSI1");
            break;
        case 3:
            PRINTF("MAT2.3");
            break;
    }
    put_rn();
    // p6(P0.8)
    PRINTF("p 6(P0. 8):");
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 16) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 8);
            break;
        case 1:
            PRINTF("I2STX_WS");
            break;
        case 2:
            PRINTF("MISO1");
            break;
        case 3:
            PRINTF("MAT2.2");
            break;
    }
    put_rn();
    // p7(P0.7)
    PRINTF("p 7(P0. 7):");
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 14) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 7);
            break;
        case 1:
            PRINTF("I2STX_CLK");
            break;
        case 2:
            PRINTF("SCK1");
            break;
        case 3:
            PRINTF("MAT2.1");
            break;
    }
    put_rn();
    // p8(P0.6)
    PRINTF("p 8(P0. 6):");
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 12) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 6);
            break;
        case 1:
            PRINTF("I2SRX_SDA");
            break;
        case 2:
            PRINTF("SSEL1");
            break;
        case 3:
            PRINTF("MAT2.0");
            break;
    }
    put_rn();
    // p9(P0.0)
    PRINTF("p 9(P0. 0):");
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 0) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 0);
            break;
        case 1:
            PRINTF("RD1");
            break;
        case 2:
            PRINTF("TXD3");
            break;
        case 3:
            PRINTF("SDA1");
            break;
    }
    put_rn();
    // p10(P0.1)
    PRINTF("p10(P0. 1):");
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 2) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 1);
            break;
        case 1:
            PRINTF("TD1");
            break;
        case 2:
            PRINTF("RXD3");
            break;
        case 3:
            PRINTF("SCL1");
            break;
    }
    put_rn();
    // p11(P0.18)
    PRINTF("p11(P0.18):");
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 4) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 18);
            break;
        case 1:
            PRINTF("DCD1");
            break;
        case 2:
            PRINTF("MOSI0");
            break;
        case 3:
            PRINTF("MOSI");
            break;
    }
    put_rn();
    // p12(P0.17)
    PRINTF("p12(P0.17):");
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 2) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 17);
            break;
        case 1:
            PRINTF("CTS1");
            break;
        case 2:
            PRINTF("MISO0");
            break;
        case 3:
            PRINTF("MISO");
            break;
    }
    put_rn();
    // p13(P0.15)
    PRINTF("p13(P0.15):");
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 30) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 15);
            break;
        case 1:
            PRINTF("TXD1");
            break;
        case 2:
            PRINTF("SCK0");
            break;
        case 3:
            PRINTF("SCK");
            break;
    }
    put_rn();
    // p14(P0.16)
    PRINTF("p14(P0.16):");
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 0) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 16);
            break;
        case 1:
            PRINTF("RXD1");
            break;
        case 2:
            PRINTF("SSEL0");
            break;
        case 3:
            PRINTF("SSEL");
            break;
    }
    put_rn();
    // p15(P0.23)
    PRINTF("p15(P0.23):");
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 14) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 23);
            break;
        case 1:
            PRINTF("AD0.0");
            break;
        case 2:
            PRINTF("I2SRX_CLK");
            break;
        case 3:
            PRINTF("CAP3.0");
            break;
    }
    put_rn();
    // p16(P0.24)
    PRINTF("p16(P0.24):");
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 16) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 24);
            break;
        case 1:
            PRINTF("AD0.1");
            break;
        case 2:
            PRINTF("I2SRX_WS");
            break;
        case 3:
            PRINTF("CAP3.1");
            break;
    }
    put_rn();
    // p17(P0.25)
    PRINTF("p17(P0.25):");
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 18) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 25);
            break;
        case 1:
            PRINTF("AD0.2");
            break;
        case 2:
            PRINTF("I2SRX_SDA");
            break;
        case 3:
            PRINTF("TXD3");
            break;
    }
    put_rn();
    // p18(P0.26)
    PRINTF("p18(P0.26):");
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 20) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 26);
            break;
        case 1:
            PRINTF("AD0.3");
            break;
        case 2:
            PRINTF("AOUT");
            break;
        case 3:
            PRINTF("RXD3");
            break;
    }
    put_rn();
    // p19(P1.30)
    PRINTF("p19(P1.30):");
    r0 = LPC_PINCON->PINSEL3;
    r0 = (r0 >> 28) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO1->FIODIR;
            r1 = LPC_GPIO1->FIOPIN;
            io_condition(r0, r1, 30);
            break;
        case 1:
            PRINTF("Reserved");
            break;
        case 2:
            PRINTF("VBUS");
            break;
        case 3:
            PRINTF("AD0.4");
            break;
    }
    put_rn();
    // p20(P1.31)
    PRINTF("p20(P1.31):");
    r0 = LPC_PINCON->PINSEL3;
    r0 = (r0 >> 30) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO1->FIODIR;
            r1 = LPC_GPIO1->FIOPIN;
            io_condition(r0, r1, 31);
            break;
        case 1:
            PRINTF("Reserved");
            break;
        case 2:
            PRINTF("SCK1");
            break;
        case 3:
            PRINTF("AD0.5");
            break;
    }
    put_rn();
}

void port_config_right(void)
{
    uint32_t r0, r1;

    // p30(P0.4)
    PRINTF("p30(P0. 4):");
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 8) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 4);
            break;
        case 1:
            PRINTF("I2SRX_CLK");
            break;
        case 2:
            PRINTF("RD2");
            break;
        case 3:
            PRINTF("CAP2.0");
            break;
    }
    put_rn();
    // p29(P0.5)
    PRINTF("p29(P0. 5):");
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 10) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 5);
            break;
        case 1:
            PRINTF("I2SRX_WS");
            break;
        case 2:
            PRINTF("TD2");
            break;
        case 3:
            PRINTF("CAP2.1");
            break;
    }
    put_rn();
    // p28(P0.10)
    PRINTF("p28(P0.10):");
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 20) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 10);
            break;
        case 1:
            PRINTF("TXD2");
            break;
        case 2:
            PRINTF("SDA2");
            break;
        case 3:
            PRINTF("MAT3.0");
            break;
    }
    put_rn();
    // p27(P0.11)
    PRINTF("p27(P0.11):");
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 22) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO0->FIODIR;
            r1 = LPC_GPIO0->FIOPIN;
            io_condition(r0, r1, 11);
            break;
        case 1:
            PRINTF("RXD2");
            break;
        case 2:
            PRINTF("SCL2");
            break;
        case 3:
            PRINTF("MAT3.1");
            break;
    }
    put_rn();
    // p26(P2.0)
    PRINTF("p26(P2. 0):");
    r0 = LPC_PINCON->PINSEL4;
    r0 = (r0 >> 0) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO2->FIODIR;
            r1 = LPC_GPIO2->FIOPIN;
            io_condition(r0, r1, 0);
            break;
        case 1:
            PRINTF("PWM1.1");
            break;
        case 2:
            PRINTF("TXD1");
            break;
        case 3:
            PRINTF("Reserved");
            break;
    }
    put_rn();
    // p25(P2.1)
    PRINTF("p25(P2. 1):");
    r0 = LPC_PINCON->PINSEL4;
    r0 = (r0 >> 2) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO2->FIODIR;
            r1 = LPC_GPIO2->FIOPIN;
            io_condition(r0, r1, 1);
            break;
        case 1:
            PRINTF("PWM1.2");
            break;
        case 2:
            PRINTF("RXD1");
            break;
        case 3:
            PRINTF("Reserved");
            break;
    }
    put_rn();
    // p24(P2.2)
    PRINTF("p24(P2. 2):");
    r0 = LPC_PINCON->PINSEL4;
    r0 = (r0 >> 4) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO2->FIODIR;
            r1 = LPC_GPIO2->FIOPIN;
            io_condition(r0, r1, 2);
            break;
        case 1:
            PRINTF("PWM1.3");
            break;
        case 2:
            PRINTF("CTS1");
            break;
        case 3:
            PRINTF("Reserved");
            break;
    }
    put_rn();
    // p23(P2.3)
    PRINTF("p23(P2. 3):");
    r0 = LPC_PINCON->PINSEL4;
    r0 = (r0 >> 6) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO2->FIODIR;
            r1 = LPC_GPIO2->FIOPIN;
            io_condition(r0, r1, 3);
            break;
        case 1:
            PRINTF("PWM1.4");
            break;
        case 2:
            PRINTF("DCD1");
            break;
        case 3:
            PRINTF("Reserved");
            break;
    }
    put_rn();
    // p22(P2.4)
    PRINTF("p22(P2. 4):");
    r0 = LPC_PINCON->PINSEL4;
    r0 = (r0 >> 8) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO2->FIODIR;
            r1 = LPC_GPIO2->FIOPIN;
            io_condition(r0, r1, 4);
            break;
        case 1:
            PRINTF("PWM1.5");
            break;
        case 2:
            PRINTF("DSR1");
            break;
        case 3:
            PRINTF("Reserved");
            break;
    }
    put_rn();
    // p21(P2.5)
    PRINTF("p21(P2. 5):");
    r0 = LPC_PINCON->PINSEL4;
    r0 = (r0 >> 10) & 0x03;
    switch (r0) {
        case 0:
            r0 = LPC_GPIO2->FIODIR;
            r1 = LPC_GPIO2->FIOPIN;
            io_condition(r0, r1, 5);
            break;
        case 1:
            PRINTF("PWM1.6");
            break;
        case 2:
            PRINTF("DTR1");
            break;
        case 3:
            PRINTF("Reserved");
            break;
    }
    put_rn();
}
#endif	// USE_PORT

//#if USE_SYS
void cpu_inf ( void )
{
    unsigned long m1, m2;

    m1 = SCB->CPUID;
    m2 = ( m1 >> 24 );
    if ( m2 == 0x41 ) {
        put_r();
        PRINTF("CPU = ARM ");
    } else {
        put_r();
        PRINTF("CPU = NOT ARM ");
    }
    m2 = ( m1 >> 4 ) & 0xfff;
    if ( m2 == 0xc23 ) {
        PRINTF("Cortex-M3");
        put_rn();
    } else {
        PRINTF("NOT Cortex-M3");
        put_rn();
    }
    m2 = ( m1 >> 20 ) & 0x0f;
    PRINTF("Variant:%x", m2);
    put_rn();
    m2 = m1 & 0x7;
    PRINTF("Revision:%x", m2);
    put_rn();
}
//#endif	// USE_SYS

//  Calculate CPU System Clock Frequency
void get_freq ( int pr )
{
    if(pr) {
        put_r();
        PRINTF("System Clock = %dHz",SystemFrequency );
        put_rn();
    }
    if (((LPC_SC->PLL0STAT >> 24) & 3) == 3) {/* If PLL0 enabled and connected      */
        if(pr) {
            PRINTF("PLL0 enabled");
            put_rn();
        }
        switch (LPC_SC->CLKSRCSEL & 0x03) {
            case 0:                /* Internal RC oscillator => PLL0     */
            case 3:                /* Reserved, default to Internal RC   */
                if(pr) {
                    PRINTF("Internal RC Oscillator");
                    put_rn();
                }
                SystemFrequency = (IRC_OSC *
                                   (((2 * ((LPC_SC->PLL0STAT & 0x7FFF) + 1))) /
                                    (((LPC_SC->PLL0STAT >> 16) & 0xFF) + 1))   /
                                   ((LPC_SC->CCLKCFG & 0xFF)+ 1));
                break;
            case 1:                /* Main oscillator => PLL0            */
                if(pr) {
                    PRINTF("Xtal Osc Clock = %dHz",XTAL );
                    put_rn();
                }
                SystemFrequency = (OSC_CLK *
                                   (((2 * ((LPC_SC->PLL0STAT & 0x7FFF) + 1))) /
                                    (((LPC_SC->PLL0STAT >> 16) & 0xFF) + 1))   /
                                   ((LPC_SC->CCLKCFG & 0xFF)+ 1));
                break;
            case 2:                /* RTC oscillator => PLL0             */
                if(pr) {
                    PRINTF("RTC Xtal Oscillator f = %dHz", RTC_CLK );
                    put_rn();
                }
                SystemFrequency = (RTC_CLK *
                                   (((2 * ((LPC_SC->PLL0STAT & 0x7FFF) + 1))) /
                                    (((LPC_SC->PLL0STAT >> 16) & 0xFF) + 1))   /
                                   ((LPC_SC->CCLKCFG & 0xFF)+ 1));
                break;
        }
    } else {
        if(pr) {
            PRINTF("PLL0 disabled");
        }
        switch (LPC_SC->CLKSRCSEL & 0x03) {
            case 0:                /* Internal RC oscillator => PLL0     */
            case 3:                /* Reserved, default to Internal RC   */
                if(pr) {
                    PRINTF("Internal RC Oscillator");
                    put_rn();
                }
                SystemFrequency = IRC_OSC / ((LPC_SC->CCLKCFG & 0xFF)+ 1);
                break;
            case 1:                /* Main oscillator => PLL0            */
                if(pr) {
                    PRINTF("Xtal Osc Clock = %dHz",XTAL );
                    put_rn();
                }
                SystemFrequency = OSC_CLK / ((LPC_SC->CCLKCFG & 0xFF)+ 1);
                break;
            case 2:                /* RTC oscillator => PLL0             */
                if(pr) {
                    PRINTF("RTC Xtal Oscillator f = %dHz", RTC_CLK );
                    put_rn();
                }
                SystemFrequency = RTC_CLK / ((LPC_SC->CCLKCFG & 0xFF)+ 1);
                break;
        }
    }
}

#if (USE_UART==1)||(USE_SPI==1)||(USE_I2C==1)
char *const uismsg0 = "is enable";
char *const uismsg1 = "is disable";
char *const uismsg2 = "(mbed pins are not avairable)";
char *const uismsg3 = "Other";
#endif

#if USE_UART
//  Show 16bit register contents
void reg_print(uint16_t size, uint16_t reg)
{
    uint16_t i, j, k, n;

    if (size == 8) {
        PRINTF(" 7, 6, 5, 4, 3, 2, 1, 0");
        put_rn();
        i = 8;
        n = 0x80;
    } else if (size == 16) {
        PRINTF( "15,14,13,12,11,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0" );
        put_rn();
        i = 16;
        n = 0x8000;
    } else {
        PRINTF("0x%08x", reg);
        return;
    }
    PUTC(' ');
    for (; i>0; i--) {
        k = n >> (size-i);
        j = reg & k;
        if (j) {
            PUTC('1');
        } else {
            PUTC('0');
        }
        PUTC(' ');
        PUTC(' ');
    }
    PRINTF("  (0x%04x)", reg);
}

void uart_reg(uint8_t uart)
{
    uint32_t c_lcr,c_fdr,c_dll,c_dlm;
    uint32_t divaddval,mulval,div,baud;

    PRINTF("<Show UART%d status>", uart);
    put_rn();
    c_lcr = c_fdr = c_dll = c_dlm = 0;
    divaddval = mulval = div = baud = 0;
    get_freq(0);
    switch (uart) {
        case 0:
            if (LPC_SC->PCONP & (1UL<<3)) {
                c_fdr = LPC_UART0->FDR;
                c_lcr = LPC_UART0->LCR;
                // set LCR[DLAB] to enable writing to divider registers
                LPC_UART0->LCR |= (1 << 7);
                c_dll = LPC_UART0->DLL;
                c_dlm = LPC_UART0->DLM;
                // clear LCR[DLAB]
                LPC_UART0->LCR &= ~(1 << 7);
                div = (LPC_SC->PCLKSEL0 >> 6) & 0x3;
            }
            break;
        case 1:
            if (LPC_SC->PCONP & (1UL<<4)) {
                c_fdr = LPC_UART1->FDR;
                c_lcr = LPC_UART1->LCR;
                // set LCR[DLAB] to enable writing to divider registers
                LPC_UART1->LCR |= (1 << 7);
                c_dll = LPC_UART1->DLL;
                c_dlm = LPC_UART1->DLM;
                // clear LCR[DLAB]
                LPC_UART1->LCR &= ~(1 << 7);
                div = (LPC_SC->PCLKSEL0 >> 8) & 0x3;
            }
            break;
        case 2:
            if (LPC_SC->PCONP & (1UL<<24)) {
                c_fdr = LPC_UART2->FDR;
                c_lcr = LPC_UART2->LCR;
                // set LCR[DLAB] to enable writing to divider registers
                LPC_UART2->LCR |= (1 << 7);
                c_dll = LPC_UART2->DLL;
                c_dlm = LPC_UART2->DLM;
                // clear LCR[DLAB]
                LPC_UART2->LCR &= ~(1 << 7);
                div = (LPC_SC->PCLKSEL1 >> 16) & 0x3;
            }
            break;
        case 3:
            if (LPC_SC->PCONP & (1UL<<25)) {
                c_fdr = LPC_UART3->FDR;
                c_lcr = LPC_UART3->LCR;
                // set LCR[DLAB] to enable writing to divider registers
                LPC_UART3->LCR |= (1 << 7);
                c_dll = LPC_UART3->DLL;
                c_dlm = LPC_UART3->DLM;
                // clear LCR[DLAB]
                LPC_UART3->LCR &= ~(1 << 7);
                div = (LPC_SC->PCLKSEL1 >> 18) & 0x3;
            }
            break;
        default:
            break;
    }
    if ((c_fdr == 0)&&(c_lcr == 0)&&(c_dll == 0)) {
        PRINTF("NOT Avairable (power off the UART module)");
    } else {
        // condition
        PRINTF("Word Length: ");
        switch ((c_lcr >> 0) & 0x03) {
            case 0:
                PUTC('5');
                break;
            case 1:
                PUTC('6');
                break;
            case 2:
                PUTC('7');
                break;
            case 3:
                PUTC('8');
                break;
        }
        PRINTF("-bit");
        put_rn();
        PRINTF("Stop bit: ");
        if (c_lcr & 0x04) {
            PUTC('1');
        } else {
            PUTC('2');
        }
        PRINTF("-bit");
        put_rn();
        PRINTF("Parity: ");
        if (c_lcr & 0x08) {
            switch ((c_lcr >> 4) & 0x03) {
                case 0:
                    PRINTF("Odd");
                    break;
                case 1:
                    PRINTF("Even");
                    break;
                case 2:
                    PRINTF("Forced '1'");
                    break;
                case 3:
                    PRINTF("Forced '0'");
                    break;
            }
            PRINTF(" parity");
        } else {
            PRINTF("None-Parity");
        }
        put_rn();
        switch (div) {
            case 0:
                div = 4;
                break;
            case 1:
                div = 1;
                break;
            case 2:
                div = 2;
                break;
            case 3:
                div = 8;
                break;
        }
        divaddval = (c_fdr >> 0) & 0x0f;
        mulval = (c_fdr >> 4) & 0x0f;
        baud =  (SystemCoreClock/div)/(16 * (256 * c_dlm + c_dll) * (1 + divaddval/mulval));
        PRINTF("Baud rate: %d bps", baud);
        put_rn();
        PRINTF(" = PCLK(=Sys/div)/(16 x (256 x DLM + DLL) x (1 + DivAddVal/MulVal)");
        put_rn();
        PRINTF(" = (%d/%d)/(16 x (256 x %d + %d) x (1 + %d/%d)",
               SystemCoreClock,div,c_dlm,c_dll,divaddval,mulval);
    }
    put_rn();
}

void uart_io_print (void)
{
    PRINTF("<Show IO Pin>");
    put_rn();
}

void uart_io_reg0 (void)
{
    uint32_t r0;

    PRINTF("UART0/USBTX(P0.2),USBRX(P0.3):");
    // P0.2
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 4) & 0x03;
    if (r0 == 1) {
        PRINTF("TXD0 ");
    } else {
        PRINTF(uismsg3);
    }
    // P0.3
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 6) & 0x03;
    if (r0 == 1) {
        PRINTF(",RXD0");
    } else {
        PRINTF(",%s", uismsg3);
    }
    PRINTF("(connected to PC via USB)");
    put_rn();
}

void uart_io_reg1 (void)
{
    uint32_t r0;

    PRINTF("UART1/ p13(P0.15), p14(P0.16):");
    // P0.15
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 30) & 0x03;
    if (r0 == 1) {
        PRINTF("TXD1 ");
    } else {
        PRINTF(uismsg3);
    }
    // P0.16
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 2) & 0x03;
    if (r0 == 1) {
        PRINTF(",RXD1");
    } else {
        PRINTF(",%s", uismsg3);
    }
    put_rn();
}

void uart_io_reg2 (void)
{
    uint32_t r0;

    PRINTF("UART2/ p28(P0.10), p27(P0.11):");
    // P0.10
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 20) & 0x03;
    if (r0 == 1) {
        PRINTF("TXD2 ");
    } else {
        PRINTF(uismsg3);
    }
    // P0.11
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 22) & 0x03;
    if (r0 == 1) {
        PRINTF(",RXD2");
    } else {
        PRINTF(",%s", uismsg3);
    }
    put_rn();
}

void uart_io_reg3 (void)
{
    uint32_t r0;

    PRINTF("UART3/ p 9(P0. 0), p10(P0. 1):");
    // P0.0
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 0) & 0x03;
    if (r0 == 2) {
        PRINTF("TXD3 ");
    } else {
        PRINTF(uismsg3);
    }
    // P0.1
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 2) & 0x03;
    if (r0 == 2) {
        PRINTF(",RXD3");
    } else {
        PRINTF(",%s", uismsg3);
    }
    put_rn();
}
#endif  //USE_UART

#if USE_SPI
void spi_io_reg (void)
{
    uint32_t r0;

    PRINTF("<Show IO Pin>");
    put_rn();
    //-----------------------------------------------------
    PRINTF("SPI /p11(P0.18),p12(P0.17),p13(P0.15):");
    // p11(P0.18)
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 4) & 0x03;
    if (r0 == 3) {
        PRINTF(",MOSI ");
    } else {
        PRINTF("%s", uismsg3);
    }
    // p12(P0.17)
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 2) & 0x03;
    if (r0 == 3) {
        PRINTF(",MISO ");
    } else {
        PRINTF(",%s", uismsg3);
    }
    // p13(P0.15)
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 30) & 0x03;
    if (r0 == 3) {
        PRINTF(",SCK ");
    } else {
        PRINTF(",%s", uismsg3);
    }
    put_rn();
    //-----------------------------------------------------
    PRINTF("SSP0/p11(P0.18),p12(P0.17),p13(P0.15):");
    // p11(P0.18)
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 4) & 0x03;
    if (r0 == 2) {
        PRINTF(",MOSI0");
    } else {
        PRINTF("%s", uismsg3);
    }
    // p12(P0.17)
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 2) & 0x03;
    if (r0 == 2) {
        PRINTF(",MISO0");
    } else {
        PRINTF(",%s", uismsg3);
    }
    // p13(P0.15)
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 30) & 0x03;
    if (r0 == 2) {
        PRINTF(",SCK0 ");
    } else {
        PRINTF(",%s", uismsg3);
    }
    put_rn();
    //-----------------------------------------------------
    PRINTF("SSP1/p 5(P0. 9),p 6(P0. 8),p 7(P0. 7):");
    // p5(P0.9)
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 18) & 0x03;
    if (r0 == 2) {
        PRINTF("MOSI1");
    } else {
        PRINTF(uismsg3);
    }
    // p6(P0.8)
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 16) & 0x03;
    if (r0 == 2) {
        PRINTF(",MISO1");
    } else {
        PRINTF(",%s", uismsg3);
    }
    // p7(P0.7)
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 14) & 0x03;
    if (r0 == 2) {
        PRINTF(",SCK1");
    } else {
        PRINTF(",%s", uismsg3);
    }
    put_rn();
}

void spi_freq (void)
{
    uint32_t div, r0, r1;

    get_freq(0);
    //-----------------------------------------------------
    // SPI
    PRINTF("<Show SPI status>");
    put_rn();
    if (LPC_SC->PCONP & (1UL<<8)) {
        div = (LPC_SC->PCLKSEL0 >> 16) & 0x3;
        switch (div) {
            case 0:
                div = 4;
                break;
            case 1:
                div = 1;
                break;
            case 2:
                div = 2;
                break;
            case 3:
                div = 8;
                break;
        }
        r0 = LPC_SPI->SPCR;
        PRINTF("Data length: ");
        if (r0 & 0x04) {
            r1 = (r0 >> 8) & 0x0f;
            if (r1 == 0) {
                r1 = 16;
            }
            PRINTF("%d", r1);
        } else {
            PUTC('8');
        }
        PRINTF("bit,  ");
        PRINTF("CPHA: ");
        PRINTF("%d", (r0 >> 3) & 0x01);
        PRINTF(",  CPOL: ");
        PRINTF("%d", (r0 >> 4) & 0x01);
        put_rn();
        if ((r0 >> 5) & 0x01) {
            PRINTF("Master");
        } else {
            PRINTF("Slave");
        }
        PRINTF(" Mode");
        put_rn();
        if ((r0 >> 6) & 0x01) {
            PRINTF("MSB(bit7)");
        } else {
            PRINTF("LSB(Bit0)");
        }
        PRINTF(" first");
        put_rn();
        r1 = LPC_SPI->SPCCR;
        PRINTF("CLK = %d Hz", SystemCoreClock/div/r1);
        put_rn();
        PRINTF("    = PCLK_SPI(=Sys/div)/SPCCR0");
        put_rn();
        PRINTF("    = (%d/%d)/%d", SystemCoreClock, div, r1, SystemCoreClock/div/r1);
    } else {
        PRINTF("NOT Avairable (power off the module)");
    }
    put_rn();
    //-----------------------------------------------------
    // SSP0
    PRINTF("<Show SSP0 status>");
    put_rn();
    if (LPC_SC->PCONP & (1UL<<21)) {
        r0 = LPC_SSP0->CR1;
        if (r0 & 0x02) {
            div = (LPC_SC->PCLKSEL1 >> 10) & 0x3;
            switch (div) {
                case 0:
                    div = 4;
                    break;
                case 1:
                    div = 1;
                    break;
                case 2:
                    div = 2;
                    break;
                case 3:
                    div = 8;
                    break;
            }
            r1 = LPC_SSP0->CR0;
            PRINTF("SSP Mode: ");
            r1 = (r1 >> 4) & 0x03;
            switch (r1) {
                case 0:
                    PRINTF("SPI");
                    break;
                case 1:
                    PRINTF("TI");
                    break;
                case 2:
                    PRINTF("Microwire");
                    break;
                case 3:
                    PRINTF("Not support");
                    break;
            }
            put_rn();
            r1 = LPC_SSP0->CR0;
            PRINTF("Data length: ");
            PRINTF("%d", r1 & 0x0f);
            PRINTF("bit,  ");
            PRINTF("CPHA: ");
            PRINTF("%d", (r1 >> 7) & 0x01);
            PRINTF(",  CPOL: ");
            PRINTF("%d", (r1 >> 6) & 0x01);
            put_rn();
            if ((r0 >> 2) & 0x01) {
                PRINTF("Slave");
            } else {
                PRINTF("Master");
            }
            PRINTF(" Mode");
            put_rn();
            r1 = LPC_SSP0->CR0;
            r1 = (r1 >> 8) & 0xff;
            r0 = LPC_SSP0->CPSR;
            r0 = (r0 >> 0) & 0x0f;
            PRINTF("CLK = %d Hz", SystemCoreClock/div/(r0 * (r1+1)));
            put_rn();
            PRINTF("    = PCLK_SSP0(=Sys/div)/(CPSDVSR x [SCR+1])");
            put_rn();
            PRINTF("    = (%d/%d)/(%d x [%d + 1])", SystemCoreClock, div, r0, r1);
        } else {
            PRINTF("SSP0 is disabled");
        }
    } else {
        PRINTF("NOT Avairable (power off the module)");
    }
    put_rn();
    //-----------------------------------------------------
    // SSP1
    PRINTF("<Show SSP1 status>");
    put_rn();
    if (LPC_SC->PCONP & (1UL<<10)) {
        r0 = LPC_SSP1->CR1;
        if (r0 & 0x02) {
            div = (LPC_SC->PCLKSEL0 >> 20) & 0x3;
            switch (div) {
                case 0:
                    div = 4;
                    break;
                case 1:
                    div = 1;
                    break;
                case 2:
                    div = 2;
                    break;
                case 3:
                    div = 8;
                    break;
            }
            r1 = LPC_SSP1->CR0;
            PRINTF("SSP Mode: ");
            r1 = (r1 >> 4) & 0x03;
            switch (r1) {
                case 0:
                    PRINTF("SPI");
                    break;
                case 1:
                    PRINTF("TI");
                    break;
                case 2:
                    PRINTF("Microwire");
                    break;
                case 3:
                    PRINTF("Not support");
                    break;
            }
            put_rn();
            r1 = LPC_SSP1->CR0;
            PRINTF("Data length: ");
            PRINTF("%d", r1 & 0x0f);
            PRINTF("bit,  ");
            PRINTF("CPHA: ");
            PRINTF("%d", (r1 >> 7) & 0x01);
            PRINTF(",  CPOL: ");
            PRINTF("%d", (r1 >> 6) & 0x01);
            put_rn();
            if ((r0 >> 2) & 0x01) {
                PRINTF("Slave");
            } else {
                PRINTF("Master");
            }
            PRINTF(" Mode");
            put_rn();
            r1 = LPC_SSP1->CR0;
            r1 = (r1 >> 8) & 0xff;
            r0 = LPC_SSP1->CPSR;
            r0 = (r0 >> 0) & 0x0f;
            PRINTF("CLK = %d Hz", SystemCoreClock/div/(r0 * (r1+1)));
            put_rn();
            PRINTF("    = PCLK_SSP1(=Sys/div)/(CPSDVSR x [SCR+1])");
            put_rn();
            PRINTF("    = (%d/%d)/(%d x [%d + 1])", SystemCoreClock, div, r0, r1);
        } else {
            PRINTF("SSP1 is disabled");
        }
    } else {
        PRINTF("NOT Avairable (power off the module)");
    }
    put_rn();
}
#endif

#if USE_I2C
void i2c_io_reg (void)
{
    uint32_t r0;

    PRINTF("<Show IO Pin>");
    put_rn();
    //-----------------------------------------------------
    PRINTF("I2C0/pxx(P0.27),pxx(P0.28):");
    // P0.27
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 22) & 0x03;
    if (r0 == 3) {
        PRINTF("SDA0");
    } else {
        PRINTF(uismsg3);
    }
    // P0.28
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 24) & 0x03;
    if (r0 == 3) {
        PRINTF(",SCL0");
    } else {
        PRINTF(",%s", uismsg3);
    }
    PRINTF(uismsg2);
    put_rn();
    //-----------------------------------------------------
    PRINTF("I2C1/p 9(P0. 0),p10(P0. 1):");
    // P0.0
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 0) & 0x03;
    if (r0 == 3) {
        PRINTF("SDA1");
    } else {
        PRINTF(uismsg3);
    }
    // P0.1
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 2) & 0x03;
    if (r0 == 3) {
        PRINTF(",SCL1");
    } else {
        PRINTF(",%s", uismsg3);
    }
    put_rn();
    //-----------------------------------------------------
    PRINTF("I2C1/pxx(P0.19),pxx(P0.20):");
    // P0.19
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 6) & 0x03;
    if (r0 == 3) {
        PRINTF("SDA1");
    } else {
        PRINTF(uismsg3);
    }
    // P0.20
    r0 = LPC_PINCON->PINSEL1;
    r0 = (r0 >> 8) & 0x03;
    if (r0 == 3) {
        PRINTF(",SCL1");
    } else {
        PRINTF(",%s", uismsg3);
    }
    PRINTF(uismsg2);
    put_rn();
    //-----------------------------------------------------
    PRINTF("I2C2/p28(P0.10),p27(P0.11):");
    // P0.10
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 20) & 0x03;
    if (r0 == 3) {
        PRINTF("SDA2");
    } else {
        PRINTF(uismsg3);
    }
    // P0.11
    r0 = LPC_PINCON->PINSEL0;
    r0 = (r0 >> 22) & 0x03;
    if (r0 == 3) {
        PRINTF(",SCL2");
    } else {
        PRINTF(",%s", uismsg3);
    }
    put_rn();
}

char *const msg_cal = "    = I2CPCLK(=Sys/div)/(SCLH+SCLL)";

void i2c_freq (void)
{
    uint32_t r0, r1, r2, r3;

    get_freq(0);
    // I2C0
    r0 = LPC_I2C0->I2SCLL;
    r1 = LPC_I2C0->I2SCLH;
    r2 = LPC_I2C0->I2CONSET;
    r3 = (LPC_SC->PCLKSEL0 >> 14) & 0x3;
    switch (r3) {
        case 0:
            r3 = 4;
            break;
        case 1:
            r3 = 1;
            break;
        case 2:
            r3 = 2;
            break;
        case 3:
            r3 = 8;
            break;
    }
    PRINTF("<I2C Status>");
    put_rn();
    //-----------------------------------------------------
    PRINTF("I2C0 ");
    if (r2 & 0x40) {
        PRINTF(uismsg0);
        put_rn();
        PRINTF("CLK = %d Hz", SystemCoreClock/r3/(r0+r1));
        put_rn();
        PRINTF(msg_cal);
        put_rn();
        PRINTF("    = (%d/%d)/(%d+%d)", SystemCoreClock,r3, r0, r1);
    } else {
        PRINTF(uismsg1);
    }
    put_rn();
    //-----------------------------------------------------
    // I2C1
    r0 = LPC_I2C1->I2SCLL;
    r1 = LPC_I2C1->I2SCLH;
    r2 = LPC_I2C1->I2CONSET;
    r3 = (LPC_SC->PCLKSEL1 >> 6) & 0x3;
    switch (r3) {
        case 0:
            r3 = 4;
            break;
        case 1:
            r3 = 1;
            break;
        case 2:
            r3 = 2;
            break;
        case 3:
            r3 = 8;
            break;
    }
    PRINTF("I2C1 ");
    if (r2 & 0x40) {
        PRINTF(uismsg0);
        put_rn();
        PRINTF("CLK = %d Hz", SystemCoreClock/r3/(r0+r1));
        put_rn();
        PRINTF(msg_cal);
        put_rn();
        PRINTF("    = (%d/%d)/(%d+%d)", SystemCoreClock,r3, r0, r1);
    } else {
        PRINTF(uismsg1);
    }
    put_rn();
    //-----------------------------------------------------
    // I2C2
    r0 = LPC_I2C2->I2SCLL;
    r1 = LPC_I2C2->I2SCLH;
    r2 = LPC_I2C2->I2CONSET;
    r3 = (LPC_SC->PCLKSEL1 >> 20) & 0x3;
    switch (r3) {
        case 0:
            r3 = 4;
            break;
        case 1:
            r3 = 1;
            break;
        case 2:
            r3 = 2;
            break;
        case 3:
            r3 = 8;
            break;
    }
    PRINTF("I2C2 ");
    if (r2 & 0x40) {
        PRINTF(uismsg0);
        put_rn();
        PRINTF("CLK = %d Hz", SystemCoreClock/r3/(r0+r1));
        put_rn();
        PRINTF(msg_cal);
        put_rn();
        PRINTF("    = (%d/%d)/(%d+%d)", SystemCoreClock,r3, r0, r1);
    } else {
        PRINTF(uismsg1);
    }
    put_rn();
}
#endif

//-----------------------------------------------------------------------------
//  Monitor Main Program
//-----------------------------------------------------------------------------
int mon_hw(void)
{
    char *ptr;

    put_r();
    PRINTF("%s [Help:'?' key]", mon_msg);
    put_rn();
    get_freq(0);
    for (;;) {
        put_r();
        PUTC('>');
        ptr = linebuf;
        get_line(ptr, buf_size);
        put_r();
        switch (*ptr++) {
                //---------------------------------------------------------------------------------
                //    Memory
                //---------------------------------------------------------------------------------
            case 'm' :
#if USE_MEM
                mem_inf(ptr);
                put_rn();
#else
                not_select();
#endif   // USE_MEM
                break;
                //--------------------------------------------------------------------------------------
                //    Register
                //--------------------------------------------------------------------------------------
            case 'r' :
                uint8_t  r_flg;
                put_r();
                PRINTF("Reg. Mode p,u,i,s,t,a,d,l,w,c & ?");
                put_rn();
                r_flg = 0;
                for (; r_flg != 0xff;) {
                    PRINTF("r>");
                    ptr = linebuf;
                    get_line(ptr, buf_size);
                    put_r();
                    switch(*ptr++) {
                        case 'p' :
#if USE_PORT
                            put_r();
                            switch(*ptr++) {
                                case 'l' :
                                    port_config_left();
                                    break;
                                case 'r' :
                                    port_config_right();
                                    break;
                                case '*' :
                                    port_config_left();
                                    port_config_right();
                                    break;
                                case '?' :
                                    PRINTF("Enter pl,pr and p* for all");
                                    put_rn();
                                    break;;
                                default:
                                    PUTC('?');
                                    put_rn();
                                    break;
                            }
#else
                            not_select();
#endif // USE_PORT
                            break;
                        case 'u' :
#if USE_UART
                            put_r();
                            switch(*ptr++) {
                                case '0' :
                                    uart_io_print();
                                    uart_io_reg0();
                                    uart_reg(0);
                                    break;
                                case '1' :
                                    uart_io_print();
                                    uart_io_reg1();
                                    uart_reg(1);
                                    break;
                                case '2' :
                                    uart_io_print();
                                    uart_io_reg2();
                                    uart_reg(2);
                                    break;
                                case '3' :
                                    uart_io_print();
                                    uart_io_reg3();
                                    uart_reg(3);
                                    break;
                                case '*' :
                                    uart_io_print();
                                    uart_io_reg0();
                                    uart_io_reg1();
                                    uart_io_reg2();
                                    uart_io_reg3();
                                    break;
                                case '?' :
                                    PRINTF("Enter u0,u1,u2,u3 for each UART ");
                                    PRINTF("or u* for UART port assignment");
                                    put_rn();
                                    break;
                                default:
                                    PUTC('?');
                                    put_rn();
                                    break;
                            }
#else
                            not_select();
#endif // USE_UART
                            break;
                        case 'i' :
#if USE_I2C
                            put_r();
                            i2c_io_reg();
                            i2c_freq();
#else
                            not_select();
#endif // USE_I2C
                            break;
                        case 's' :
#if USE_SPI
                            put_r();
                            spi_io_reg();
                            spi_freq();
#else
                            not_select();
#endif // USE_SPI
                            break;
                        case 't' :		//
                            not_yet_impliment();
                            break;
                        case 'a' :		//
                            not_yet_impliment();
                            break;
                        case 'd' :		//
                            not_yet_impliment();
                            break;
                        case 'w' :		//
                            not_yet_impliment();
                            break;
                        case 'l' :        //
                            not_yet_impliment();
                            break;
                        case 'c' :		//
                            not_yet_impliment();
                            break;
                        case 'x' :		//
                            not_yet_impliment();
                            break;
                        case 'y' :		//
                            not_yet_impliment();
                            break;
                        case 'q' :        // quit
                            r_flg = 0xff;
                            break;
                        case '?' :
                            PRINTF("p - I/O Pin Config. pl(p5 to p20) & pr(p21 to p30)");
                            put_rn();
                            PRINTF("u - UART u0,1,2,3 or u* (port assignment)");
                            put_rn();
                            PRINTF("i - I2C");
                            put_rn();
                            PRINTF("s - SPI/SSP");
                            put_rn();
                            PRINTF("t - TIMER");
                            put_rn();
                            PRINTF("a - ADC");
                            put_rn();
                            PRINTF("d - DAC");
                            put_rn();
                            PRINTF("l - LDC");
                            put_rn();
                            PRINTF("w - WWDG");
                            put_rn();
                            PRINTF("c - COMP");
                            put_rn();
                            PRINTF("q - Exit mode");
                            put_rn();
                            break;
                        default:
                            PUTC('?');
                            put_rn();
                    }
                }
                PRINTF("Return to All Mode");
                put_rn();
                break;
                //---------------------------------------------------------------------------------
                //    System
                //---------------------------------------------------------------------------------
            case 's' :    // System related information
#if USE_SYS
                switch (*ptr++) {
                    case 'f' :    // sc - show system clock frequency
                        get_freq(1);
                        break;
                    case 'c' :    // sc - show system CPU information
                        cpu_inf();
                        break;
                    case '?' :
                    default:
                        put_r();
                        PRINTF("sc - System CPU information");
                        put_rn();
                        PRINTF("sf - System Clock");
                        put_rn();
                        break;
                }
#else
                not_select();
#endif   // USE_SYS
                break;
                //-----------------------------------------------------------------------------------------
                //    Help
                //-----------------------------------------------------------------------------------------
            case '?' :
                msg_hlp_hw();
                break;
                //-----------------------------------------------------------------------------------------
                //    Return to main routine
                //-----------------------------------------------------------------------------------------
            case 'q' :        // Quit
                put_r();
                PRINTF("Return to monitor");
                put_rn();
                return 0;
                //-----------------------------------------------------------------------------------------
                //    Special command for DEBUG
                //-----------------------------------------------------------------------------------------
            case 'x' :
                not_yet_impliment();
                break;
                //-----------------------------------------------------------------------------------------
                //    no support
                //-----------------------------------------------------------------------------------------
            default:
                PUTC('?');
                put_rn();
                break;
        }
    }
}
#endif	// defined(TARGET_LPC1768)
