/*
 * Mbed Application program / Using Akizuki BLE Module AE-TYBLE16
 *      on-board module : TAIYO YUDEN BLE 4.2 TYSA-B (EYSGJNAWY-WX)
 *
 *  http://akizukidenshi.com/catalog/g/gK-12339/
 *
 *  Refernce document
 *  https://www.yuden.co.jp/wireless_module/document/datareport2/jp/
 *                  TY_BLE_EYSGJNAWY_WX_BriefDataReport_V1_3_20170925J.pdf
 *
 * Copyright (c) 2017 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created:    October   27th, 2017
 *      Revised:    October   28th, 2017
 */

/*
    Tested condition
        mbed-os-5.6.3 & mbed Rev.154
    Nucleo-F446RE   TYBLE16  (also F411 & F042K6(only mbed))
        PA_9         pin5 / P0.03(UART_RX)
        PA_10        pin6 / P0.01(UART_TX)
                     pin7 / P0.02(UART_CTS) connected to pin8 / P0.00(UART_RTS)
        +3.3v        pin14 / +V
        GND          pin13 / GND
    Communcation with iPhone 7 iOS 11.0.3
        TYs Terminal App
        https://itunes.apple.com/jp/app/tys-terminal-app/id1184640145?mt=8
 */

//  Include --------------------------------------------------------------------
#include    "mbed.h"

//  Definition -----------------------------------------------------------------
// No input/output action then send '\n' after X sec
#define     TIME_OUT         5      // 5 sec
// after terminate Advertising then restart X sec 
#define     ADT_TIME        15      // 15 sec

//  Object/ Constructor --------------------------------------------------------
Serial      pc(USBTX,USBRX);
Serial      tyble16(PA_9, PA_10);
Ticker      t;

//  RAM ------------------------------------------------------------------------
bool        state_connection    = false;    // Connect Client or not 
bool        received_data       = false;    // data received
uint32_t    cmd_timeout_cntr    = ADT_TIME;

//  ROM / Constant data --------------------------------------------------------

//  Function prototypes --------------------------------------------------------
void check_connection(void);
void parse_input(char *buf);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
int main()
{
    pc.printf("\r\n\r\nApplication for AE-TYBLE16 Module\r\n");
    pc.printf("  created on %s %s\r\n", __DATE__, __TIME__);
    t.attach(check_connection, 1);
    while (true){
        static char rcv_bf[128];
        static uint8_t n = 0;
        char c = 0;
        while (tyble16.readable()){
            //----- data receive from Client -----------------------------------
            received_data = true;
            c = tyble16.getc();         // received data from client
            pc.putc(c);                 // show to console
            rcv_bf[n++] = c;            // save int buf
            if (c == '\n'){             // end one line
                pc.putc('\r');
                rcv_bf[n] = 0;
                if (n > 4){
                    parse_input(rcv_bf);    // ckeck command or not
                }
                n = 0;
            }
        }
        while (pc.readable()){
            //----- data send to Client ----------------------------------------
            c = pc.getc();
            tyble16.putc(c);
            pc.putc(c);                 // echo back
            if(c == '\r'){              // if CR then put LF
                tyble16.putc('\n');
                pc.putc('\n');
            }
        }
    }
}

// Check Response Events
void parse_input(char *buf)
{
    if        (strstr(buf, "CON")){     // Connection successful
        state_connection = true;
    } else if (strstr(buf, "DCO")){     // Disconnect
        state_connection = false; 
    } else if (strstr(buf, "ADT")){     // Advertising Timeout
        state_connection = false;
        cmd_timeout_cntr = 0;
    } else if (strstr(buf, "NAK")){     // Failed
        state_connection = false;
        if (cmd_timeout_cntr == 0){
            // Both Advertising Start. Connectable and Discoverable.
            tyble16.printf("BCD3\r\n");     // send above command
            cmd_timeout_cntr = ADT_TIME;
        }
    }
}

// Every one second, check communication status
void check_connection(void)
{
    static int8_t counter = 0;

    if ((received_data == false) && (state_connection == false)){
        if (++counter >= TIME_OUT){
            tyble16.putc('\n');
            counter = 0;
        }
    } else {
        counter = 0;
    }
    if (cmd_timeout_cntr != 0){
        --cmd_timeout_cntr;
    }
    received_data = false;
}
