/* mbed Microcontroller Library
 * Copyright (c) 2006-2014 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 *  Modified by Kenji Arai
 *      http://www.page.sannet.ne.jp/kenjia/index.html
 *      https://os.mbed.com/users/kenjiArai/
 *
 *      Started:    January    3rd, 2016
 *      Revised:    Feburary  13th, 2016
 *      Revised:    December  19th, 2019
 */

//#define EXAMPLE_9_MONITOR
#ifdef EXAMPLE_9_MONITOR

//  Include --------------------------------------------------------------------
#include "mbed.h"
#include "BLE.h"
#include "BatteryService.h"
#include "nRF51_Vdd.h"
#include "BME280.h"
#include "TYBLE16_BASE.h"

//  Definition -----------------------------------------------------------------
/* Set this if you need debug messages on the console;
 * it will have an impact on code-size and power consumption.
 */
#define NEED_CONSOLE_OUTPUT     1

#if NEED_CONSOLE_OUTPUT
#define DEBUG(...) { pc.printf(__VA_ARGS__); }
#else
#define DEBUG(...) /* nothing */
#endif /* #if NEED_CONSOLE_OUTPUT */

//  Object ---------------------------------------------------------------------
DigitalOut  led1(LED1, 1);
DigitalIn   sw(BUTTON1);
AnalogIn    ana0(A0);
DigitalIn   sw1(A1);
Serial      pc(USBTX, USBRX);
I2C         i2c(I2C_SDA, I2C_SCL);
BME280      hmtp(i2c);
SPI         spi(SPI_PSELMOSI0, SPI_PSELMISO0, SPI_PSELSCK0);
nRF51_Vdd   vdd(3.6f, 2.6f);
Ticker      t;

//  ROM / Constant data --------------------------------------------------------
const char *deviceName = "mbedMon";
const char *const opngmsg =
    "\x1b[2J\x1b[H" __FILE__ "\r\n" __DATE__ " " __TIME__ " (UTC)\r\n""\r\n";

//  RAM ------------------------------------------------------------------------
BatteryService *batteryService = NULL;
uint8_t batteryLevel = 50;

//  Function prototypes --------------------------------------------------------
extern void debug_interface(uint8_t mode);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
void disconnectionCallback
(const Gap::DisconnectionCallbackParams_t *disconnectionParams)
{
    DEBUG("Disconnected handle %u!\n\r", disconnectionParams->handle);
    DEBUG("Restarting the advertising process\n\r");
    // restart advertising
    BLE::Instance(BLE::DEFAULT_INSTANCE).gap().startAdvertising();
}

void blink(void)
{
    led1 = !led1;
}

int main(void)
{
    // ******************************************************
    // Here is a example to impliment the monitor
    // para: 1 -> goto montor and never comeback
    // ******************************************************
#if 1
    debug_interface(1);
#endif

    pc.puts(opngmsg);
    // Application program runs at here
    t.attach(blink, 1.0f);

    DEBUG("Initialising the nRF5x chip\r\n");

    // Check TYBLE-16 configuration
    cpu_sys();
    compile_condition();
    //
    BLE& ble = BLE::Instance(BLE::DEFAULT_INSTANCE);
    ble.init();
    ble.setDeviceName((const uint8_t *)deviceName);
    ble.onDisconnection(disconnectionCallback);

    batteryService = new BatteryService(ble, batteryLevel);

    /* setup advertising */
    ble.accumulateAdvertisingPayload
    (GapAdvertisingData::BREDR_NOT_SUPPORTED |
     GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                     (const uint8_t *)deviceName,
                                     strlen(deviceName));
    ble.setAdvertisingInterval(1000); /* 1000ms; in multiples of 0.625ms. */
    ble.startAdvertising();

    while (true) {
        // this will return upon any system event
        ble.waitForEvent();

        // the magic battery processing
        batteryLevel++;
#if 1
        if (batteryLevel > 100) {
            batteryLevel = 20;
        }
        DEBUG("Vdd: %4.3f\r\n", vdd.read_real_value());
#else
        DEBUG("Vdd: %f\r\n", vdd.read_real_value());
        batteryLevel = vdd.read();
#endif

        batteryService->updateBatteryLevel(batteryLevel);
        DEBUG("battery=%d\r\n", batteryLevel);

        DEBUG("analog input = %f\r\n", ana0.read());
        DEBUG("Temperature= %+5.2f [degC]\r\n", hmtp.getTemperature());

        // ******************************************************
        // Here is a example to impliment the monitor
        // para: 0 -> if Uart RX Data is ready then goto montor
        // ******************************************************
        debug_interface(0);
        ThisThread::sleep_for(1000);
    }
}

#endif
