/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
    Modified by Kenji Arai, December 19th, 2019
        https://os.mbed.com/users/kenjiArai/
 */

//#define EXAMPLE_6_THERMO
#ifdef EXAMPLE_6_THERMO

#include    <events/mbed_events.h>
#include    "mbed.h"
#include    "ble/BLE.h"
#include    "ble/services/HealthThermometerService.h"
#include    "TYBLE16_BASE.h"
#include    "TextLCD.h"
#include    "BME280.h"

#define NO_SENSOR   0
#define HAS_SENSOR  1
#define USE_LCD     1

DigitalOut      led1(LED1, 1);
Serial          pc(USBTX, USBRX);
I2C             i2c(I2C_SDA, I2C_SCL);
BME280          hmtp(i2c);
#if USE_LCD
TextLCD_I2C_N   lcd(&i2c, 0x7c, TextLCD::LCD16x2);  // LCD(Akizuki  AQM1602A)
#endif

const static char     DEVICE_NAME[]        = "TYBLE16";
static const uint16_t uuid16_list[]
= {GattService::UUID_HEALTH_THERMOMETER_SERVICE};

const char *const opngmsg =
    "\x1b[2J\x1b[H" __FILE__ "\r\n" __DATE__ " " __TIME__ " (UTC)\r\n""\r\n";

static float                      currentTemperature   = 39.6f;
static HealthThermometerService *thermometerServicePtr;

static EventQueue eventQueue(/* event count */ 16 * EVENTS_EVENT_SIZE);

/* Restart Advertising on disconnection*/
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *)
{
    BLE::Instance().gap().startAdvertising();
}

void updateSensorValue(void)
{
    static uint32_t count = 0;

#if NO_SENSOR
    /* Do blocking calls or whatever is necessary for sensor polling.
       In our case, we simply update the Temperature measurement. */
    currentTemperature
    = (currentTemperature + 0.1f > 43.0f) ? 39.6f : currentTemperature + 0.1f;
    thermometerServicePtr->updateTemperature(currentTemperature);
#elif HAS_SENSOR
    currentTemperature = hmtp.getTemperature();
    pc.printf("Temperature= %+5.2f [degC]", currentTemperature);
    pc.printf(", count = %8u\r\n", count++);
#if USE_LCD
    lcd.locate(0, 1);
    lcd.printf(" %+5.2f  %7u", currentTemperature, count);
#endif
    thermometerServicePtr->updateTemperature(currentTemperature);
#else
    thermometerServicePtr->updateTemperature(currentTemperature);
#endif
}

void periodicCallback(void)
{
    led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */

    if (BLE::Instance().gap().getState().connected) {
        eventQueue.call(updateSensorValue);
    }
}

void onBleInitError(BLE &ble, ble_error_t error)
{
    /* Initialization error handling should go here */
}

void printMacAddress()
{
    /* Print out device MAC address to the console*/
    Gap::AddressType_t addr_type;
    Gap::Address_t address;
    BLE::Instance().gap().getAddress(&addr_type, address);
    pc.printf("DEVICE MAC ADDRESS: ");
    for (int i = 5; i >= 1; i--) {
        pc.printf("%02x:", address[i]);
    }
    pc.printf("%02x\r\n", address[0]);
}

void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        onBleInitError(ble, error);
        return;
    }

    if (ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup primary service. */
    thermometerServicePtr =
        new HealthThermometerService(ble,
                                     currentTemperature,
                                     HealthThermometerService::LOCATION_EAR);

    /* setup advertising */
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::BREDR_NOT_SUPPORTED |
        GapAdvertisingData::LE_GENERAL_DISCOVERABLE
    );
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS,
        (uint8_t *)uuid16_list,
        sizeof(uuid16_list)
    );
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::THERMOMETER_EAR
    );
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::COMPLETE_LOCAL_NAME,
        (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME)
    );
    ble.gap().setAdvertisingType(
        GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED
    );
    ble.gap().setAdvertisingInterval(1000); /* 1000ms */
    ble.gap().startAdvertising();

    printMacAddress();
}

void scheduleBleEventsProcessing(BLE::OnEventsToProcessCallbackContext* context)
{
    BLE &ble = BLE::Instance();
    eventQueue.call(Callback<void()>(&ble, &BLE::processEvents));
}

int main()
{
    pc.puts(opngmsg);
#if USE_LCD
    lcd.locate(0, 0);
    //        1234567890123456
    lcd.puts("TYBLE16/Mbed-os5");
    lcd.locate(0, 1);
    //        1234567890123456
    lcd.puts(" by JH1PJL      ");
    lcd.setCursor(TextLCD_Base::CurOff_BlkOff);
    lcd.setContrast(0x19);
    ThisThread::sleep_for(2000);
#endif
    // Check TYBLE-16 configuration
    cpu_sys();
    if (compile_condition() == false) {
        pc.printf("This is wrong configuration!!\r\n");
        while(true) {
            led1 = !led1;
            ThisThread::sleep_for(200);
        }
    }
#if USE_LCD
    lcd.locate(0, 0);
    //          123456 7890123456
    lcd.printf("Temp: %cC   count", 0xdf);
#endif
    eventQueue.call_every(1000, periodicCallback);

    BLE &ble = BLE::Instance();
    ble.onEventsToProcess(scheduleBleEventsProcessing);
    ble.init(bleInitComplete);

    eventQueue.dispatch_forever();

    return 0;
}

#endif
