/*
 * mbed Application program for the mbed nRF51 series
 *
 *  Copyright (c) 2016,'18 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  https://os.mbed.com/users/kenjiArai/
 *      Created:    Feburary   1st, 2016
 *      Revised:    Feburary  18th, 2016
 *      Revised:    April     14th, 2018
 *
 */

#if defined(TARGET_NRF51822)

//  Include --------------------------------------------------------------------
#include "mbed.h"
#include "mon_hw_config.h"
#include "mon_hw_common.h"
#include "TYBLE16_BASE.h"

//  Object ---------------------------------------------------------------------

//  Definition -----------------------------------------------------------------
#define SIMPLE_NOT_SHOW      1   // 1=Simple

//  RAM ------------------------------------------------------------------------
uint32_t SystemFrequency;
#if USE_MEM
//  Memory range data (This is special case: save in RAM area! (others mbed's are in FLASH))
//      follows are only initial data and check FICR data for ROM & RAM size
uint32_t mem_range[][2] = { // Memory access range
    // On-chip non-volatile memory    //128 or 256KB Flash
    { 0x00000000, 0x0003ffff },
    // On-chip SRAM                    //16 or 32KB RAM
    { 0x20000000, 0x20007fff },
    { 0x10000000, 0x1000007f },            // FICR
    { 0x10001000, 0x100010ff },            // UICR
    { 0x40000000, 0x400245ff },            // Peripheral
    { 0x50000500, 0x500007ff }             // GPIO
};
#endif  // USE_MEM

//  ROM / Constant data --------------------------------------------------------
char *const mon_msg =
    "HW monitor only for mbed nRF51 series created on "__DATE__","__TIME__"";

char *const xmsg0 = "Not implimented yet";
char *const hmsg0 = "m  - Entry Memory Mode";
char *const hmsg1 = "m>? -> Help ";
char *const hmsg2 = "r  - Show I2C,SPI,UART,Timer,ADC Reg.";
char *const hmsg3 = "r>? -> Help";
char *const hmsg4 = "c  - CPU information";
char *const hmsg5 = "f  - Show Clock frequency";
char *const hmsg6 = "p  - GPIO information";
char *const hmsg7 = "w  - Power management";
char *const hmsg8 = "x  - Special command for Debug";
char *const hmsg9 = "q  - Quit (back to called routine)";

//  Function prototypes --------------------------------------------------------

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
// No function
void not_yet_impliment( void )
{
    put_r();
    PRINTF(xmsg0);
    put_rn();
}

// No function
#if (USE_MEM==0)||(USE_PORT==0)||(USE_UART==0)|| \
    (USE_SPI==0)||(USE_I2C==0)||(USE_SYS==0)
void not_select( void )
{
    PRINTF("Not select the function (refer mon_hw_config.h)");
    put_rn();
}
#endif

//  Help Massage
void msg_hlp_hw (void)
{
    PRINTF(mon_msg);
    put_rn();
    PRINTF(hmsg0);
    put_rn();
    PRINTF(hmsg1);
    put_rn();
    PRINTF(hmsg2);
    put_rn();
    PRINTF(hmsg3);
    put_rn();
    PRINTF(hmsg4);
    put_rn();
    PRINTF(hmsg5);
    put_rn();
    PRINTF(hmsg6);
    put_rn();
    PRINTF(hmsg7);
    put_rn();
    PRINTF(hmsg8);
    put_rn();
    PRINTF(hmsg9);
    put_rn();
}

#if USE_MEM
char *const rmsg0 = "FLASH      ";
char *const rmsg1 = "SRAM       ";
char *const rmsg2 = "FIR        ";
char *const rmsg3 = "UIR        ";
char *const rmsg4 = "Peripheral ";
char *const rmsg5 = "GPIO       ";

#include "mon_hw_mem.h"
#endif   // USE_MEM

//  Show Xbit register contents
void reg_print(uint16_t size, uint16_t reg)
{
    uint16_t i, j, k, n;

    if (size == 8) {
        PRINTF(" 7, 6, 5, 4, 3, 2, 1, 0");
        put_rn();
        i = 8;
        n = 0x80;
    } else if (size == 16) {
        PRINTF( "15,14,13,12,11,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0" );
        put_rn();
        i = 16;
        n = 0x8000;
    } else {
        PRINTF("0x%08x", reg);
        return;
    }
    PUTC(' ');
    for (; i>0; i--) {
        k = n >> (size-i);
        j = reg & k;
        if (j) {
            PUTC('1');
        } else {
            PUTC('0');
        }
        PUTC(' ');
        PUTC(' ');
    }
    PRINTF("  (0x%04x)", reg);
}

uint8_t bit_position(uint32_t x)
{
    uint8_t n;

    for (n = 0; n < 32; n++) {
        x = x >> 1UL;
        if (x & 0x1) {
            break;
        }
    }
    return n;
}

#if USE_FREQ
void freq_reg(void)
{
    uint32_t r0;

//    NRF_CLOCK->TASKS_HFCLKSTART = 0xffffffff;
//    NRF_CLOCK->HFCLKRUN = 0x00010001;

    put_r();
    //  Start HFCLK clock source
    r0 = NRF_CLOCK->TASKS_HFCLKSTART;
    PRINTF( "TASKS_HFCLKSTART:" );
    reg_print(SIZE32, r0);
    put_rn();
    //  Stop HFCLK clock source
    r0 = NRF_CLOCK->TASKS_HFCLKSTOP;
    PRINTF( "TASKS_HFCLKSTOP: " );
    reg_print(SIZE32, r0);
    put_rn();
    // Start LFCLK clock source
    r0 = NRF_CLOCK->TASKS_LFCLKSTART;
    PRINTF( "TASKS_LFCLKSTART:" );
    reg_print(SIZE32, r0);
    put_rn();
    //  Stop LFCLK clock source
    r0 = NRF_CLOCK->TASKS_LFCLKSTOP;
    PRINTF( "TASKS_LFCLKSTOP: " );
    reg_print(SIZE32, r0);
    put_rn();
    // Start calibration of LFCLK RC oscillator
    r0 = NRF_CLOCK->TASKS_CAL;
    PRINTF( "TASKS_CAL:       " );
    reg_print(SIZE32, r0);
    put_rn();
    //  Start calibration timer
    r0 = NRF_CLOCK->TASKS_CTSTART;
    PRINTF( "TASKS_CTSTART:   " );
    reg_print(SIZE32, r0);
    put_rn();
    // Stop calibration timer
    r0 = NRF_CLOCK->TASKS_CTSTOP;
    PRINTF( "TASKS_CTSTOP:    " );
    reg_print(SIZE32, r0);
    put_rn();
    //  HFCLK oscillator started
    r0 = NRF_CLOCK->EVENTS_HFCLKSTARTED;
    PRINTF( "EVENTS_HFCLKSTARTED:" );
    reg_print(SIZE32, r0);
    put_rn();
    // LFCLK oscillator started
    r0 = NRF_CLOCK->EVENTS_LFCLKSTARTED;
    PRINTF( "EVENTS_LFCLKSTARTED:" );
    reg_print(SIZE32, r0);
    put_rn();
    // Calibration of LFCLK RC oscillator completed
    r0 = NRF_CLOCK->EVENTS_DONE;
    PRINTF( "EVENTS_DONE:     " );
    reg_print(SIZE32, r0);
    put_rn();
    // Calibration timer timeout
    r0 = NRF_CLOCK->EVENTS_CTTO;
    PRINTF( "EVENTS_CTTO:     " );
    reg_print(SIZE32, r0);
    put_rn();
    // Interrupt enable set register
    r0 = NRF_CLOCK->INTENSET;
    PRINTF( "INTENSET:        " );
    reg_print(SIZE32, r0);
    put_rn();
    // Interrupt enable clear register
    r0 = NRF_CLOCK->INTENCLR;
    PRINTF( "INTENCLR:        " );
    reg_print(SIZE32, r0);
    put_rn();
    // Task HFCLKSTART trigger status
    r0 = NRF_CLOCK->HFCLKRUN;
    PRINTF( "HFCLKRUN:        " );
    reg_print(SIZE32, r0);
    put_rn();
    // high frequency clock status
    r0 = NRF_CLOCK->HFCLKSTAT;
    PRINTF( "HFCLKSTAT:       " );
    reg_print(SIZE32, r0);
    put_rn();
    // Task LFCLKSTART trigger status
    r0 = NRF_CLOCK->LFCLKRUN;
    PRINTF( "LFCLKRUN:        " );
    reg_print(SIZE32, r0);
    put_rn();
    // Low frequency clock status
    r0 = NRF_CLOCK->LFCLKSTAT;
    PRINTF( "LFCLKSTAT:       " );
    reg_print(SIZE32, r0);
    put_rn();
    // lock source for the LFCLK clock
    r0 = NRF_CLOCK->LFCLKSRCCOPY;
    PRINTF( "LFCLKSRCCOPY:    " );
    reg_print(SIZE32, r0);
    put_rn();
    // Clock source for the LFCLK clock
    r0 = NRF_CLOCK->LFCLKSRC;
    PRINTF( "LFCLKSRC:        " );
    reg_print(SIZE32, r0);
    put_rn();
    // Calibration timer interval
    r0 = NRF_CLOCK->CTIV;
    PRINTF( "CTIV:            " );
    reg_print(SIZE32, r0);
    put_rn();
    // Crystal frequency
    r0 = NRF_CLOCK->XTALFREQ;
    PRINTF( "XTALFREQ:        " );
    reg_print(SIZE32, r0);
    put_rn();
}
#endif  // USE_FREQ

#if USE_PORT
void port_all(void)
{
    uint32_t p0, p1;
    uint8_t n;

    put_r();
    PRINTF( "GPIO registers" );
    put_rn();
    p0 = NRF_GPIO->OUTSET;
    PRINTF( "OUT: " );
    reg_print(SIZE32, p0);
    PRINTF( " ;0=L, 1=H" );
    put_rn();
    p0 = NRF_GPIO->IN;
    PRINTF( "IN:  " );
    reg_print(SIZE32, p0);
    PRINTF(" ;0=L, 1=H");
    put_rn();
    p0 = NRF_GPIO->DIRSET;
    PRINTF("DIR: ");
    reg_print(SIZE32, p0);
    PRINTF(" ;0=in,1=out");
    put_rn();
    PRINTF("Dc=Disconnect,Cn=Connect,Ot=Out,Dis=Disabled,PU=Pullup,");
    PRINTF("PD=Pulldown,Hi=High,Lo=Low");
    for (n=0; n < 32; n++) {
        put_rn();
        p0 = NRF_GPIO->PIN_CNF[n];
        PRINTF("P0_%02d: DIR=", n);
        if (p0 & 0x1) {
            PRINTF("Ot");
        } else {
            PRINTF("In");
        }
        PRINTF(", IN=");
        if (p0 & 0x2) {
            PRINTF("Dc");
        } else {
            PRINTF("Cn");
        }
        PRINTF(", PULL=");
        p1 = (p0 >> 2) & 0x3;
        switch (p1) {
            case 0:
                PRINTF("Dis");
                break;
            case 1:
                PRINTF("PD ");
                break;
            case 3:
                PRINTF("PU ");
                break;
            default:
                PRINTF("?  ");
                break;
        }
        PRINTF(", Drive=");
        p1 = (p0 >> 8) & 0x7;
        switch (p1) {
            case 0:
                PRINTF("S0S1");
                break;
            case 1:
                PRINTF("H0S1");
                break;
            case 2:
                PRINTF("S0H1");
                break;
            case 3:
                PRINTF("H0H1");
                break;
            case 4:
                PRINTF("D0S1");
                break;
            case 5:
                PRINTF("D0H1");
                break;
            case 6:
                PRINTF("S0D1");
                break;
            case 7:
                PRINTF("H0D1");
                break;
            default:
                PRINTF("?   ");
                break;
        }
        PRINTF(", SENSE=");
        p1 = (p0 >> 16) & 0x3;
        switch (p1) {
            case 0:
                PRINTF("Dis");
                break;
            case 2:
                PRINTF("Hi ");
                break;
            case 3:
                PRINTF("Lo ");
                break;
            default:
                PRINTF("?  ");
                break;
        }
    }
}
#endif  // USE_PORT

#if USE_UART
void uart_reg(void)
{
    uint32_t p0, p1;

    put_r();
    PRINTF("Registers/UART");
    put_rn();
    p0 = NRF_UART0->ENABLE;
    p1 = p0 & 0x7;
    PRINTF("UART is ");
    if (p1 == 4) {
        PRINTF("enabled");
        put_rn();
        p0 = NRF_UART0->BAUDRATE;
        switch(p0) {
            case UART_BAUDRATE_BAUDRATE_Baud1200:
                p1 = 1200;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud2400:
                p1 = 2400;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud4800:
                p1 = 4800;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud9600:
                p1 = 9600;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud14400:
                p1 = 14400;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud19200:
                p1 = 19200;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud28800:
                p1 = 28800;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud38400:
                p1 = 38400;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud57600:
                p1 = 576000;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud76800:
                p1 = 76800;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud115200:
                p1 = 115200;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud230400:
                p1 = 230400;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud250000:
                p1 = 250000;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud460800:
                p1 = 460800;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud921600:
                p1 = 921600;
                break;
            case UART_BAUDRATE_BAUDRATE_Baud1M:
                PRINTF("Baud rate: 1M");
                p1 = 1;
                break;
            default:
                PRINTF("Baud rate: Unknown");
                p1 = 0;
        }
        if (p1 > 10) {
            PRINTF("Baud rate: %dbps", p1);
        }
        put_rn();
        p0 = NRF_UART0->PSELTXD;
        PRINTF("TXD Pin: ");
        if (p0 == 0xffffffff) {
            PRINTF("Unknown");
        } else {
            PRINTF("P0_%02d", p0);
        }
        put_rn();
        p0 = NRF_UART0->PSELRXD;
        PRINTF("RXD Pin: ");
        if (p0 == 0xffffffff) {
            PRINTF("Unknown");
        } else {
            PRINTF("P0_%02d", p0);
        }
        put_rn();
        p0 = NRF_UART0->PSELRTS;
        PRINTF("RTS Pin: ");
        if (p0 == 0xffffffff) {
            PRINTF("Unknown");
        } else {
            PRINTF("P0_%02d", p0);
        }
        put_rn();
        p0 = NRF_UART0->PSELCTS;
        PRINTF("CTS Pin: ");
        if (p0 == 0xffffffff) {
            PRINTF("Unknown");
        } else {
            PRINTF("P0_%02d", p0);
        }
#if SIMPLE_NOT_SHOW
        put_rn();
        p0 = NRF_UART0->SHORTS;
        PRINTF("SHORTS:  ");
        reg_print(SIZE32, p0);
        put_rn();
        p0 = NRF_UART0->ERRORSRC;
        PRINTF("ERRORSRC:");
        reg_print(SIZE32, p0);
#endif
    } else {
        PRINTF("disabled");
    }
    put_rn();
}
#endif  //USE_UART

#if USE_SPI
void spi_reg(uint8_t n)
{
    uint32_t r0, r1, r2, r3;

    put_r();
    PRINTF("Registers/SPI");
    put_rn();
    if (n == 0) {
        r0 = NRF_SPI0->ENABLE;
        PRINTF("SPI0 is ");
    } else {
        r0 = NRF_SPI1->ENABLE;
        PRINTF("SPI1 is ");
    }
    r1 = r0 & 0x7;
    if (r1 == 1) {
        PRINTF("enabled");
        put_rn();
        if (n == 0) {
            r0 = NRF_SPI0->FREQUENCY;
            r1 = NRF_SPI0->PSELMOSI;
            r2 = NRF_SPI0->PSELMISO;
            r3 = NRF_SPI0->PSELSCK;
        } else {
            r0 = NRF_SPI1->FREQUENCY;
            r1 = NRF_SPI1->PSELMOSI;
            r2 = NRF_SPI1->PSELMISO;
            r3 = NRF_SPI1->PSELSCK;
        }
        PRINTF("Clock:");
        switch(r0) {
            case SPI_FREQUENCY_FREQUENCY_K125:
                PRINTF("125k");
                break;
            case SPI_FREQUENCY_FREQUENCY_K250:
                PRINTF("250k");
                break;
            case SPI_FREQUENCY_FREQUENCY_K500:
                PRINTF("500k");
                break;
            case SPI_FREQUENCY_FREQUENCY_M1:
                PRINTF("1M");
                break;
            case SPI_FREQUENCY_FREQUENCY_M2:
                PRINTF("2M");
                break;
            case SPI_FREQUENCY_FREQUENCY_M4:
                PRINTF("4M");
                break;
            case SPI_FREQUENCY_FREQUENCY_M8:
                PRINTF("8M");
                break;
            default:
                PRINTF("Unknown");
        }
        PRINTF("bps");
        put_rn();
        PRINTF("MOSI Pin: ");
        if (r1 == 0xffffffff) {
            PRINTF("Unknown");
        } else {
            PRINTF("P0_%02d", r1);
        }
        put_rn();
        PRINTF("MISO Pin: ");
        if (r2 == 0xffffffff) {
            PRINTF("Unknown");
        } else {
            PRINTF("P0_%02d", r2);
        }
        put_rn();
        PRINTF("SCK  Pin: ");
        if (r3 == 0xffffffff) {
            PRINTF("Unknown");
        } else {
            PRINTF("P0_%02d", r3);
        }
    } else {
        PRINTF("disabled");
    }
    put_rn();
}
#endif  //USE_SPI

#if USE_ADC
#if 0
void adc_reg(void)
{
    uint32_t r0, r1;

    put_r();
    PRINTF("Registers/ADC");
    put_rn();
    r0 = NRF_ADC->ENABLE;
    r1 = r0 & 0x3;
    if (r1 == 1) {
        PRINTF("enabled");
        put_rn();
#if 1
        put_rn();
        PRINTF("RES   :  ");
        reg_print(SIZE32, NRF_ADC->RES);
        put_rn();
        PRINTF("INPSEL:  ");
        reg_print(SIZE32, NRF_ADC->INPSEL);
        PRINTF("REFSEL:  ");
        reg_print(SIZE32, NRF_ADC->REFSEL);
        put_rn();
        PRINTF("PSEL  :  ");
        reg_print(SIZE32, NRF_ADC->PSEL);
        PRINTF("EXTREFSEL:  ");
        reg_print(SIZE32, NRF_ADC->EXTREFSEL);
        put_rn();
#endif
    } else {
        PRINTF("disabled");
    }
    put_rn();
}
#else

#if 0
TASKS_START;                       // Start an ADC conversion.                                              */
TASKS_STOP;                        // Stop ADC.                                                             */
EVENTS_END;                        // ADC conversion complete.                                              */
INTENSET;                          // Interrupt enable set register.                                        */
INTENCLR;                          // Interrupt enable clear register.                                      */
BUSY;                              // ADC busy register.                                                    */
ENABLE;                            // ADC enable.                                                           */
CONFIG;                            // ADC configuration register.                                           */
RESULT;                            // Result of ADC conversion.                                             */
POWER;                             // Peripheral power control.
#endif

void adc_reg(void)
{
    uint32_t r0, r1;

    put_r();
    PRINTF("Registers/ADC");
    put_rn();
    PRINTF("TASKS_START:  ");
    reg_print(SIZE32, NRF_ADC->TASKS_START);
    put_rn();
    PRINTF("TASKS_STOP :  ");
    reg_print(SIZE32, NRF_ADC->TASKS_STOP);
    put_rn();
    PRINTF("EVENTS_END :  ");
    reg_print(SIZE32, NRF_ADC->EVENTS_END);
    put_rn();
    PRINTF("INTENSET   :  ");
    reg_print(SIZE32, NRF_ADC->INTENSET);
    put_rn();
    PRINTF("INTENCLR   :  ");
    reg_print(SIZE32, NRF_ADC->INTENCLR);
    put_rn();
    PRINTF("BUSY       :  ");
    reg_print(SIZE32, NRF_ADC->BUSY);
    put_rn();
    PRINTF("ENABLE     :  ");
    reg_print(SIZE32, NRF_ADC->ENABLE);
    put_rn();
    PRINTF("CONFIG     :  ");
    reg_print(SIZE32, NRF_ADC->CONFIG);
    put_rn();
    PRINTF("RESULT     :  ");
    reg_print(SIZE32, NRF_ADC->RESULT);
    put_rn();
    PRINTF("POWER      :  ");
    reg_print(SIZE32, NRF_ADC->POWER);
    put_rn();
    // judge each condition
    r0 = NRF_ADC->ENABLE;
    r1 = r0 & 0x3;
    PRINTF("ADC is ");
    if (r1 == 1) {
        PRINTF("enabled");
    } else {
        PRINTF("disabled");
    }
    put_rn();
    r0 = NRF_ADC->CONFIG;
    r1 = r0 & 0x03;
    PRINTF("Resolution : ");
    if (r1 == 0) {
        PRINTF("8bit");
    } else if (r1 == 1) {
        PRINTF("9bit");
    } else if (r1 == 2) {
        PRINTF("10bit");
    }
    put_rn();
    r1 = (r0 & 0x1c) >> 2;
    PRINTF("INPSEL: ");
    switch (r1) {
        case 0:
            PRINTF("Pin specified by CONFIG.PSEL with no prescaling");
            break;
        case 1:
            PRINTF("Pin specified by CONFIG.PSEL with 2/3 prescaling");
            break;
        case 2:
            PRINTF("Pin specified by CONFIG.PSEL with 1/3 prescaling");
            break;
        case 5:
            PRINTF("VDD with 2/3 prescaling");
            break;
        case 6:
            PRINTF("VDD with 1/3 prescaling");
            break;
        default:
            PRINTF("???");
            break;
    }
    put_rn();
    r1 = (r0 & 0x60) >> 6;
    PRINTF("REFSEL: ");
    switch (r1) {
        case 0:
            PRINTF("VBG // Use internal 1.2 V band gap reference");
            break;
        case 1:
            PRINTF("External // Use external ");
            PRINTF("reference specified by CONFIG. EXTREFSEL");
            break;
        case 2:
            PRINTF("SupplyOneHalfPrescaling // Use VDD with 1/2 prescaling");
            break;
        case 3:
            PRINTF("SupplyOneThirdPrescaling // Use VDD with 1/3 prescaling");
            break;
        default:
            PRINTF("???");
            break;
    }
    put_rn();
    r1 = (r0 & 0xff00) >> 8;
    PRINTF("PSEL: ");
    switch (r1) {
        case 0:
            PRINTF("Disabled");
            break;
        case 1:
            PRINTF("AnalogInput0");
            break;
        case 2:
            PRINTF("AnalogInput1");
            break;
        case 4:
            PRINTF("AnalogInput2");
            break;
        case 8:
            PRINTF("AnalogInput3");
            break;
        case 16:
            PRINTF("AnalogInput4");
            break;
        case 32:
            PRINTF("AnalogInput5");
            break;
        case 64:
            PRINTF("AnalogInput6");
            break;
        case 128:
            PRINTF("AnalogInput7");
            break;
        default:
            PRINTF("???");
            break;
    }
    put_rn();
}
#endif
#endif  //USE_ADC

#if 0
TASKS_STARTRX;  //Start 2-Wire master receive sequence
TASKS_STARTTX;  //Start 2-Wire master transmit sequence
TASKS_STOP;     //Stop 2-Wire transaction
TASKS_SUSPEND;  //Suspend 2-Wire transaction
TASKS_RESUME;   //Resume 2-Wire transaction
EVENTS_STOPPED; //Two-wire stopped
EVENTS_RXDREADY;//Two-wire ready to deliver new RXD byte received
EVENTS_TXDSENT; //Two-wire finished sending last TXD byte
EVENTS_ERROR;   //Two-wire error detected
EVENTS_BB;      //Two-wire byte boundary
EVENTS_SUSPENDED;//Two-wire suspended
SHORTS;         //Shortcuts for TWI
INTENSET;       //Interrupt enable set register
INTENCLR;       //Interrupt enable clear register
ERRORSRC;       //Two-wire error source. Write error field to 1 to clear error
ENABLE;         //Enable two-wire master
PSELSCL;        //Pin select for SCL
PSELSDA;        //Pin select for SDA
RXD;            //RX data register
TXD;            //TX data register
FREQUENCY;      //Two-wire frequency
ADDRESS;// Address used in the two-wire transfer.                                */
POWER;//Peripheral power control.
#endif

#if USE_I2C
void i2c_reg(uint8_t n)
{
    uint32_t r0, r1, r2;
#if SIMPLE_NOT_SHOW
    uint32_t r3, r4;
#endif

    put_r();
    PRINTF("Registers/TWI(I2C)");
    put_rn();
    if (n == 0) {
        r0 = NRF_TWI0->ENABLE;
        PRINTF("TWI0 is ");
    } else {
        r0 = NRF_TWI1->ENABLE;
        PRINTF("TWI1 is ");
    }
    r1 = r0 & 0x7;
    if (r1 == 5) {
        PRINTF("enabled");
        put_rn();
        if (n == 0) {
            r0 = NRF_TWI0->FREQUENCY;
            r1 = NRF_TWI0->PSELSCL;
            r2 = NRF_TWI0->PSELSDA;
#if SIMPLE_NOT_SHOW
            r3 = NRF_TWI0->SHORTS;
            r4 = NRF_TWI0->ERRORSRC;
#endif
        } else {
            r0 = NRF_TWI1->FREQUENCY;
            r1 = NRF_TWI1->PSELSCL;
            r2 = NRF_TWI1->PSELSDA;
#if SIMPLE_NOT_SHOW
            r3 = NRF_TWI1->SHORTS;
            r4 = NRF_TWI1->ERRORSRC;
#endif
        }
        PRINTF("Clock: ");
        if (r0 == TWI_FREQUENCY_FREQUENCY_K100) {
            PRINTF("100KHz");
        } else if (p0 == TWI_FREQUENCY_FREQUENCY_K250) {
            PRINTF("250KHz");
        } else if (p0 == TWI_FREQUENCY_FREQUENCY_K400) {
            PRINTF("400KHz");
        } else {
            PRINTF("Unknown");
        }
        put_rn();
        PRINTF("SCL Pin: ");
        if (r1 == 0xffffffff) {
            PRINTF("Unknown");
        } else {
            PRINTF("P0_%02d", r1);
        }
        put_rn();
        PRINTF("SDA Pin: ");
        if (r2 == 0xffffffff) {
            PRINTF("Unknown");
        } else {
            PRINTF("P0_%02d", r2);
        }
#if SIMPLE_NOT_SHOW
        put_rn();
        PRINTF("SHORTS:  ");
        reg_print(SIZE32, r3);
        put_rn();
        PRINTF("ERRORSRC:");
        reg_print(SIZE32, r4);
#endif
    } else {
        PRINTF("disabled");
    }
    put_rn();
}
#endif  //USE_I2C

#if USE_POWER
void power_reg(void)
{
    put_r();
    PRINTF("POWER Management");
    put_rn();
    // Enable low power mode (variable latency)
    PRINTF("TASKS_CONSTLAT:");
    reg_print(SIZE32, NRF_POWER->TASKS_CONSTLAT);
    put_rn();
    // Enable low power mode (variable latency)
    PRINTF("TASKS_LOWPWR:  ");
    reg_print(SIZE32, NRF_POWER->TASKS_LOWPWR);
    put_rn();
    // Power failure warning
    PRINTF("EVENTS_POFWARN:");
    reg_print(SIZE32, NRF_POWER->EVENTS_POFWARN);
    put_rn();
    // Interrupt enable set register
    PRINTF("INTENSET:  ");
    reg_print(SIZE32, NRF_POWER->INTENSET);
    put_rn();
    // Interrupt enable clear register
    PRINTF("INTENCLR:  ");
    reg_print(SIZE32, NRF_POWER->INTENCLR);
    put_rn();
    // Reset reason
    PRINTF("RESETREAS: ");
    reg_print(SIZE32, NRF_POWER->RESETREAS);
    put_rn();
    // Ram status register
    PRINTF("RAMSTATUS: ");
    reg_print(SIZE32, NRF_POWER->RAMSTATUS);
    put_rn();
    // System off register
    PRINTF("SYSTEMOFF: ");
    reg_print(SIZE32, NRF_POWER->SYSTEMOFF);
    put_rn();
    // Power failure configuration
    PRINTF("POFCON:    ");
    reg_print(SIZE32, NRF_POWER->POFCON);
    put_rn();
    // General purpose retention register
    PRINTF("GPREGRET:  ");
    reg_print(SIZE32, NRF_POWER->GPREGRET);
    put_rn();
    // Ram on/off
    PRINTF("RAMON:     ");
    reg_print(SIZE32, NRF_POWER->RAMON);
    put_rn();
    // Ram on/off
    PRINTF("RAMONB:    ");
    reg_print(SIZE32, NRF_POWER->RAMONB);
    put_rn();
    // DCDC converter enable configuration register
    PRINTF("DCDCEN:    ");
    reg_print(SIZE32, NRF_POWER->DCDCEN);
    put_rn();
    // DCDC power-up force register
    PRINTF("DCDCFORCE: ");
    reg_print(SIZE32, NRF_POWER->DCDCFORCE);
    put_rn();
}
#endif  //USE_TIMER

#if USE_POWER
void timer_reg(uint8_t n)
{
    uint32_t r0, r1, r2, r3, r4, r5, r6, r7, r8, r9, r10;

    put_r();
    PRINTF("Registers/TIMER");
    if (n == 0) {
        r0 = NRF_TIMER0->SHORTS;
        r1 = NRF_TIMER0->INTENSET;
        r2 = NRF_TIMER0->INTENCLR;
        r3 = NRF_TIMER0->MODE;
        r4 = NRF_TIMER0->BITMODE;
        r5 = NRF_TIMER0->PRESCALER;
        r6 = NRF_TIMER0->CC[0];
        r7 = NRF_TIMER0->CC[1];
        r8 = NRF_TIMER0->CC[2];
        r9 = NRF_TIMER0->CC[3];
        r10 = NRF_TIMER0->POWER;
        PRINTF("0");
    } else if (n == 1) {
        r0 = NRF_TIMER1->SHORTS;
        r1 = NRF_TIMER1->INTENSET;
        r2 = NRF_TIMER1->INTENCLR;
        r3 = NRF_TIMER1->MODE;
        r4 = NRF_TIMER1->BITMODE;
        r5 = NRF_TIMER1->PRESCALER;
        r6 = NRF_TIMER1->CC[0];
        r7 = NRF_TIMER1->CC[1];
        r8 = NRF_TIMER1->CC[2];
        r9 = NRF_TIMER1->CC[3];
        r10 = NRF_TIMER1->POWER;
        PRINTF("1");
    } else if (n == 2) {
        r0 = NRF_TIMER2->SHORTS;
        r1 = NRF_TIMER2->INTENSET;
        r2 = NRF_TIMER2->INTENCLR;
        r3 = NRF_TIMER2->MODE;
        r4 = NRF_TIMER2->BITMODE;
        r5 = NRF_TIMER2->PRESCALER;
        r6 = NRF_TIMER2->CC[0];
        r7 = NRF_TIMER2->CC[1];
        r8 = NRF_TIMER2->CC[2];
        r9 = NRF_TIMER2->CC[3];
        r10 = NRF_TIMER2->POWER;
        PRINTF("2");
    } else {
        return;
    }
    put_rn();
    PRINTF("SHORTS:   ");
    reg_print(SIZE32, r0);
    put_rn();
    PRINTF("INTENSET: ");
    reg_print(SIZE32, r1);
    put_rn();
    PRINTF("INTENCLR: ");
    reg_print(SIZE32, r2);
    put_rn();
    PRINTF("MODE:     ");
    reg_print(SIZE32, r3);
    put_rn();
    PRINTF("BITMODE:  ");
    reg_print(SIZE32, r4);
    put_rn();
    PRINTF("PRESCALER:");
    reg_print(SIZE32, r5);
    put_rn();
    PRINTF("cc[0]: ");
    reg_print(SIZE32, r6);
    put_rn();
    PRINTF("cc[1]: ");
    reg_print(SIZE32, r7);
    put_rn();
    PRINTF("cc[2]: ");
    reg_print(SIZE32, r8);
    put_rn();
    PRINTF("cc[3]: ");
    reg_print(SIZE32, r9);
    put_rn();
    PRINTF("POWER: ");
    reg_print(SIZE32, r10);
    put_rn();
}
#endif  //USE_TIMER

//------------------------------------------------------------------------------
//  Monitor Main Program
//------------------------------------------------------------------------------
int mon_hw (void)
{
    char *ptr;

    put_r();
    PRINTF("%s [Help:'?' key]", mon_msg);
    put_rn();
    for (;;) {
        put_r();
        PUTC('>');
        ptr = linebuf;
        get_line(ptr, buf_size);
        switch (*ptr++) {
                //--------------------------------------------------------------
                //    Memory
                //--------------------------------------------------------------
            case 'm' :
#if USE_MEM
                mem_inf(ptr);
                put_rn();
#else
                not_select();
#endif   // USE_MEM
                break;
                //--------------------------------------------------------------
                //    POWER
                //--------------------------------------------------------------
            case 'w' :
#if USE_POWER
                power_reg();
#else
                not_select();
#endif   // USE_POWER
                break;
                //--------------------------------------------------------------
                //    Frequency(Clock)
                //--------------------------------------------------------------
            case 'f' :
#if USE_FREQ
                freq_reg();
#else
                not_select();
#endif   // USE_FREQ
                break;
                //--------------------------------------------------------------
                //    GPIO
                //--------------------------------------------------------------
            case 'p' :
#if USE_PORT
                port_all();
                put_rn();
#else
                not_select();
#endif   // USE_PORT
                break;
                //--------------------------------------------------------------
                //    Register
                //--------------------------------------------------------------
            case 'r' :
                uint8_t  r_flg;
                put_r();
                PRINTF("Reg. Mode u,i,s,t,a & ?");
                put_rn();
                r_flg = 0;
                for (; r_flg != 0xff;) {
                    PRINTF("r>");
                    ptr = linebuf;
                    get_line(ptr, buf_size);
                    put_r();
                    switch(*ptr++) {
                        case 'u' :
#if USE_UART
                            uart_reg();
#else
                            not_select();
#endif // USE_UART
                            break;
                        case 'i' :
#if USE_I2C
                            put_r();
                            i2c_reg(0);
                            put_rn();
                            i2c_reg(1);
#else
                            not_select();
#endif // USE_I2C
                            break;
                        case 's' :
#if USE_SPI
                            spi_reg(0);
                            put_rn();
                            spi_reg(1);
#else
                            not_select();
#endif // USE_SPI
                            break;
                        case 't' :      //
#if USE_TIMER
                            timer_reg(0);
                            put_rn();
                            timer_reg(1);
                            put_rn();
                            timer_reg(2);
#else
                            not_select();
#endif // USE_TIMER
                            break;
                        case 'a' :      //
#if USE_ADC
                            adc_reg();
#else
                            not_select();
#endif // USE_SPI
                            break;
                        case 'q' :        // quit
                            r_flg = 0xff;
                            break;
                        case '?' :
                            PRINTF("u - UART");
                            put_rn();
                            PRINTF("i - I2C");
                            put_rn();
                            PRINTF("s - SPI");
                            put_rn();
                            PRINTF("t - TIMER");
                            put_rn();
                            PRINTF("a - ADC");
                            put_rn();
                            PRINTF("q - Exit mode");
                            put_rn();
                            break;
                        default:
                            PUTC('?');
                            put_rn();
                    }
                }
                PRINTF("Return to All Mode");
                put_rn();
                break;
                //--------------------------------------------------------------
                //    CPU
                //--------------------------------------------------------------
            case 'c' :    // CPU related information
#if USE_SYS
                put_r();
                cpu_sys();
#else
                not_select();
#endif   // USE_SYS
                break;
                //--------------------------------------------------------------
                //    Help
                //--------------------------------------------------------------
            case '?' :
                put_r();
                msg_hlp_hw();
                break;
                //--------------------------------------------------------------
                //    Return to main routine
                //--------------------------------------------------------------
            case 'q' :        // Quit
                PRINTF("\rReturn to monitor ");
                return 0;
                //--------------------------------------------------------------
                //    Special command for DEBUG
                //--------------------------------------------------------------
            case 'x' :
                not_yet_impliment();
                break;
                //--------------------------------------------------------------
                //    no support
                //--------------------------------------------------------------
            default:
                PUTC('?');
                put_rn();
                break;
        }
    }
}

#endif  // defined(TARGET_NRF51822)
