/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 *  /////// Works well on TYBLE16 Module ///////
 *  Modified by Kenji Arai
 *      http://www.page.sannet.ne.jp/kenjia/index.html
 *      https://os.mbed.com/users/kenjiArai/
 *
 *      Started:  Feburary   18th, 2018
 *      Revised:  April      14th, 2018
 *
 *  Original:
 *      nRF51822_SimpleControls
 *      https://developer.mbed.org/teams
 *                      /Bluetooth-Low-Energy/code/BLE_EddystoneBeacon_Service/
 *  Tested Controller Device:
 *      iPhone6 Physical Web (PhyWeb) By Viet Hoa Dinh
 *      https://itunes.apple.com/us/app/physical-web/id927653608?mt=8
 */

//#define EXAMPLE_2_EDDYSTONE_BEACON
#ifdef EXAMPLE_2_EDDYSTONE_BEACON

//  Include --------------------------------------------------------------------
#include "mbed.h"
#include "TYBLE16_BASE.h"
#include "BLE.h"
//#include "EddystoneService.h"
#include "../Eddystone/EddystoneService.h"

//  Object ---------------------------------------------------------------------
DigitalOut  led(P0_5);
Serial      pc(USBTX, USBRX);

//  Definition -----------------------------------------------------------------

//  RAM ------------------------------------------------------------------------
EddystoneService    *eddyServicePtr;
Gap::Address_t      my_mac;

//  ROM / Constant data --------------------------------------------------------
char *const opngmsg =
    "\x1b[2J\x1b[H"__FILE__ "\r\n"__DATE__ " " __TIME__ " (UTC)\r\n""\r\n";

//  Function prototypes --------------------------------------------------------
int8_t check_dice(void);
uint16_t update_vdd(uint16_t x);
uint16_t update_temp(uint16_t x);
void onBleInitError(BLE::InitializationCompleteCallbackContext* initContext);
void bleInitComplete(BLE::InitializationCompleteCallbackContext* initContext);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
// !!!!!!!! Please select number of 0 to 6 !!!!!!!!!!!!!!!!!
int8_t check_dice(void)
{
    return 5;
}

void bleInitComplete(BLE::InitializationCompleteCallbackContext* initContext)
{
    BLE         &ble  = initContext->ble;
    ble_error_t error = initContext->error;

    if (error != BLE_ERROR_NONE) {
        onBleInitError(initContext);
        return;
    }
    // Set UID and TLM frame data
    const UIDNamespaceID_t uidNamespaceID =
    {0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99};
    const UIDInstanceID_t  uidInstanceID  =
    {0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF};
    uint8_t tlmVersion = 0x00;

    Gap::AddressType_t my_mac_type;
    ble.gap().getAddress(&my_mac_type, my_mac);
    // Initialize a EddystoneBeaconConfig service
    // Values for ADV packets related to firmware levels,
    // calibrated based on measured values at 1m
    static const PowerLevels_t defaultAdvPowerLevels = {-47, -33, -21, -13};
    // Values for radio power levels, provided by manufacturer.
    static const PowerLevels_t radioPowerLevels      = {-30, -16, -4, 4};
    eddyServicePtr = new EddystoneService(ble,
                                          defaultAdvPowerLevels,
                                          radioPowerLevels);
    // created short cut web addres by http://bitly.oshiire.org/
    switch (check_dice()) {
        case 1:     // Switch sience(mbed)
            eddyServicePtr->setURLData("http://bit.ly/1oJh91B");
            break;
        case 2:    // switch sience(HP)
            eddyServicePtr->setURLData("http://bit.ly/1oJhP7g");
            break;
        case 3:    // Taiyo Yuden BLE
            eddyServicePtr->setURLData("http://bit.ly/1VvuCVr");
            break;
        case 4:    // Taiyo Yuden
            eddyServicePtr->setURLData("http://bit.ly/1Vvtp0l");
            break;
        case 5:    // JH1PJL(mbed)
            eddyServicePtr->setURLData("http://bit.ly/1Vvt51J");
            break;
        case 6:    // JH1PJL(HP)
            eddyServicePtr->setURLData("http://bit.ly/1VvteT0");
            break;
        case 0:
        default:   // Mbed
            eddyServicePtr->setURLData("http://mbed.org");
            break;
    }
    eddyServicePtr->setUIDData(&uidNamespaceID, &uidInstanceID);
    eddyServicePtr->setTLMData(tlmVersion);
    // Start Eddystone in config mode
    eddyServicePtr->startBeaconService(5, 5, 5);
}

int main(void)
{
    pc.puts(opngmsg);
    led = 1;
    // Check TYBLE-16 configuration
    cpu_sys();
    if (compile_condition() == false) {
        pc.printf("This is wrong configuration!!\r\n");
        while(true) {
            led = !led;
            wait(0.2);
        }
    }
    BLE& ble = BLE::Instance(BLE::DEFAULT_INSTANCE);
    ble.init(bleInitComplete);
    while (ble.hasInitialized() == false) {
        ;
    }
    //
    led = 0;
    //pc.printf("line:%d\r\n", __LINE__);
    while (true) {
        ble.waitForEvent();
    }
}

void onBleInitError(BLE::InitializationCompleteCallbackContext* initContext)
{
    // Initialization error handling goes here...
    (void) initContext;
}

#endif
