/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
    Modified by Kenji Arai, Feburary 25th, 2018
 */

#include "mbed.h"
#include "TYBLE16_BASE.h"
#include "ble/BLE.h"
#include "ble/services/HealthThermometerService.h"
#include "TextLCD.h"
#include "HDC1000.h"

#define USE_LCD     0

DigitalOut      led1(P0_5);
Serial          pc(P0_1, P0_3);
I2C             i2c(P0_4, P0_6);
HDC1000         hmtp(i2c);
#if USE_LCD
TextLCD_I2C_N   lcd(&i2c, 0x7c, TextLCD::LCD8x2);  // LCD(Akizuki  AQM0802A)
#endif

static HealthThermometerService *thermometerServicePtr;

const static char     DEVICE_NAME[]        = "TYBLE16";
static const uint16_t uuid16_list[]        
                            = {GattService::UUID_HEALTH_THERMOMETER_SERVICE};
static volatile bool  triggerSensorPolling = false;
static float          currentTemperature   = 21.0;

/* Restart Advertising on disconnection*/
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    BLE::Instance().gap().startAdvertising();
}

void periodicCallback(void)
{
    led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */
    triggerSensorPolling = true;
}

/**
 * This function is called when the ble initialization process has failed
 */
void onBleInitError(BLE &ble, ble_error_t error)
{
    /* Avoid compiler warnings */
    (void) ble;
    (void) error;
    /* Initialization error handling should go here */
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }
    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup primary service. */
    thermometerServicePtr = new HealthThermometerService(
            ble,
            currentTemperature,
            HealthThermometerService::LOCATION_EAR
    );
    /* setup advertising */
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::BREDR_NOT_SUPPORTED |
        GapAdvertisingData::LE_GENERAL_DISCOVERABLE
    );
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS,
        (uint8_t *)uuid16_list,
        sizeof(uuid16_list)
    );
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::THERMOMETER_EAR
    );
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::COMPLETE_LOCAL_NAME,
        (uint8_t *)DEVICE_NAME,
        sizeof(DEVICE_NAME)
    );
    ble.gap().setAdvertisingType(
        GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED
    );
    ble.gap().setAdvertisingInterval(1000); /* 1000ms */
    ble.gap().startAdvertising();    
}

int main(void)
{
    led1 = 1;
    Ticker ticker;
    ticker.attach(periodicCallback, 1);
#if USE_LCD
    lcd.locate(0, 0);
    //        12345678
    lcd.puts("12345678");
    lcd.locate(0, 1);
    //        12345678
    lcd.puts(" JH1PJL ");
    lcd.setCursor(TextLCD_Base::CurOff_BlkOff);
    lcd.setContrast(0x19);
    wait(2.0f);
#endif
    BLE &ble = BLE::Instance();
    ble.init(bleInitComplete);
    while (ble.hasInitialized()  == false) { /* spin loop */ }
    // Check TYBLE-16 configuration
    cpu_sys();
    if (compile_condition() == false) {
        pc.printf("This is wrong configuration!!\r\n");
        while(true) {
            led1 = !led1;
            wait(0.2);
        }
    }
    //
    while (true) {
        if (triggerSensorPolling && ble.gap().getState().connected) {
            triggerSensorPolling = false;
            currentTemperature = hmtp.temperature();
            thermometerServicePtr->updateTemperature(currentTemperature);
            pc.printf("Temperature= %+5.2f [degC]\r\n", currentTemperature);
#if USE_LCD
            lcd.locate(0, 0);
            //          123456 78
            lcd.printf("Temp: %cC", 0xdf);
            lcd.locate(0, 1);
            lcd.printf(" %+5.2f ", currentTemperature);
#endif
        } else {
            ble.waitForEvent();
        }
        hmtp.get();    // Triger conversion
    }
}
