/*
 * Mbed Application program / Using Akizuki BLE Module AE-TYBLE16
 *      on-board module : TAIYO YUDEN BLE 4.2 TYSA-B (EYSGJNAWY-WX)
 *
 *  http://akizukidenshi.com/catalog/g/gK-12339/
 *
 *  Refernce document
 *  https://www.yuden.co.jp/wireless_module/document/datareport2/jp/
 *                  TY_BLE_EYSGJNAWY_WX_BriefDataReport_V1_3_20170925J.pdf
 *
 * Copyright (c) 2017 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created:    October   27th, 2017
 *      Revised:    November  19th, 2017
 */

/*
    Tested condition
        mbed-os-5.6.4 & mbed Rev.157(release 155?)
    Nucleo-F446RE   TYBLE16 /also F411RE
        PA_9 (D8)    pin5  / P0.03(UART_RX)
        PA_10(D2)    pin6  / P0.01(UART_TX)
                     pin7  / P0.02(UART_CTS) connected to pin8 / P0.00(UART_RTS)
        PB_3 (D3)    pin15 / P0.23(MODE) Mode indication
        +3.3v        pin14 / +V
        GND          pin13 / GND
 */

/*
 --- HOW TO ESTABLISH CENTRAL AND PERIPHERAL UART ---
 1) Prepare one peripheral module/TYBLE16
 2) Prepare another TYBLE16 module which was programed as "Central mode"
    Please refer "TYBLE16_set_Central_or_ Peripheral" program
 3) Programing "TYBLE16_Uart_Central_and_Peripheral" into Mbed boards 
    (same program into both "Central" and "Peripheral" Mbed board)
 4) If you turn on both the "Central" and "Peripheral" power at the same time,
    the connection is automatically started
 */

//  Include --------------------------------------------------------------------
#include    "mbed.h"
#include    "RingBuffer.h"

//  Definition -----------------------------------------------------------------
#define     TIME_OUT            5      // 5 sec
#define     BF_SIZE_BLE         256
#define     BF_SIZE_PC          256
#define     BF_SIZE_LINE        32

//  Object/ Constructor --------------------------------------------------------
Serial      pc(USBTX,USBRX);
Serial      tyble16(D8, D2);
DigitalIn   tyble16_mode(D3);
Ticker      t;
RingBuffer  ble_bf(BF_SIZE_BLE);
RingBuffer  pc_bf(BF_SIZE_PC);

//  RAM ------------------------------------------------------------------------

//  ROM / Constant data --------------------------------------------------------

//  Function prototypes --------------------------------------------------------
void check_connection(void);
void excute_command(char *);
void ble_rx(void);
void pc_rx(void);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
int main()
{
    static char rcv_bf[BF_SIZE_LINE];
    static uint8_t n = 0;
    char c = 0;

    tyble16.attach(&ble_rx, Serial::RxIrq);     // Set interrupt handler
    pc.attach(&pc_rx, Serial::RxIrq);           // Set interrupt handler
    pc.printf("\r\nUART/TYBLE16(%s, %s UTC)\r\n", __DATE__, __TIME__);
    tyble16.printf("I'm TYBLE16\r\n");
    t.attach(check_connection, 1);
    while(true){
        while(ble_bf.check() != 0){
            //----- receive data -----------------------------------------------
            c = ble_bf.read();          // data received
            pc.putc(c);                 // show to console
            rcv_bf[n++] = c;            // save int buf
            // avoid buffer overflow
            if (n >= (BF_SIZE_LINE - 5)){ n = (BF_SIZE_LINE - 5);}
            if (c == '\n'){             // end one line
                pc.putc('\r');
                rcv_bf[n] = 0;
                if (n >3){  // command ~?\r\n ?=specified one character 
                    if ((rcv_bf[0] == '~') && (rcv_bf[2] == '\r')){
                        excute_command(rcv_bf);
                    }
                }
                n = 0;  // Clear buffer
            }
        }
        while(pc_bf.check() != 0){
            //----- send data --------------------------------------------------
            char c = pc_bf.read();
            tyble16.putc(c);
            pc.putc(c);                 // echo back
            if(c == '\r'){              // if CR then put LF
                tyble16.putc('\n');
                pc.putc('\n');
            }
        }
    }
}

// special command (You can make your own commands)
void excute_command(char *buf)
{
    char c = buf[1];
    switch (c){
        case 'x':   // corrent -> nothing to do
            tyble16.printf("Accept 'x' Command\r\n");
            break;
        case '?':       // Help
            tyble16.printf("Command not available\r\n");
            tyble16.printf("Please set your own command\r\n");
            break;
        default:        // no available command
            tyble16.printf("Command not available\r\n");
            break;
    }
}

// Interrupt routine / Data from BLE
void ble_rx()
{
    ble_bf.save(tyble16.getc());    // BLE RX data save into ring buffer
}

// Interrupt routine / Data from PC
void pc_rx()
{
    pc_bf.save(pc.getc());          // PC RX data save into ring buffer
}

// Every one second, check communication status
void check_connection(void)
{
    static int8_t counter = 0;

    if (tyble16_mode == 0){ // Not connected
        if (++counter >= TIME_OUT){
            // Software reset
            tyble16.puts("BRS\r\n");
            counter = 0;
        }
    } else {
        counter = 0;
    }
}
