/*
 * Mbed Application program / Using Akizuki BLE Module AE-TYBLE16
 *      on-board module : TAIYO YUDEN BLE 4.2 TYSA-B (EYSGJNAWY-WX)
 *
 *  http://akizukidenshi.com/catalog/g/gK-12339/
 *
 *  Refernce document
 *  https://www.yuden.co.jp/wireless_module/document/datareport2/jp/
 *                  TY_BLE_EYSGJNAWY_WX_BriefDataReport_V1_3_20170925J.pdf
 *
 * Copyright (c) 2017 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created:    October   27th, 2017
 *      Revised:    October   30th, 2017
 */

/*
    Tested condition
        mbed-os-5.6.3 & mbed Rev.154
    Nucleo-F446RE   TYBLE16 /also F411RE & F042K6(only mbed & pin assign differ)
        PA_9         pin5 / P0.03(UART_RX)
        PA_10        pin6 / P0.01(UART_TX)
                     pin7 / P0.02(UART_CTS) connected to pin8 / P0.00(UART_RTS)
        +3.3v        pin14 / +V
        GND          pin13 / GND
    Nucleo-F446RE   BME280
        PB_9(SDA)    SDI
        PB_8(SCL)    SDK
                     SDO = GND, CSB = +3.3V
    Communcation with iPhone 7 iOS 11.0.3
        TYs Terminal App
        https://itunes.apple.com/jp/app/tys-terminal-app/id1184640145?mt=8
 */

//  Include --------------------------------------------------------------------
#include "mbed.h"
#include "BME280.h"

//  Definition -----------------------------------------------------------------
#define     TIME_OUT         5       // 5 sec
#define     ADT_TIME        15       // 15 sec

//  Object/ Constructor --------------------------------------------------------
Serial      pc(USBTX,USBRX);
Serial      tyble16(PA_9, PA_10);
BME280      bme280(I2C_SDA, I2C_SCL);

//  RAM ------------------------------------------------------------------------
bool        state_connection    = false;
bool        received_data       = false;
uint32_t    cmd_timeout_cntr    = ADT_TIME;
Ticker      t;

//  ROM / Constant data --------------------------------------------------------

//  Function prototypes --------------------------------------------------------
void check_connection(void);
void parse_input(char *buf);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
// special command for BME280 data collection
bool excute_command(char *buf)
{
    char c = buf[1];
    switch (c){
        case 'a':       // All data
        case 'A':
            tyble16.printf("Pressure: %04.2f hPa\r\n",
                            bme280.getPressure());
            tyble16.printf("Temperature: %2.2f degC\r\n",
                            bme280.getTemperature());
            tyble16.printf("Humidity: %2.2f %%\r\n",
                            bme280.getHumidity());
            break;
        case 'p':       // Pressure
        case 'P':
            tyble16.printf("Pressure: %04.2f hPa\r\n",
                            bme280.getPressure());
            break;
        case 't':       // Temperature
        case 'T':
            tyble16.printf("Temperature: %2.2f degC\r\n",
                            bme280.getTemperature());
            break;
        case 'h':       // Humidity
        case 'H':
            tyble16.printf("Humidity: %2.2f %%\r\n",
                            bme280.getHumidity());
            break;
        case '?':       // Help
            tyble16.printf("Command syntax\r\n");
            tyble16.printf("@a All of data\r\n");
            tyble16.printf("@p Pressure\r\n");
            tyble16.printf("@t Temperature\r\n");
            tyble16.printf("@h Humidity\r\n");
            break;
        default:        // no available command
            tyble16.printf("ommand not available, hit '@?'\r\n");
            break;
    }
    return true;
}

int main()
{
    static char rcv_bf[128];
    static uint8_t n = 0;
    char c = 0;

    pc.printf("\r\n\r\nApplication for AE-TYBLE16 Module ");
    pc.printf(" Peripheral(Sever) side\r\n");
    pc.printf("  created on %s %s\r\n", __DATE__, __TIME__);
    tyble16.printf("I'm TYBLE16 module!\r\n");
    // warming up the sensor
    for (int n = 0; n < 6; n++){
        bme280.getPressure();
        bme280.getTemperature();
        bme280.getHumidity();
    }
    t.attach(check_connection, 1);
    while(true){
        while(tyble16.readable()){
            //----- data receive from Client -----------------------------------
            received_data = true;
            c = tyble16.getc();         // received data from client
            pc.putc(c);                 // show to console
            rcv_bf[n++] = c;            // save int buf
            if (c == '\n'){             // end one line
                pc.putc('\r');
                rcv_bf[n] = 0;
                if (n >3){
                    parse_input(rcv_bf);    // ckeck command or not
                    /*for (int i=0;; i++){
                        if (rcv_bf[i] == 0){break;}
                        pc.printf("0x%x,", rcv_bf[i]);
                    }*/
                    if ((rcv_bf[0] == '@') && (rcv_bf[2] == '\r')){
                        excute_command(rcv_bf); // BME280 data output 
                    }
                }
                n = 0;  // Clear buffer
            }
        }
        while(pc.readable()){
            //----- data send to Client ----------------------------------------
            char c = pc.getc();
            tyble16.putc(c);
            pc.putc(c);                 // echo back
            if(c == '\r'){              // if CR then put LF
                tyble16.putc('\n');
                pc.putc('\n');
            }
        }
    }
}

// Check Response Events
void parse_input(char *buf)
{
    if        (strstr(buf, "CON")){     // Connection successful
        state_connection = true;
    } else if (strstr(buf, "DCO")){     // Disconnect
        state_connection = false; 
    } else if (strstr(buf, "ADT")){     // Advertising Timeout
        state_connection = false;
        cmd_timeout_cntr = 0;
    } else if (strstr(buf, "NAK")){     // Failed
        state_connection = false;
        if (cmd_timeout_cntr == 0){
            // Both Advertising Start. Connectable and Discoverable.
            tyble16.printf("BCD3\r\n");     // send Advertising command
            cmd_timeout_cntr = ADT_TIME;
        }
    }
}

// Every one second, check communication status
void check_connection(void)
{
    static int8_t counter = 0;

    if ((received_data == false) && (state_connection == false)){
        if (++counter >= TIME_OUT){
            tyble16.putc('\n');
            counter = 0;
        }
    } else {
        counter = 0;
    }
    if (cmd_timeout_cntr != 0){
        --cmd_timeout_cntr;
    }
    received_data = false;
}
