/*!
 * @file Adafruit_TSL2561_U.h
 *
 * This is part of Adafruit's FXOS8700 driver for the Arduino platform.  It is
 * designed specifically to work with the Adafruit FXOS8700 breakout:
 * https://www.adafruit.com/products/3463
 *
 * These sensors use I2C to communicate, 2 pins (SCL+SDA) are required
 * to interface with the breakout.
 *
 * Adafruit invests time and resources providing this open source code,
 * please support Adafruit and open-source hardware by purchasing
 * products from Adafruit!
 *
 * Written by Kevin "KTOWN" Townsend for Adafruit Industries.
 *
 * BSD license, all text here must be included in any redistribution.
 *
 */

//------- Modified by ----------------------------------------------------------
//  Kenji Arai / JH1PJL
//  http://www.page.sannet.ne.jp/kenjia/index.html
//  http://mbed.org/users/kenjiArai/
//      Created: Feburary   21st, 2015
//      Revised: August     23rd, 2017
//      Revised: Feburary   20th, 2018   bug fix -> read_ID() & who_am_i()
//                                       Thanks PARK JAICHANG
//      Revised: March      31st, 2018   Added "Auto Range mode"
//                                       Use Adafruit library
//
// Original information     https://www.adafruit.com/product/439
// Original source files    https://github.com/adafruit/TSL2561-Arduino-Library
//                          https://github.com/adafruit/Adafruit_TSL2561
// Change for Mbed platform
//    modified -> all related files
//------------------------------------------------------------------------------

#ifndef TSL2561_H_
#define TSL2561_H_

#include "mbed.h"

// Luminosity sensor, TSL2561
// Address b7=0,b6=1,b5=1,b4=1,b3=0,b2=0,b1=1, b0=R/W
#define TSL2561_ADDRESS_GND       (0x29 << 1)
#define TSL2561_ADDR_LOW          (0x29 << 1)
#define TSL2561_ADDRESS_FLOAT     (0x39 << 1)
#define TSL2561_ADDR_FLOAT        (0x39 << 1)
#define TSL2561_ADDRESS_VDD       (0x49 << 1)
#define TSL2561_ADDR_HIGH         (0x49 << 1)

// Lux calculations differ slightly for CS package
//#define TSL2561_PACKAGE_CS                ///< Chip scale package
#define TSL2561_PACKAGE_T_FN_CL             ///< Dual Flat No-Lead package

// ID
#define I_AM_TSL2561CS             0x01
#define I_AM_TSL2561T_FN_CL        0x05

// COMMAND register bit definition
#define TSL2561_COMMAND_BIT       (0x80)    ///< Must be 1
///< Clears any pending interrupt (write 1 to clear)
#define TSL2561_CLEAR_BIT         (0x40)
///< 1 = read/write word (rather than byte)
#define TSL2561_WORD_BIT          (0x20)
///< 1 = using block read/write
#define TSL2561_BLOCK_BIT         (0x10)

// Control register setting to turn on/off
#define TSL2561_CONTROL_POWERON   (0x03)
#define TSL2561_CONTROL_POWEROFF  (0x00)

#define TSL2561_LUX_LUXSCALE      (14)      ///< Scale by 2^14
#define TSL2561_LUX_RATIOSCALE    (9)       ///< Scale ratio by 2^9
#define TSL2561_LUX_CHSCALE       (10)      ///< Scale channel values by 2^10
#define TSL2561_LUX_CHSCALE_TINT0 (0x7517)  ///< 322/11 * 2^TSL2561_LUX_CHSCALE
#define TSL2561_LUX_CHSCALE_TINT1 (0x0FE7)  ///< 322/81 * 2^TSL2561_LUX_CHSCALE

/** TSL2561 I2C Registers */
enum {
    TSL2561_REGISTER_CONTROL          = 0x00, // Control/power register
    TSL2561_REGISTER_TIMING           = 0x01, // Set integration time register
    TSL2561_REGISTER_THRESHHOLDL_LOW  = 0x02, // Interrupt low threshold low-byte
    TSL2561_REGISTER_THRESHHOLDL_HIGH = 0x03, // Interrupt low threshold high-byte
    TSL2561_REGISTER_THRESHHOLDH_LOW  = 0x04, // Interrupt high threshold low-byte
    TSL2561_REGISTER_THRESHHOLDH_HIGH = 0x05, // Interrupt high threshold high-byte
    TSL2561_REGISTER_INTERRUPT        = 0x06, // Interrupt settings
    TSL2561_REGISTER_CRC              = 0x08, // Factory use only
    TSL2561_REGISTER_ID               = 0x0A, // TSL2561 identification setting
    TSL2561_REGISTER_CHAN0_LOW        = 0x0C, // Light data channel 0, low byte
    TSL2561_REGISTER_CHAN0_HIGH       = 0x0D, // Light data channel 0, high byte
    TSL2561_REGISTER_CHAN1_LOW        = 0x0E, // Light data channel 1, low byte
    TSL2561_REGISTER_CHAN1_HIGH       = 0x0F  // Light data channel 1, high byte
};

/** Three options for how long to integrate readings for */
typedef enum {
    TSL2561_INTEGRATIONTIME_13MS  = 0x00,    // 13.7ms
    TSL2561_INTEGRATIONTIME_101MS = 0x01,    // 101ms
    TSL2561_INTEGRATIONTIME_402MS = 0x02     // 402ms
}
TSL2561IntegrationTime_t;

/** TSL2561 offers 2 gain settings */
typedef enum {
    TSL2561_GAIN_1X               = 0x00,    // No gain
    TSL2561_GAIN_16X              = 0x10,    // 16x gain
}
TSL2561Gain_t;

/** struct sensors_color_s is used to return color data in a common format. */
typedef struct {
    union {
        float c[3];
        /* RGB color space */
        struct {
            float r;       /**< Red component */
            float g;       /**< Green component */
            float b;       /**< Blue component */
        };
    };
    uint32_t rgba;         /**< 24-bit RGBA value */
} sensors_color_t;

#define SENSOR_TYPE_LIGHT       5

/* Sensor details */
/** struct sensor_s is used to describe basic information
        about a specific sensor. */
typedef struct {
    float    max_value; /**< maximum value of this sensor's value in SI units */
    float    min_value; /**< minimum value of this sensor's value in SI units */
    /**< smallest difference between two values reported by this sensor */
    float    resolution;
    /**< min delay in microseconds between events. zero = not a constant rate */
    int32_t  min_delay;
} sensor_t;

/** Interface for Luminosity sensor, TSL2561
 * @code
 * #include "mbed.h"
 * #include "TSL2561.h"
 *
 * // I2C Communication
 *  TSL2561      lum(dp5,dp27);    // TSL2561 SDA, SCL
 * // If you connected I2C line not only this device but also other devices,
 * //     you need to declare following method.
 *  I2C          i2c(dp5,dp27); // SDA, SCL
 *  TSL2561      lum(i2c); // TSL2561 SDA, SCL (Data available every 400mSec)
 *
 * int main() {;
 *   while(true){
 *      printf("Illuminance: %+7.2f [Lux]\r\n", lum.lux());
 *      wait(1.0);
 *   }
 * }
 * @endcode
 */

/**************************************************************************/
/*!
    @brief  Class that stores state and functions
                 for interacting with TSL2561 Light Sensor
*/
/**************************************************************************/
class TSL2561
{
public:
    /** Configure data pin
      * @param data SDA and SCL pins
      */
    TSL2561(PinName p_sda, PinName p_scl);
    TSL2561(PinName p_sda, PinName p_scl, uint8_t addr);

    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      */
    TSL2561(I2C& p_i2c);
    TSL2561(I2C& p_i2c, uint8_t addr);

    /** Get approximates the human eye response
      *  in the commonly used Illuminance unit of Lux
      * @param none
      * @return Lux
      */
    float lux(void);

    /** Get approximates the human eye response with "Auto Range"
      *  in the commonly used Illuminance unit of Lux
      * @param none
      * @return Lux
      */
    float lux_auto(void);

    /** Set timing register
      * @param timing parameter
      * @return timing read data
      */
    uint8_t set_timing_reg(uint8_t parameter);

    /** Read timing register
      * @param timing parameter
      * @return timing read data
      */
    uint8_t read_timing_reg(void);

    /** Set I2C clock frequency
      * @param freq.
      * @return none
      */
    void frequency(int hz);

    /** check Device ID number
      * @param none
      * @return TSL2561 = 1, others  0
      */
    uint8_t who_am_i(void);

    /** Read ID and Revision Number
      * @param none
      * @return ID + REVNO
      */
    uint8_t read_ID(void);

    /** Power Up/Down
      * @param none
      * @return none
      */
    void power_up(void);
    void power_down(void);

    //---------- Adafruit_TSL2561_U.h Original functions part ------------------
    bool init();

    /* TSL2561 Functions */
    void enableAutoRange(bool enable);
    void setIntegrationTime(TSL2561IntegrationTime_t time);
    void setGain(TSL2561Gain_t gain);
    void getLuminosity (uint16_t *broadband, uint16_t *ir);
    uint32_t calculateLux(uint16_t broadband, uint16_t ir);

    /* Unified Sensor API Functions */
    bool getEvent(uint32_t*);
    void getSensor(sensor_t*);

private:
    I2C *_i2c_p;
    I2C &_i2c;
    Timer   t;

    int8_t  _addr;
    bool    _TSL2561Initialised;
    bool    _TSL2561AutoGain;
    TSL2561IntegrationTime_t _TSL2561IntegrationTime;
    TSL2561Gain_t _TSL2561Gain;
    uint32_t ch0;
    uint32_t ch1;
    int8_t   gain;
    uint8_t  id_number;
    float    integ_time;

    uint8_t  dt[4];

    void     enable (void);
    void     disable (void);
    void     write8 (uint8_t reg, uint8_t value);
    uint8_t  read8 (uint8_t reg);
    uint16_t read16 (uint8_t reg);
    void     getData (uint16_t *broadband, uint16_t *ir);
};

#endif // TSL2561_H
