/*
    EXTEND for SPI interface LCD which is using ST7565 controller
        Modified by Kenji Arai / JH1PJL
         http://www7b.biglobe.ne.jp/~kenjia/
         https://os.mbed.com/users/kenjiArai/
            Started: September 20th, 2014
            Revised: November  29th, 2014
            Revised: August     5th, 2020

    original file: C12832.h
    original Library name: C12832
*/

//---------- ORIGINAL Header ---------------------------------------------------
/* mbed library for the mbed Lab Board  128*32 pixel LCD
 * use C12832 controller
 * Copyright (c) 2012 Peter Drescher - DC2PD
 * Released under the MIT License: http://mbed.org/license/mit
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef ST7565_SPI_LCD_H
#define ST7565_SPI_LCD_H

#include "mbed.h"
#include "GraphicsDisplay.h"

/** Bitmap
 */
struct Bitmap {
    int xSize;
    int ySize;
    int Byte_in_Line;
    char* data;
};

/** SPI LCD control library for ST7565 Controller
 *      http://www.ladyada.net/learn/lcd/st7565.html
 *
 *      AD-12864-SPI    http://www.aitendo.com/product/1622
 *      AQM12848A       http://akizukidenshi.com/catalog/g/gK-07007/
 *
 * @code
 * #include "mbed.h"
 *
 * #if 1
 * //           mosi, sck, reset, a0, ncs
 * ST7565   lcd(dp2, dp6, dp10, dp4, dp9, ST7565::AD12864SPI);
 * #else
 * SPI      spi_lcd(dp2, dp1, dp6); // mosi, miso, sck
 * //           spi, reset, a0, ncs
 * ST7565   lcd(spi_lcd, dp10, dp4, dp9, ST7565::AD12864SPI);
 * #endif
 *
 * int main() {
 *   lcd.cls();
 *   lcd.set_contrast(0x06);
 *   lcd.printf("123456789012345678901234567890\r\n");
 *   lcd.rect(10,10,100,50,BLACK);
 *   lcd.circle(10,10,10,BLACK);
 *   lcd.fillcircle(50,40,10,BLACK);
 *   lcd.line(0,0,110,60,BLACK);
 *   while(1){;}
 * }
 * @endcode
 */

class ST7565 : public GraphicsDisplay
{
public:

    /** LCD panel format */
    enum LCDType {
        LCD128x32,
        LCD128x48,
        LCD128x64,
        AQM1248A   = LCD128x48,
        AD12864SPI = LCD128x64,
        ST7565LCD  = LCD128x64
    };

    /**
     * Create a ST7565 object connected to SPI1
     */
    ST7565(PinName mosi, PinName sck,
           PinName reset,
           PinName a0,
           PinName ncs,
           LCDType type,
           const char* name = "LCD"
          );
    ST7565(SPI& _spi,
           PinName reset,
           PinName a0,
           PinName ncs,
           LCDType type,
           const char* name = "LCD"
          );

    /**
     * Get the width of the screen in pixel
     *
     * @returns width of screen in pixel
     *
     */
    virtual int width();

    /**
     * Get the height of the screen in pixel
     *
     * @returns height of screen in pixel
     */
    virtual int height();

    /**
     * Draw a pixel at x,y black or white
     *
     * @param x horizontal position
     * @param y vertical position
     * @param color - 1 set pixel, 0 erase pixel
     */
    virtual void pixel(int x, int y,int colour);

    /**
      * Draw a circle
      *
      * @param x0,y0 center
      * @param r radius
      * @param color - 1 set pixel, 0 erase pixel
      */
    void circle(int x, int y, int r, int colour);

    /**
     * Draw a filled circle
     *
     * @param x0,y0 center
     * @param r radius
     * @param color - 1 set pixel, 0 erase pixel
     *
     * Use circle with different radius,
     * Can miss some pixels
     */
    void fillcircle(int x, int y, int r, int colour);

    /**
     * Draw a 1 pixel line
     *
     * @param x0,y0 start point
     * @param x1,y1 stop point
     * @param color - 1 set pixel, 0 erase pixel
     */
    void line(int x0, int y0, int x1, int y1, int colour);

    /**
     * Draw a rect
     *
     * @param x0,y0 top left corner
     * @param x1,y1 down right corner
     * @param color - 1 set pixel, 0 erase pixel
     */
    void rect(int x0, int y0, int x1, int y1, int colour);

    /**
     * Draw a filled rect
     *
     * @param x0,y0 top left corner
     * @param x1,y1 down right corner
     * @param color - 1 set pixel, 0 erase pixel
     */
    void fillrect(int x0, int y0, int x1, int y1, int colour);

    /**
      * set the contrast level
      */
    void set_contrast(unsigned int o);

    /**
      * read the contrast level
      */
    unsigned int get_contrast(void);

    /**
     * Invert the screen
     *
     * @param o = 0 normal, 1 invert
     */
    void invert(unsigned int o);

    /**
     * Clear the screen
     */
    virtual void cls(void);

    /**
     * Set the drawing mode
     *
     * @param mode NORMAl or XOR
     */
    void setmode(int mode);

    /**
     * Calculate the max number of columns.
     * Depends on actual font size
     *
     * @returns max column
     */
    virtual int columns(void);

    /**
     * calculate the max number of rows
     *
     * @returns max rows
     * depends on actual font size
     *
     */
    virtual int rows(void);

    /**
     * Draw a character on given position out of the active font to the LCD
     *
     * @param x x-position of char (top left)
     * @param y y-position
     * @param c char to print
     */
    virtual void character(int x, int y, int c);

    /**
     * Setup cursor position
     *
     * @param x x-position (top left)
     * @param y y-position
     */
    virtual void locate(int x, int y);

    /**
     * Setup auto update of screen
     *
     * @param up 1 = on , 0 = off
     *
     * if switched off the program has to call copy_to_lcd()
     * to update screen from framebuffer
     */
    void set_auto_up(unsigned int up);

    /**
     * Get status of the auto update function
     *
     * @returns if auto update is on
     */
    unsigned int get_auto_up(void);

    /**
     * Select the font to use
     *
     * @param f pointer to font array
     *
     * font array can created with GLCD Font Creator from http://www.mikroe.com
     * you have to add 4 parameter at the beginning of the font array to use:
     *   - the number of byte / char
     *   - the vertial size in pixel
     *   - the horizontal size in pixel
     *   - the number of byte per vertical line
     *   you also have to change the array to char[]
     */
    void set_font(unsigned char* f);

    /**
     * Print bitmap to buffer
     *
     * @param bm Bitmap in flash
     * @param x  x start
     * @param y  y start
     */
    void print_bm(Bitmap bm, int x, int y);

#if DOXYGEN_ONLY
    /**
     * Write a character to the LCD
     *
     * @param c The character to write to the display
     */
    int putc(int c);

    /**
     * Write a formated string to the LCD
     *
     * @param format A printf-style format string, followed by the
     *               variables to use in formating the string.
     */
    int printf(const char* format, ...);
#endif

    unsigned char* font;
    unsigned int draw_mode;

protected:
    /** Draw color
      */
    enum {WHITE = 0,BLACK};

    /** Draw mode
      * NORMAl
      * XOR set pixel by xor the screen
      */
    enum {NORMAL,XOR};

    /** Vars     */
    SPI *_spi_p;
    SPI &_spi;
    DigitalOut _reset;
    DigitalOut _A0;
    DigitalOut _CS;

    void hline(int x0, int x1, int y, int colour);
    void vline(int y0, int y1, int x, int colour);
    void lcd_reset();
    void wr_dat(unsigned char value);
    void wr_cmd(unsigned char value);
    void wr_cnt(unsigned char cmd);
    void line_clear(int y);
    void copy_to_lcd(void);
    void initialize(LCDType type);

    virtual int _putc(int value);

    unsigned int lcd_width;
    unsigned int lcd_height;
    unsigned int char_x;
    unsigned int char_y;
    unsigned char buffer[1024];
    unsigned int contrast;
    unsigned int auto_up;
};

#endif
