/*
 * Mbed Application program
 *  SD Card file control function with FatFs on Mbed-os6
 *
 * Copyright (c) 2018,'19,'20,'21 Kenji Arai / JH1PJL
 *  http://www7b.biglobe.ne.jp/~kenjia/
 *  https://os.mbed.com/users/kenjiArai/
 *      Created:    April      4th, 2018
 *      Revised:    January   14th, 2021
 */

/*
    Tested board on OS6.6.0
        Nucleo-F446RE
        FRDM-K64F
    Tested board on OS5.15.0(Expected to work on OS6.6.0)
        Nucleo-F401RE, -F411RE, -L152RE, -L073RZ, -L476RG
        nRF52840-DK, nRF52-DK
 */

//  Include --------------------------------------------------------------------
#include "mbed.h"
#include "SDBlockDevice.h"
#include "FATFileSystem.h"
#include "mon.h"
#include "uart_as_stdio.h"
#include <stdlib.h>

//  Definition -----------------------------------------------------------------
#define     USER_SW_ON      0
#define     LOOP_TIME       150      // 150mS

#define DEBUG  0

#if DEBUG
#define DBG(...)    printf(__VA_ARGS__)
#else
#define DBG(...)    {;}
#endif

//using namespace std::chrono;

//  Constructor ----------------------------------------------------------------
#if defined(TARGET_NRF52840_DK) || defined(TARGET_NRF52_DK) ||\
    defined(TARGET_K64F)
DigitalIn       user_sw(BUTTON1, PullUp);
#else
DigitalIn       user_sw(USER_BUTTON);
#endif
#if defined(TARGET_NRF52840_DK) || defined(TARGET_NRF52_DK)
SDBlockDevice   sd(D11, D12, D13, D10, 8000000);
#else
SDBlockDevice   sd(SPI_MOSI, SPI_MISO, SPI_SCK, SPI_CS, 12000000);
#endif
FATFileSystem   fs("fs");
Timer tmr;

//  RAM ------------------------------------------------------------------------

//  ROM / Constant data --------------------------------------------------------
const char *const opening_msg0 = "microSD Card test program";
const char *const opening_msg1 = " -> run on Mbed OS-6\r\n";
const char *const opening_msg2 = "microSD Card is ready for use\r\n";
const char *const opening_msg3 = "Please hit any key to start!\r\n";

//  Function prototypes --------------------------------------------------------
extern void print_revision(void);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
int main()
{
    time_t      seconds;
    uint32_t data0 = 10000U;
    uint32_t data1 = 20000U;
    uint32_t data2 = 30000U;
    uint32_t data3 = 40000U;
    uint32_t data4 = 50000U;
    uint32_t data5 = 60000U;

    printf("\r\n\r\n");
    print_revision();
    DBG("line:%d\r\n", __LINE__);
    /* Init SD CARD reader */
    sd.init();
    fs.mount(&sd);
    FILE* fp = fopen("/fs/mydata.txt", "a");
    if (fp != 0) {
        DBG("line:%d\r\n", __LINE__);
        printf("%s%s",  opening_msg0, opening_msg1);
        fprintf(fp,"%s%s", opening_msg0, opening_msg1);
        printf("%s",  opening_msg2);
        fclose(fp);
    } else {
        printf("ERROR\r\n");
    }
    printf("%s",  opening_msg3);
    while (readable() == 0) { ;}
    char c = getc(); // dummy read
    while (true) {
        DBG("line:%d\r\n", __LINE__);
        tmr.reset();
        tmr.start();
        float size_disk = get_disk_freespace() / (1024.0f * 1024.0f * 1024.0f);
        uint32_t size_file = get_data_file_size("mydata.txt") / 1024;
        printf(", free disk: %.3f GB, file: %u kB, ", size_disk, size_file);
        fp = fopen("/fs/mydata.txt", "a");
        if(fp != 0) {
            char tmp[64];
            DBG("line:%d\r\n", __LINE__);
            seconds = time(NULL);
            strftime(tmp, 64, "DATE %H:%M:%S,%Y/%m/%d,", localtime(&seconds));
            printf("%s", tmp);
            fprintf(fp, "%s", tmp);
            printf("%08d;%08d;%08d;%08d;%08d;%08d\r\n",
                   ++data0, ++data1, ++data2, ++data3, ++data4, ++data5);
            fprintf(fp, "%08d;%08d;%08d;%08d;%08d;%08d\r\n",
                    data0,   data1,   data2,   data3,   data4,   data5);
            fclose(fp);
        } else {
            printf("ERROR\r\n");
        }
        uint32_t time_sd = chrono::duration_cast<chrono::milliseconds>(
                               tmr.elapsed_time()).count();
        printf("time:%3d ", time_sd);
        if (time_sd < LOOP_TIME -2) {
            ThisThread::sleep_for(chrono::milliseconds(LOOP_TIME - time_sd));
        }
        if (user_sw == USER_SW_ON) {
            break;
        }
        if (readable()) {
            mon();
        }
    }
    while(true) {
        mon();
        system_reset();
    }
}
