/*
 * Mbed Application program
 *  SD Card file control function with FatFs on Mbed-os5 or os2
 *
 * Copyright (c) 2018,'19,'20 Kenji Arai / JH1PJL
 *  http://www7b.biglobe.ne.jp/~kenjia/
 *  https://os.mbed.com/users/kenjiArai/
 *      Created:    April      4th, 2018
 *      Revised:    May        2nd, 2020
 */

/*
    Tested board on OS2:
        Nucleo-F401RE, -F411RE, -F446RE, -L073RZ, -L152RE, -L476RG
    Tested board on OS5.15.0
        Nucleo-F401RE, -F411RE, -F446RE, -L152RE, -L073RZ, -L476RG          
        nRF52840-DK, nRF52-DK
        FRDM-K64F
 */

//  Include --------------------------------------------------------------------
#include    "mbed.h"
#if (MBED_MAJOR_VERSION == 2)
#include    "SDFileSystem.h"
#elif (MBED_MAJOR_VERSION == 5)
#include    "SDBlockDevice.h"
#include    "FATFileSystem.h"
#endif
#include    "mon.h"
#include    <stdlib.h>

//  Definition -----------------------------------------------------------------
#define     USER_SW_ON      0
#if (MBED_MAJOR_VERSION == 2)
#define     LOOP_TIME       100     // 100mS
#elif (MBED_MAJOR_VERSION == 5)
#define     LOOP_TIME       50      // 50mS
#endif

#define DEBUG  0

#if DEBUG
#define DBG(...)    pc.printf(__VA_ARGS__)
#else
#define DBG(...)    {;}
#endif

//  Constructor ----------------------------------------------------------------
Serial          pc(USBTX, USBRX, 115200);
#if defined(TARGET_NRF52840_DK) || defined(TARGET_NRF52_DK) ||\
    defined(TARGET_K64F)
DigitalIn       user_sw(BUTTON1, PullUp);
#else
DigitalIn       user_sw(USER_BUTTON);
#endif
#if (MBED_MAJOR_VERSION == 5)
#if defined(TARGET_NRF52840_DK) || defined(TARGET_NRF52_DK)
SDBlockDevice   sd(D11, D12, D13, D10, 8000000);
#else
SDBlockDevice   sd(SPI_MOSI, SPI_MISO, SPI_SCK, SPI_CS, 12000000);
#endif
FATFileSystem   fs("fs");
#endif
Timer tmr;

//  RAM ------------------------------------------------------------------------

//  ROM / Constant data --------------------------------------------------------
const char *const opening_msg0 = "microSD Card test program";
#if (MBED_MAJOR_VERSION == 2)
char *const opening_msg1 = " -> run on Mbed Classic\r\n";
#elif (MBED_MAJOR_VERSION == 5)
const char *const opening_msg1 = " -> run on Mbed OS-5\r\n";
#endif
const char *const opening_msg2 = "microSD Card is ready for use\r\n";
const char *const opening_msg3 = "Please hit any key to start!\r\n";

//  Function prototypes --------------------------------------------------------
extern void print_revision(void);
extern uint32_t get_disk_freespace(void);
extern uint32_t get_data_file_size(const char *const file_name);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
int main()
{
    time_t      seconds;
    uint32_t data0 = 10000U;
    uint32_t data1 = 20000U;
    uint32_t data2 = 30000U;
    uint32_t data3 = 40000U;
    uint32_t data4 = 50000U;
    uint32_t data5 = 60000U;

    pc.printf("\r\n\r\n");
    print_revision();
    DBG("line:%d\r\n", __LINE__);
#if (MBED_MAJOR_VERSION == 5)
    /* Init SD CARD reader */
    sd.init();
    fs.mount(&sd);
#else
    SDFileSystem    sd(D11, D12, D13, D10, "fs");  // do,di,clk,cs
    wait_ms(100);
#endif
    FILE* fp = fopen("/fs/mydata.txt", "a");
    if (fp != 0) {
        DBG("line:%d\r\n", __LINE__);
        pc.printf("%s%s",  opening_msg0, opening_msg1);
        fprintf(fp,"%s%s", opening_msg0, opening_msg1);
        pc.printf("%s",  opening_msg2);
        fclose(fp);
    } else {
        pc.printf("ERROR\r\n");
    }
    pc.printf("%s",  opening_msg3);
    while (pc.readable() == 0) { ;}
    char c = pc.getc(); // dummy read
    while (true) {
        DBG("line:%d\r\n", __LINE__);
        tmr.reset();
        tmr.start();
        uint32_t size_disk = get_disk_freespace();
        uint32_t size_file = get_data_file_size("mydata.txt");
        pc.printf("free disk:%10u, file:%8u  ", size_disk, size_file);
        fp = fopen("/fs/mydata.txt", "a");
        if(fp != 0) {
            char tmp[64];
            DBG("line:%d\r\n", __LINE__);
            seconds = time(NULL);
            strftime(tmp, 64, "DATE %H:%M:%S,%Y/%m/%d,", localtime(&seconds));
            pc.printf("%s", tmp);
            fprintf(fp, "%s", tmp);
            pc.printf("%08d;%08d;%08d;%08d;%08d;%08d\r\n",
                      ++data0, ++data1, ++data2, ++data3, ++data4, ++data5);
            fprintf(fp, "%08d;%08d;%08d;%08d;%08d;%08d\r\n",
                        data0,   data1,   data2,   data3,   data4,   data5);
            fclose(fp);
        } else {
            pc.printf("ERROR\r\n");
        }
        uint32_t time_sd = tmr.read_ms();
        pc.printf("time:%3d ", time_sd);
#if (MBED_MAJOR_VERSION == 2)
        if (time_sd < LOOP_TIME -2){
            wait_ms(LOOP_TIME - time_sd);
        }
#elif (MBED_MAJOR_VERSION == 5)
        if (time_sd < LOOP_TIME -2){
            ThisThread::sleep_for(LOOP_TIME - time_sd);
        }
#endif
        if (user_sw == USER_SW_ON) {
            break;
        }
        if (pc.readable()) {
            mon();
        }
    }
    while(true) {
        mon();
        system_reset();
    }
}
