/*
 * mbed library program
 *  S11059 I2C interface-compatible color sensor by Hamamatsu Photonics K.K.
 *
 * Copyright (c) 2018 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Started:    December   1st, 2018
 *      Revised:    December   2nd, 2018
 */
/*
 *---------------- REFERENCE ---------------------------------------------------
 * Original Information
 *  https://www.hamamatsu.com/jp/en/product/type/S11059-03DS/index.html
 *  https://www.hamamatsu.com/jp/ja/product/type/S11059-03DS/index.html
 * Aplied Board
 *      AE-S11059 Module
 *          http://akizukidenshi.com/catalog/g/gK-08316/
 */

#ifndef S11059_H
#define S11059_H

#include "mbed.h"

//  S11059
//  Only one address
#define S11059_ADDR      (0x2a << 1)

////////////// PARAMETER DEFINITION ///////////////////////
typedef struct {
    
} S11059_paramtr_TypeDef;

////////////// DATA TYPE DEFINITION ///////////////////////
typedef struct {
    uint16_t red;
    uint16_t green;
    uint16_t blue;
    uint16_t infrared;
} S11059_Data_TypeDef;


/** S11059 Color sensor by Hamamatsu Photonics K.K.
 * @code
 * #include    "mbed.h"
 * #include    "S11059.h"
 *
 * Serial pc(USBTX,USBRX);
 * I2C    (I2C_SDA, I2C_SCL);
 * S11059 lux(i2c);
 *
 * int main() {
 *     S11059_data_TypeDef  dt;
 *
 *     while(1) {
 *         lux.read(&dt);
 *         pc.printf(": r=%6d, g=%6d, b=%6d, ir=%6d \r\n",
 *                    dt.red, dt.green, dt.blue, dt.infrared);
 *         wait(0.5f);
 *     }
 * }
 * @endcode
 */

class S11059
{
public:
    /** Configure data pin
      * @param data SDA and SCL pins
      */
    S11059(PinName p_sda, PinName p_scl);

    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      */
    S11059(I2C& p_i2c);

    /** Get color sensor data
     * @param data saving address
     */
    void get_data(S11059_Data_TypeDef *dt);

    /** Get approximates the human eye response
      *  in the commonly used Illuminance unit of Lux
      * @param none
      * @return Lux
      */
    float lux(void);

    /** Reset chip
     * @param none
     */
    void reset_chip(void);

    /** Set gain X1 (Default mode)
     * @param none
     */
    void set_gain_one(void);

    /** Set gain X10
     * @param none
     */
    void set_gain_ten(void);

    /** Set fixed period mode (Default mode)
     * @param none
     */
    void set_fixed_mode(void);

    /** Set manual mode with constant number (*N)
     * @param 0 to 0xffff (integration time * parameter)
     */
    void set_manual_mode(uint16_t num);

    /** Set I2C clock frequency
      * @param freq.
      * @return none
      */
    void frequency(int hz);

    /** Read register
      * @param register's address
      * @return register data
      */
    uint8_t read_reg(uint8_t addr);

    /** Write register
      * @param register's address
      * @param data
      * @return register data
      */
    uint8_t write_reg(uint8_t addr, uint8_t data);

protected:
    void initialize(void);

    I2C *_i2c_p;
    I2C &_i2c;

private:
    char    dt[16];     // working buffer
    uint8_t chip_addr;
    uint8_t gain;       // gain x1= 1, x10=10
    bool    mode;       // ture = fixed priode mode, false = manual mode
    float   lum;
};

#endif      // S11059_H
