/*
 * mbed library program
 *  Control RX-8025NB Real Time Clock Module
 *  EPSON
 *
 * Copyright (c) 2015,'16,'17,'19,'20 Kenji Arai / JH1PJL
 *  http://www7b.biglobe.ne.jp/~kenjia/
 *  https://os.mbed.com/users/kenjiArai/
 *      Created: June        3rd, 2015
 *      Revised: August      7th, 2020
 */

/*
 *---------------- REFERENCE ---------------------------------------------------
 * Original Information
 *  http://www5.epsondevice.com/ja/products/rtc/rx8025nb.html
 * Akizuki
 *  http://akizukidenshi.com/catalog/g/gK-08585/
 */

#ifndef RX8025_H
#define RX8025_H

#include "mbed.h"

// RTC EPSON RX8025
//  7bit address = 0b0110010(No other choice)
#define RX8025ADDR  (0x32 << 1)

// Register definition
#define RX8025_REG_SEC         0x0
#define RX8025_REG_MIN         0x1
#define RX8025_REG_HOUR        0x2
#define RX8025_REG_WDAY        0x3
#define RX8025_REG_DAY         0x4
#define RX8025_REG_MON         0x5
#define RX8025_REG_YEAR        0x6
#define RX8025_REG_OFFSET      0x7
#define RX8025_REG_ALARMW_MIN  0x8
#define RX8025_REG_ALARMW_HOUR 0x9
#define RX8025_REG_ALARMW_WDAY 0xa
#define RX8025_REG_ALARMD_MIN  0xb
#define RX8025_REG_ALARMD_HOUR 0xc
#define RX8025_REG_RESERVED    0xd
#define RX8025_REG_CONTL1      0xe
#define RX8025_REG_CONTL2      0xf

// Buffer size
#define RTC_BUF_SIZ            (RX8025_REG_CONTL2 + 5)

typedef enum {
    RTC_SQW_NONE = 0,
    RTC_SQW_32KHZ
} sq_wave_t;

/** Interface for RTC (I2C Interface)  EPSON RX8025
 *
 *  Standalone type RTC via I2C interface
 *
 * @code
 * #include "mbed.h"
 * #include "RX8025NB.h"
 *
 * // I2C Communication
 *  RX8025     RX8025(dp5,dp27); // RTC(RX8025) SDA, SCL (Fixed address)
 * // If you connected I2C line not only this device but also other devices,
 * //     you need to declare following method.
 *  I2C        i2c(dp5,dp27);    // SDA, SCL
 *  RX8025     RX8025(i2c);      // RTC(RX8025) (Fixed address)
 *
 * int main() {
 * tm t;
 * time_t seconds;
 * char buf[40];
 *
 *   while(1){
 *      RX8025.get_time_rtc(&t);   // read RTC data
 *      seconds = mktime(&t);
 *      strftime(buf, 40, "%I:%M:%S %p (%Y/%m/%d)", localtime(&seconds));
 *      printf("Date: %s\r\n", buf);
 *   }
 * }
 * @endcode
 */

class RX8025
{
public:

    typedef struct {    // BCD format
        uint8_t rtc_seconds;
        uint8_t rtc_minutes;
        uint8_t rtc_hours;
        uint8_t rtc_weekday;
        uint8_t rtc_date;
        uint8_t rtc_month;
        uint8_t rtc_year_raw;
        uint16_t rtc_year;
    } rtc_time;

    /** Configure data pin (with other devices on I2C line)
      * @param data SDA and SCL pins
      */
    RX8025(PinName p_sda, PinName p_scl);

    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      */
    RX8025(I2C& p_i2c);

    /** Read RTC data with Standard C "struct tm" format
      * @param tm (data save area)
      * @return none but all data in tm
      */
    void read_rtc_std(tm *);
    void get_time_rtc(tm *);

    /** Write data to RTC data with Standard C "struct tm" format
      * @param tm (save writing data)
      * @return none but all data in tm
      */
    void write_rtc_std(tm *);
    void set_time_rtc(tm *);

    /** Set Alarm-D / INTA time
      * @param next time (unit: minutes) from now on minimum = 2 minuteｓ!!
      * @return none
      */
    void set_next_alarmD_INTA(uint16_t);
    void set_next_IRQ(uint16_t);

    /** Clear Alarm-D / INTA interrupt
      * @param none
      * @return none
      */
    void clear_alarmD_INTA(void);
    void clear_IRQ(void);

    /** Read one byte from specific register
      * @param register address
      * @return register data
      */
    uint8_t read_reg_byte(uint8_t);

    /** Write one byte into specific register
      * @param register address, data
      * @return register saved data
      */
    uint8_t write_reg_byte(uint8_t, uint8_t);

    /** Read RTC data with own format
      * @param tm (data save area)
      * @return none but all data in tm
      */
    void read_rtc_direct(rtc_time *);

    /** Read RTC data with own format
      * @param tm (save writing data)
      * @return none but all data in tm
      */
    void write_rtc_direct(rtc_time *);

    /** Set I2C clock frequency
      * @param freq.
      * @return none
      */
    void frequency(int);

    /** Control FOUT output port (FOE pin must high when 32.768kHz output)
      * @param output_mode 
      * @return none
      */
    void set_sq_wave(sq_wave_t);

protected:
    I2C *_i2c_p;
    I2C &_i2c;

    uint8_t bin2bcd(uint8_t);
    uint8_t bcd2bin(uint8_t);
    void set_alarmD_reg (uint16_t);
    void init(void);

private:
    uint8_t RX8025_addr;
    uint8_t rtc_buf[RTC_BUF_SIZ];   // buffer for RTC

};

#endif      // RX8025_H
