/*
 * Mbed Application program
 *  RAM Disck function with FatFs on Mbed-os5
 *
 * Copyright (c) 2018,'19,'20 Kenji Arai / JH1PJL
 *  http://www7b.biglobe.ne.jp/~kenjia/
 *  https://os.mbed.com/users/kenjiArai/
 *      Created:    April      7th, 2018
 *      Revised:    May        2nd, 2020
 */

//  Include --------------------------------------------------------------------
#include    "mbed.h"
#include    "FATFileSystem.h"
#include    "HeapBlockDevice.h"
#include    "mon.h"
#include    <stdlib.h>
#include    <stdio.h>
#include    <errno.h>

//  Definition -----------------------------------------------------------------
#if defined(TARGET_DISCO_F769NI) || defined(TARGET_DISCO_F746NG)
#define     USER_SW_ON      1
#else
#define     USER_SW_ON      0
#endif

#define DISK_SIZE_LIMIT     (5 * 512)
#define RAM_DISK_SIZE_4MB   (8192 * 512)
#define RAM_DISK_SIZE_2MB   (4096 * 512)
#define RAM_DISK_SIZE_1MB   (2048 * 512)
#define RAM_DISK_SIZE_512KB (1024 * 512)
#define RAM_DISK_SIZE_128KB (256 * 512)
#define RAM_DISK_SIZE_64KB  (128 * 512)

#if defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F411RE) ||\
    defined(TARGET_NUCLEO_F446RE) || defined(TARGET_NUCLEO_L476RG) ||\
    defined(TARGET_NUCLEO_L152RE)
#define RAM_DISK_SIZE       RAM_DISK_SIZE_64KB
#elif defined(TARGET_K64F) || defined(TARGET_NRF52840_DK) ||\
      defined(TARGET_DISCO_F746NG) || defined(TARGET_DISCO_F469NI)
    #define RAM_DISK_SIZE       RAM_DISK_SIZE_128KB
#elif defined(TARGET_GR_LYCHEE)
    #define RAM_DISK_SIZE       RAM_DISK_SIZE_512KB
#elif defined(TARGET_RZ_A1H)
    #define RAM_DISK_SIZE       RAM_DISK_SIZE_2MB
#else
    #warning "make sure disk size for your board!" 
    #define RAM_DISK_SIZE       RAM_DISK_SIZE_64KB
#endif

#define DEBUG  0
#if DEBUG
#define DBG(...)    pc.printf(__VA_ARGS__)
#else
#define DBG(...)    {;}
#endif

//  Constructor ----------------------------------------------------------------
DigitalOut      led(LED1);
DigitalIn       user_sw(BUTTON1, PullUp);
Serial          pc(USBTX, USBRX, 115200);
HeapBlockDevice bd(RAM_DISK_SIZE, 512);
FATFileSystem   fs("fs");
Timer tmr;

//  RAM ------------------------------------------------------------------------

//  ROM / Constant data --------------------------------------------------------
const char *const opening_msg0 = "RAM Disk (use Heap area) test program";
const char *const opening_msg1 = " -> run on Mbed OS-5\r\n";
const char *const opening_msg2 = "microSD Card is ready for use\r\n";
const char *const opening_msg3 = "Please hit any key to start!\r\n";

//  Function prototypes --------------------------------------------------------
void return_error (int ret_val);
void errno_error (void* ret_val);
extern void print_revision(void);
extern uint32_t get_disk_freespace(void);
extern uint32_t get_data_file_size(const char *const file_name);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
int main()
{
    time_t      seconds;
    uint32_t data0 = 10000U;
    uint32_t data1 = 20000U;
    uint32_t data2 = 30000U;
    uint32_t data3 = 40000U;
    uint32_t data4 = 50000U;
    uint32_t data5 = 60000U;

    pc.printf("\r\n\r\n");
    print_revision();
    DBG("line:%d\r\n", __LINE__);
    pc.printf("\r\nStart\r\n");
    int error = 0;
    pc.printf("Welcome to the filesystem example.\r\n");
    pc.printf("Formatting a FAT, RAM-backed filesystem.\r\n");
    error = FATFileSystem::format(&bd);
    return_error(error);
    pc.printf("Mounting the filesystem on \"/fs\". \r\n");
    error = fs.mount(&bd);
    return_error(error);

    FILE* fp = fopen("/fs/mydata.txt", "a");
    errno_error(fp);
    if (fp != 0) {
        pc.printf("%s%s",  opening_msg0, opening_msg1);
        fprintf(fp,"%s%s", opening_msg0, opening_msg1);
        pc.printf("%s",  opening_msg2);
        fclose(fp);
    } else {
        pc.printf("ERROR\r\n");
    }
    pc.printf("%s",  opening_msg3);
    while (pc.readable() == 0) { ;}
    char c = pc.getc(); // dummy read
    while (true) {
        DBG("line:%3d\r\n", __LINE__);
        tmr.reset();
        tmr.start();
        uint32_t size_disk = get_disk_freespace();
        uint32_t size_file = get_data_file_size("mydata.txt");
        pc.printf("free disk:%8u, file:%8u  ", size_disk, size_file);
        fp = fopen("/fs/mydata.txt", "a");
        if (size_disk <= DISK_SIZE_LIMIT) {
            pc.printf("Reached RAM Disk size limitation!!\r\n");
            break;
        }
        DBG("line:%3d\r\n", __LINE__);
        if(fp != 0) {
            char tmp[64];
            DBG("line:%d\r\n", __LINE__);
            seconds = time(NULL);
            strftime(tmp, 64, "DATE %H:%M:%S,%Y/%m/%d,", localtime(&seconds));
            pc.printf("%s", tmp);
            fprintf(fp, "%s", tmp);
            pc.printf("%08d;%08d;%08d;%08d;%08d;%08d\r\n",
                      ++data0, ++data1, ++data2, ++data3, ++data4, ++data5);
            fprintf(fp, "%08d;%08d;%08d;%08d;%08d;%08d\r\n",
                        data0,   data1,   data2,   data3,   data4,   data5);
            fclose(fp);
        } else {
            pc.printf("ERROR\r\n");
        }
        uint32_t time_sd = tmr.read_ms();
        pc.printf("time[ms]:%3d ,", time_sd);
        ThisThread::sleep_for(20);
        if (user_sw == USER_SW_ON) {
            break;
        }
        if (pc.readable()) {
            mon();
        }
        led = !led;
    }
    while(true) {
        mon();
        system_reset();
    }
}

void return_error (int ret_val)
{
    if (ret_val) {
        pc.printf("retrun error/Failure. %d\r\n", ret_val);
    }
}

void errno_error (void* ret_val)
{
    if (ret_val == NULL) {
        pc.printf("error #/Failure. %d \r\n", errno);
    }
}
