/*
 * mbed Application program for the mbed
 *      RTOS Test program for Nucleo L152RE
 *
 * Copyright (c) 2014,'15 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: November  29th, 2014
 *      Revised: May       20th, 2015
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

//  Include ---------------------------------------------------------------------------------------
#include "mbed.h"
#include "rtos.h"
#include "SetRTC.h"     // Own lib. / RTC control

//  Definition ------------------------------------------------------------------------------------
#define DO_DEBUG        1

#if DO_DEBUG
#define DEBUG_LINE      printf("line:%d\r\n", __LINE__);
#else
#define DEBUG_LINE      {;}
#endif
 
// Com
#if 1
#define BAUD(x)         pc.baud(x)
#define GETC(x)         pc.getc(x)
#define PUTC(x)         pc.putc(x)
#define PUTS(x)         pc.puts(x)
#define PRINTF(...)     pc.printf(__VA_ARGS__)
#define READABLE(x)     pc.readable(x)
#else
#define BAUD(x)         {;}
#define GETC(x)         {;}
#define PUTC(x)         {;}
#define PUTS(x)         {;}
#define PRINTF(...)     {;}
#define READABLE(x)     {;}
#endif

#define TIME_BASE_S     0.02

#define step_one        (0.0625f)

//  Object ----------------------------------------------------------------------------------------
// LED's !! DON'T USE LED1(PA_5) because PA_5 uses for DAC
DigitalOut LEDs[3] = {
    DigitalOut(PC_10), DigitalOut(PC_11), DigitalOut(PC_12)
};
// OS check
DigitalOut  task0(PC_2);
DigitalOut  task1(PC_3);
// Swiches
DigitalIn   usr_sw(PC_13);
// DAC to ADC
#if defined(TARGET_NUCLEO_L152RE)
AnalogOut   dac0(PA_4);
AnalogOut   dac1(PA_5);
#endif
AnalogIn    adc1(A1);
AnalogIn    adc3(A3);
// I2C
I2C         i2c1(D14,D15);      // SDA, SCL
// com
Serial      pc(USBTX, USBRX);   // Communication with Host
// Mutex
Mutex       uart_mutex; 
Mutex       i2c_mutex; 

//  RAM -------------------------------------------------------------------------------------------
Queue<uint32_t, 2> queue0;

float fg[3];
float fa[3];

float analog[2];

/* Mail */
typedef struct {
  float    voltage; /* AD result of measured voltage */
  float    current; /* AD result of measured current */
  uint32_t counter; /* A counter value               */
} mail_t;
 
Mail<mail_t, 16> mail_box;

uint8_t show_flag0;
uint8_t show_flag1;
uint32_t count;

//  ROM / Constant data ---------------------------------------------------------------------------
const float out_data[16] = {step_one * 15, step_one * 14, step_one * 13, step_one * 12, step_one * 11,
                    step_one * 10, step_one *  9, step_one *  8, step_one *  7, step_one *  6,
                    step_one *  5, step_one *  4, step_one *  3, step_one *  2, step_one *  1, 0};
            
//  Function prototypes ---------------------------------------------------------------------------

//  Function prototypes ---------------------------------------------------------------------------
extern int mon( void);

//-------------------------------------------------------------------------------------------------
//  Control Program
//-------------------------------------------------------------------------------------------------
void send_thread (void const *args) {
    uint32_t i = 0;
    while (true) {
        i++; // fake data update
        mail_t *mail = mail_box.alloc();
        mail->voltage = (i * 0.1) * 33; 
        mail->current = (i * 0.1) * 11;
        mail->counter = i;
        mail_box.put(mail);
        Thread::wait(1000);
    }
}

void blink(void const *n) {
    LEDs[(int)n] = !LEDs[(int)n];
}

// Read switch status
int read_sw(){
    if (usr_sw.read()){
        return 0;
    } else {
        return 1;
    }
}

// Monitor program
void monitor(void const *args){
    Thread::wait(100);
    while (true){
DEBUG_LINE
        mon();
    }
}

void watch_time (void const *args) {
    time_t seconds;
    char buf[64];

    while (true) {
        seconds = time(NULL);
        strftime(buf, 40, "%B %d,'%y, %H:%M:%S", localtime(&seconds));
        if (show_flag1){
//            printf("[TASK1] %s, No:%5d, Ticker:%10u\r\n",buf, i++,  us_ticker_read());
        }
        Thread::wait(1000);
    }
}

// Interrupt routine
void queue_isr0() {
    queue0.put((uint32_t*)1);
}

// Update sensor data
void update_sensor(void const *args){ // No need to connect I2C line
    char cmd[2];
    uint8_t i = 0;

DEBUG_LINE
    osDelay(1000);
    i2c1.frequency(50000);
    while (true) {
        osEvent evt = queue0.get();
        // ---->lock
        i2c_mutex.lock();
        cmd[0] = 0x55;
        cmd[1] = 0xaa;
        i2c1.write(0xa4, cmd, 2);
        if (i2c1.read(0xa4, cmd, 2)){
            fa[0] = fa[1] = fa[2] = 1.11f + (++i / 100);
            fg[0] = fg[1] = fg[2] = 2.22f + (++i / 100);
        } else {
            fa[0] = fa[1] = fa[2] = -0.01f - (++i / 100);
            fg[0] = fg[1] = fg[2] =  0.01f + (++i / 100);
        }
        // <----unlock
        i2c_mutex.unlock();
        // debug
        task0 = !task0;
        osDelay(1000);
    }
}

// Update sensor data
void adc(void const *args){
    while (true) {
#if defined(TARGET_NUCLEO_L152RE)
        analog[0] = adc1.read();
        analog[1] = adc3.read();
#else
        analog[0] = 1.2f;
        analog[1] = 3.4f;
#endif
        osDelay(500);   // miliseconds
        task1 = !task1;
    }
}

// Update sensor data
void pwm_and_dac(void const *args){
    uint8_t i = 0;
    while (true) {
#if defined(TARGET_NUCLEO_L152RE)
        dac0.write(out_data[i & 0x0f]);
        dac1.write(out_data[i & 0x0f]);
#endif
        osDelay(500);   // miliseconds
        i++;
    }
}

// Thread definition
osThreadDef(update_sensor, osPriorityNormal, 1024);
osThreadDef(monitor, osPriorityNormal, 1024);
osThreadDef(watch_time, osPriorityNormal, 1024);
osThreadDef(adc, osPriorityAboveNormal, 1024);
osThreadDef(pwm_and_dac, osPriorityAboveNormal, 1024);

int main(void) {
    osDelay(1000);  // wait 1sec
//    PUTS("\x1b[2J\x1b[H");
    PUTS("\r\nCreated on "__DATE__ " " __TIME__ " (UTC)\r\n""\r\n");
DEBUG_LINE
    if (SetRTC(0) == 1) {
        PRINTF("Use External CLK (Good for RTC)\r\n");
    } else {
        PRINTF("Use Internal CLK (Bad for RTC)\r\n");
    }
DEBUG_LINE
    time_enter_mode();
DEBUG_LINE
    RtosTimer led_1_timer(blink, osTimerPeriodic, (void *)0);
    RtosTimer led_2_timer(blink, osTimerPeriodic, (void *)1);
    RtosTimer led_3_timer(blink, osTimerPeriodic, (void *)2);
DEBUG_LINE   
    led_1_timer.start(2000);
    led_2_timer.start(1000);
    led_3_timer.start(500);
DEBUG_LINE     
    Thread thread(send_thread);
DEBUG_LINE
    // IRQ
    Ticker ticker0;
    Ticker ticker1;
    ticker0.attach(queue_isr0, TIME_BASE_S);
DEBUG_LINE
#if 0
/// Create a thread and add it to Active Threads and set it to state READY
osThreadId osThreadCreate (const osThreadDef_t *thread_def, void *argument) {
  if (__exceptional_mode()) return NULL;           // Not allowed in ISR
  if ((__get_mode() != MODE_USR) && (os_running == 0)) {
    // Privileged and not running
    return   svcThreadCreate(thread_def, argument);
  } else {
    return __svcThreadCreate(thread_def, argument);
  }
}
#endif
    // Starts threads
    if (osThreadCreate(osThread(monitor), NULL) == NULL){
        PRINTF("ERROR/monitor() thread\r\n");
    }
DEBUG_LINE
    if (osThreadCreate(osThread(update_sensor), NULL) == NULL){
        PRINTF("ERROR/update_sensor() thread\r\n");
    }
DEBUG_LINE
    if (osThreadCreate(osThread(watch_time), NULL) == NULL){
        printf("ERROR/watch_time() thread\r\n");
    }
DEBUG_LINE
    if (osThreadCreate(osThread(adc), NULL) == NULL){
        PRINTF("ERROR/print4com() therad\r\n");
    }
DEBUG_LINE
    if (osThreadCreate(osThread(pwm_and_dac), NULL) == NULL){
        PRINTF("ERROR/print4com() therad\r\n");
    }
DEBUG_LINE
    count = 0;
DEBUG_LINE
    while (true) {
        osEvent evt = mail_box.get();
//DEBUG_LINE
        if (evt.status == osEventMail) {
            mail_t *mail = (mail_t*)evt.value.p;
            if (show_flag0){
                PRINTF("[MAIN]\r\n");
                PRINTF("This is dummy!, ");
                PRINTF("Volt: %.2f V, "   , mail->voltage);
                PRINTF("Current: %.2f A, "     , mail->current);
                PRINTF("# of cycles: %u\r\n", mail->counter);
            }
            mail_box.free(mail);
        }
    }
}

// rtos error routine
void mbed_die(void) {
    PRINTF("Error, came from os_error()!\r\n");
    gpio_t led_1; gpio_init_out(&led_1, LED1);
    gpio_t led_2; gpio_init_out(&led_2, PC_10);
    gpio_t led_3; gpio_init_out(&led_3, PC_11);
    gpio_t led_4; gpio_init_out(&led_4, PC_12);

    while (1) {
        gpio_write(&led_1, 1);
        gpio_write(&led_2, 0);
        gpio_write(&led_3, 0);
        gpio_write(&led_4, 1);
        wait_ms(100);
        gpio_write(&led_1, 0);
        gpio_write(&led_2, 1);
        gpio_write(&led_3, 1);
        gpio_write(&led_4, 0);
        wait_ms(100);
    }
}
