/*
 * mbed Application program for the mbed ST Nucleo L152RE & L401RE
 *      Stepping Motor
 *
 * Copyright (c) 2014 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: July      12th, 2014
 *      Revised: August    24th, 2014
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
//
//  Stepping Motor SPG27-1702 COPAL ELECTRONICS
//      http://akizukidenshi.com/catalog/g/gP-05708/
//  Drive voltage: 5V       Driver IC: TD62003AP
//  
//  CAUTION: This is only for Unipolar Type Stepping Motor!
//           Cannot use for Bipolar Type
//      Plese refer http://en.wikipedia.org/wiki/Stepper_motor
//

//  Include ---------------------------------------------------------------------------------------
#include    "mbed.h"
#include    "stepper.h"

//  Object ----------------------------------------------------------------------------------------
#if defined(TARGET_LPC1114)
Serial      pc(dp16,dp15);      // Communication with Host
STEPPER     sm_r(dp1,  dp2, dp4, dp6);
STEPPER     sm_l(dp17,dp18,dp25,dp26);
#elif defined(TARGET_NUCLEO_L152RE) || defined(TARGET_STM32F401RE)
Serial      pc(SERIAL_TX, SERIAL_RX);  // Communication with Host
STEPPER     sm_r(D5, D4, D3, D2);
STEPPER     sm_l(D9, D8, D7, D6);
#endif

//  Definition ------------------------------------------------------------------------------------
#define TIMEBASE                18000

#define FIXED_STEPS             480

#define BAUD(x)                 pc.baud(x)
#define GETC(x)                 pc.getc(x)
#define PUTC(x)                 pc.putc(x)
#define PRINTF(...)             pc.printf(__VA_ARGS__)
#define READABLE(x)             pc.readable(x)

//  RAM -------------------------------------------------------------------------------------------
char lb[32];

uint8_t pls_width[MT_SLOP_STEP] = {5, 4, 3, 2, 1, 1, 1, 1, 1, 1 };

//  ROM / Constant data ---------------------------------------------------------------------------
static char *const msg0 = "\r\nStepping Motor Control by JH1PJL, created on "__DATE__"\r\n";

//  Function prototypes ---------------------------------------------------------------------------
void help(void);
void get_line (char *buff, int len);
int xatoi (char **str, int32_t *res);

//-------------------------------------------------------------------------------------------------
//  Control Program
//-------------------------------------------------------------------------------------------------
// Main control -----------------------------------------------------------------------------------
int main() {
int32_t p1;
uint8_t cnt = 0;
char *ptr;
char c;

    sm_r.set_max_speed(TIMEBASE);
    sm_l.set_max_speed(TIMEBASE);
    PRINTF(msg0);
    for (;;) {
        PUTC('>');
        ptr = lb;
        get_line(ptr, sizeof(lb));
        switch (*ptr++){
            case 'b' :
                if (xatoi(&ptr, &p1)) {
                    sm_r.move(p1);
                    sm_l.move(p1);
                } 
                PRINTF("\rR turn: %\+-d\r\n", p1);
                PRINTF("L turn: %\+-d\r\n", p1);
                break;
            case 't' :
                if (xatoi(&ptr, &p1)) {
                    sm_r.move(p1);
                    sm_l.move(-p1);
                }
                PRINTF("\rR turn: %\+-d\r\n", p1);
                PRINTF("L turn: %\+-d\r\n", -p1);
                break;
            case 'r' :
                if (xatoi(&ptr, &p1)) {
                    sm_r.move(p1);
                }
                PRINTF("\rR turn: %\+-d\r\n", p1);
                break;
            case 'l' :
                if (xatoi(&ptr, &p1)) {
                    sm_l.move(p1);
                }
                PRINTF("\rL turn: %\+-d\r\n", p1);
                break;
            case 's' :
                PRINTF("\rR ");
                if (sm_r.status()){
                    PRINTF("running");
                } else {
                    PRINTF("stooped");
                }
                PRINTF("\r\nL ");
                if (sm_l.status()){
                    PRINTF("running");
                } else {
                    PRINTF("stopped");
                }
                PRINTF("\r\n");
                break;
            case 0x0d :
                if (cnt & 1){       // cnt = 1 or 3
                    if (cnt == 3){  // cnt = 3
                        sm_l.move(-FIXED_STEPS);
                        c = '-';
                        cnt = 0;
                    } else {        // cnt = 1
                        sm_l.move(FIXED_STEPS);
                        c = '+';
                        cnt++;
                    }
                    PRINTF("\rL turn: %c%d\r\n", c, FIXED_STEPS);
                } else {            // cnt = 0 or 2
                    if (cnt == 2){  // cnt = 2
                        sm_r.move(-FIXED_STEPS);
                        c = '-';
                    } else {        // cnt = 0
                        sm_r.move(FIXED_STEPS);
                        c = '+';
                    }
                    cnt++;
                    PRINTF("\rR turn: %c%d\r\n", c, FIXED_STEPS);
                }
                break;         
            case '?' :
                help();
                break;
            default:
                PRINTF("\r?\r\n");
                help();
        }
    }
}

void help(void){
    PRINTF("\rRight motor:  r +/-step  e.g. r +100\r\n");
    PRINTF("Left  motor:  l +/-step  e.g. l -100\r\n");
    PRINTF("Turn  motors: t +/-step  e.g. t  100\r\n");
    PRINTF("Both  motors: b +/-step  e.g. b -100\r\n");
    PRINTF("Check Status: s\r\n");
    PRINTF("fixed data & reverse hit <Enter> key\r\n");
}

// Com line control -------------------------------------------------------------------------------
//  Change string -> number
int xatoi (char **str, int32_t *res){
unsigned long val;
unsigned char c, radix, s = 0;

    while ((c = **str) == ' '){ (*str)++;}
    if (c == '-') {
        s = 1;
        c = *(++(*str));
    } else if (c == '+') {
        s = 0;
        c = *(++(*str));
    }
    if (c == '0') {
        c = *(++(*str));
        if (c <= ' ') { *res = 0;   return 1; }
        if (c == 'x') {
            radix = 16;
            c = *(++(*str));
        } else {
            if (c == 'b') {
                radix = 2;
                c = *(++(*str));
            } else {
                if ((c >= '0')&&(c <= '9')){ radix = 8;
                } else { return 0;}
            }
        }
    } else {
        if ((c < '1')||(c > '9')){  return 0;}
        radix = 10;
    }
    val = 0;
    while (c > ' ') {
        if (c >= 'a'){ c -= 0x20;}
        c -= '0';
        if (c >= 17){
            c -= 7;
            if (c <= 9){ return 0;}
        }
        if (c >= radix){ return 0;}
        val = val * radix + c;
        c = *(++(*str));
    }
    if (s){ val = -val;}
    *res = val;
    return 1;
}

//  Get key input data
void get_line (char *buff, int len){
char c;
int idx = 0;

    for (;;) {
        c = GETC();
        if (c == '\r') {
            buff[idx++] = c;
            break;
        }
        if ((c == '\b') && idx) {
            idx--;
            PUTC(c);
            PUTC(' ');
            PUTC(c);
        }
        if (((uint8_t)c >= ' ') && (idx < len - 1)) {
            buff[idx++] = c;
            PUTC(c);
        }
    }
    buff[idx] = 0;
    PUTC('\n');
}
