/*
 * Mbed Library program
 *  UART MP3 Voice Module with 8MB Flash Memory / DFR0534
 *
 * Copyright (c) 2019 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  https://os.mbed.com/users/kenjiArai/
 *      Created:    December  28th, 2019
 *      Revised:    December  29th, 2019
 */

/*
    Reference information:
        http://akizukidenshi.com/catalog/g/gM-13708/
        https://wiki.dfrobot.com/Voice_Module_SKU__DFR0534
 */

#ifndef DFR0534_H
#define DFR0534_H

#include "mbed.h"

#define VOL_MIN     0
#define VOL_MAX     30

enum simple_commands{
    Play = 0x02,
    Pause,
    Playstop,
    Prev_Audio,
    Next_Audio,
    The_Directory_of_the_previous_folder = 0x0e,
    The_directory_of_the_next_folder,
    End_to_play,
    Increase_the_volume = 0x14,
    Reduce_the_volume,
    End_Combination_Play = 0x1c,
    End_repetition = 0x21,
    End_to_send_the_playing_time = 0x26
};

/** UART MP3 Voice Module with 8MB Flash Memory / DFR0534
 * @code
 * #include    "mbed.h"
 * #include    "drf0534.h"
 *
 * DFR0534 mp3(PA_0, PA_1, PA_4);  // tx, rx, busy
 *
 * int main() {
 *     mp3.volume_setting(VOL_MAX / 2);
 *     uint32_t num = mp3.total_num_of_audio();
 *     while(true) {
 *         for (uint32_t i = 1; i < num + 1; i++) {
 *             mp3.play_one(i);
 *             mp3.get_file_name(name_buf);
 *             printf("File name = %s\r\n", name_buf);
 *         }
 *     }
 * }
 * @endcode
 */
 
class DFR0534
{
public:
    /** Create the DFR0534 object
     *  @param PinName communication line TX & RX & Busy signal input
     */
    DFR0534(PinName tx, PinName rx, PinName busy);

    /** Play specific truck 
     * @param truck number
     */
    void play_one(uint16_t track_num);

    /** set volume
     * @param volume value(VOL_MIN to VOL_MAX, 0 to 30)
     */
    void volume_setting(uint8_t vol);

    /** increase volume
     * @param none
     */
    void volume_increase();

    /** decrease volume
     * @param none
     */
    void volume_decrease();

    /** direct command (simple type -> No parameter & no return value)
     * @param command
     */
    void command(enum simple_commands cmds);

    /** Check the total number of audio track
     * @param none
     * @return total number of audio
     */
    uint16_t total_num_of_audio();

    /** Check the current audio track number
     * @param none
     * @return number
     */
    uint16_t current_num_of_audio();

    /** get file name
     * @param buffer pointer (buffer size > 32)
     * @return name into buffer
     */
    void get_file_name(char *buf);

    /** get play time
     * @param none
     * @return time[seconds]
     */
    uint32_t get_play_time();

    /** get play back time
     * @param none
     * @return time[seconds]
     */
    uint32_t get_play_back_time();

private:
    RawSerial  _mp3;
    DigitalIn  _bsy;

    void _rx_handler(void);
    void _command(uint8_t cmd_no);
    void _command_w_para(uint8_t cmd_no, uint8_t *parameter, uint8_t len);
    void _wait_busy(void);
    uint32_t get_time(void);

    CircularBuffer<char, 32> _rxbuf; // small ring buffer
    uint8_t _sum;
    char _buf[32];

};

#endif // DFR0534_H