/*
 * mbed library program
 *  2 channels 16-Bit(available 14/12bit mode) Analog-to-Digital Converter
 *      with I2C Interface ---- MCP3427 by Microchip Technology Inc.
 *
 * Copyright (c) 2018 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created:    June       7th, 2018
 *      Revised:    July      13th, 2018
 */
/*
 *---------------- REFERENCE ---------------------------------------------------
 * Data Sheet DS22226B  Revision A(December 2009) by Microchip Technology Inc.
 *      https://www.microchip.com/wwwproducts/en/MCP3427
 */

#ifndef MCP3427_H
#define MCP3427_H

#include "mbed.h"

////////////// I2C Chip address ////////////////////////////////////////////////
// The device code is followed by three address bits (A2, A1, A0)
//   which are also programmed at the Microchip factory
// Address b7=1,b6=1,b5=0,b4=1,b3(A2)=x,b2(A1)=x,b1(A0)=x, b0=R/W
#define MCP3427GG_ADDR             (0x68 << 1)     // Adr0=0, Adr1=0
#define MCP3427GV_ADDR             (0x6a << 1)     // Adr0=0, Adr1=1
#define MCP3427VG_ADDR             (0x6c << 1)     // Adr0=1, Adr1=0
#define MCP3427VV_ADDR             (0x6e << 1)     // Adr0=1, Adr1=1

////////////// Configration Registers parameter ////////////////////////////////
#define PGA_GAIN_1                  0
#define PGA_GAIN_2                  1
#define PGA_GAIN_4                  2
#define PGA_GAIN_8                  3

#define CONV_MODE_CONTINUOUS        1
#define CONV_MODE_ONE_SHOT          0

#define SAMPLE_RATE_240SPS_12BIT    0
#define SAMPLE_RATE_60SPS_14BIT     1
#define SAMPLE_RATE_15SPS_16BIT     2

typedef struct configration_reg {
    // CH1
    uint8_t     pga_gain1;
    uint8_t     sample_rate1;
    uint8_t     conversion_mode1;
    // CH2
    uint8_t     pga_gain2;
    uint8_t     sample_rate2;
    uint8_t     conversion_mode2;
} mcp3427_config_t;

/** Interface for ADC MCP3427
 * @code
 * #include "mbed.h"
 * #include "MCP3427.h"
 *
 * // Default parameter setting
 * // Continuous Conversion + 15 SPS (16 bits) + Gain x1
 *
 * // I2C Communication
 * MCP3427      adc(I2C_SDA, I2C_SCL, MCP3427GG_ADDR);
 * // If you connected I2C line not only this device but also other devices,
 * //     you need to declare following method.
 * I2C          i2c(I2C_SDA, I2C_SCL);
 * MCP3427      adc(i2c, MCP3427GG_ADDR);
 *
 * mcp3427_config_t my_cnfg;
 *
 * int main() {
 *   my_cnfg.pga_gain1 = PGA_GAIN_1;
 *   my_cnfg.sample_rate1 = SAMPLE_RATE_15SPS_16BIT;
 *   my_cnfg.conversion_mode1 = CONV_MODE_CONTINUOUS;
 *   my_cnfg.pga_gain2 = PGA_GAIN_8;
 *   my_cnfg.sample_rate2 = SAMPLE_RATE_15SPS_16BIT;
 *   my_cnfg.conversion_mode2 = CONV_MODE_CONTINUOUS;
 *   adc.set_config(&my_cnfg);
 *   while(true){
 *      printf("ADC: %6.5f [V]\r\n", adc.read_voltage(1)); // read channel 1
 *      wait(0.2f);
 *      printf("ADC: %6.5f [V]\r\n", adc.read_voltage(2)); // read channel 2
 *      wait(0.2f);
 *   }
 * }
 * @endcode
 */

class MCP3427
{
public:
    /** Configure data pin
      * @param data SDA and SCL pins
      * @param I2C address
      */
    MCP3427(PinName p_sda, PinName p_scl, uint8_t addr);

    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      * @param I2C address
      */
    MCP3427(I2C& p_i2c, uint8_t addr);

    /** Get ADC data
      * @param channel ch1(1) or ch2(2)
      * @return ADC data in 16bit
      */
    int16_t read_16bit(uint8_t ch);

    /** Get ADC data
      * @param channel ch1(1) or ch2(2)
      * @return voltage (includes Gain, Data rate, offset, vref compensation)
      */
    float read_voltage(uint8_t ch);

    /** Start ADC conversion
      * @param channel ch1(1) or ch2(2)
      * @return none
      */
    void start_conversion(uint8_t ch);

    /** Get ADC data
      * @param none
      * @return voltage (includes Gain, Data rate, offset, vref compensation)
      */
    float read_voltage_from_current_ch(void);

    /** Set voltage offset (only effective for read_voltage() function)
      * @param channel ch1(set 1) or ch2(set 2)
      * @param offset voltage (0.0f = default)
      * @return none
      */
    void set_offset_volt(uint8_t ch, float offset);

    /** Set refrence volt compensation (only effective for read_voltage())
      * @param compensation data (1.00f = default)
      * @return none
      */
    void set_vref_compensation(float compensation);

    /** Set  Configuration Register
      * @param pointer for register parameter
      * @return none
      */
    void set_config(mcp3427_config_t *parameter);

    /** Set I2C clock frequency
      * @param freq.
      * @return none
      */
    void frequency(uint32_t hz);

protected:
    I2C *_i2c_p;
    I2C &_i2c;

    void init(void);
    void convert_config2byte(void);
    void _set_ch_and_config(uint8_t ch);
    int16_t _read_16bit(void);
    float _read_voltage(float dt16);

private:
    uint8_t mcp3427_addr;
    uint8_t buf[4];
    uint8_t current_ch;
    uint8_t config_byte[2];
    uint8_t pga_gain[2];
    uint8_t sample_rate[2];
    uint8_t conversion_mode[2];
    
    uint8_t _ch;

    float   offset_volt[2];
    float   compensation_ref;

    int16_t dt_adc16;
    float   dt_adc_f;

};

#endif      // MCP3427_H
