/*
 * mbed library program
 *  16-Bit Analog-to-Digital Converter with I2C Interface
 *  MCP3425 by Microchip Technology Inc.
 *
 * Copyright (c) 2018 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Modify:     March     17th, 2018
 *      Revised:    March     18th, 2018
 */
/*
 *---------------- REFERENCE ---------------------------------------------------
 * Data Sheet DS22072B  Revision B(August 2009) by Microchip Technology Inc. 
 *      http://www.microchip.com/wwwproducts/en/MCP3425
 *      http://akizukidenshi.com/catalog/g/gK-08018/
 */

#ifndef MCP3425_H
#define MCP3425_H

#include "mbed.h"

////////////// I2C Chip address ////////////////////////////////////////////////
// The device code is followed by three address bits (A2, A1, A0)
//   which are also programmed at the Microchip factory
// Address b7=1,b6=1,b5=0,b4=1,b3(A2)=0,b2(A1)=x,b1(A0)=x, b0=R/W
#define MCP3425A0T_ADDR             (0x68 << 1)
#define MCP3425A1T_ADDR             (0x69 << 1)     // option by the factory
#define MCP3425A2T_ADDR             (0x6a << 1)     // option by the factory
#define MCP3425A3T_ADDR             (0x6b << 1)     // option by the factory

////////////// Configration Registers parameter ////////////////////////////////
#define PGA_GAIN_1                  0
#define PGA_GAIN_2                  1
#define PGA_GAIN_4                  2
#define PGA_GAIN_8                  3

#define CONV_MODE_CONTINUOUS        1
#define CONV_MODE_ONE_SHOT          0

#define SAMPLE_RATE_240SPS_12BIT    0
#define SAMPLE_RATE_60SPS_14BIT     1
#define SAMPLE_RATE_15SPS_16BIT     2

typedef struct configration_reg {
    uint8_t     pga_gain;
    uint8_t     sample_rate;
    uint8_t     conversion_mode;
} mcp3425_config_t;

/** Interface for ADC MCP3425
 * @code
 * #include "mbed.h"
 * #include "MCP3425.h"
 *
 * // Default parameter setting
 * // Continuous Conversion + 15 SPS (16 bits) + Gain x1
 *
 * // I2C Communication
 *  MCP3425      adc(I2C_SDA, I2C_SCL);
 * // If you connected I2C line not only this device but also other devices,
 * //     you need to declare following method.
 *  I2C          i2c((I2C_SDA, I2C_SCL));
 *  MCP3425      adc(i2c);
 *
 * int main() {
 *   while(true){
 *      printf("ADC: %6.5f [V]\r\n", adc.read_voltage());
 *      wait(0.2f);
 *   }
 * }
 * @endcode
 */

class MCP3425
{
public:
    /** Configure data pin
      * @param data SDA and SCL pins
      */
    MCP3425(PinName p_sda, PinName p_scl);
    MCP3425(PinName p_sda, PinName p_scl, uint8_t addr);

    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      */
    MCP3425(I2C& p_i2c);
    MCP3425(I2C& p_i2c, uint8_t addr);

    /** Get ADC data
      * @param none
      * @return ADC data in 16bit
      */
    int16_t read_16bit(void);

    /** Get ADC data
      * @param none
      * @return voltage (includes Gain, Data rate, offset, vref compensation)
      */
    float read_voltage(void);

    /** Set voltage offset (only effective for read_voltage() function)
      * @param offset voltage (0.0f = default)
      * @return none
      */
    void set_offset_volt(float offset);

    /** Set refrence volt compensation (only effective for read_voltage())
      * @param compensation data (1.00f = default)
      * @return none
      */
    void set_vref_compensation(float compensation);

    /** Set  Configuration Register
      * @param pointer for register parameter
      * @return none
      */
    void set_config(mcp3425_config_t *parameter);

    /** Read Configuration Register
      * @param pointer for register parameter
      * @return parameter (into above pointer)
      */
    void read_config(mcp3425_config_t *parameter);

    /** Set I2C clock frequency
      * @param freq.
      * @return none
      */
    void frequency(uint32_t hz);

protected:
    I2C *_i2c_p;
    I2C &_i2c;

    void init(void);
    uint8_t convert_config2byte(mcp3425_config_t *config);

private:
    uint8_t mcp3425_addr;
    uint8_t buf[4];
    uint8_t config_byte;
    mcp3425_config_t config;

    float   offset_volt;
    float   compensation_ref;

    int16_t dt_adc16;
    float   dt_adc_f;

};

#endif      // MCP3425_H
