/*
 * mbed library program
 *  Control M41T62 RTC Module
 *  STMicroelectronics
 *
 * Copyright (c) 2014,'15,'17,'20 Kenji Arai / JH1PJL
 *  http://www7b.biglobe.ne.jp/~kenjia/
 *  https://os.mbed.com/users/kenjiArai/
 *      Created: June       21st, 2014
 *      Revised: August      7th, 2020
 */

/*
 *---------------- REFERENCE ---------------------------------------------------
 *  http://www.st-japan.co.jp/web/jp/
 *                  catalog/sense_power/FM151/CL1410/SC403/PF82507
 *  http://strawberry-linux.com/catalog/items?code=12062
 */

#ifndef M41T62_H
#define M41T62_H

#include "mbed.h"

// RTC STmicro M41T62
// Address b7=1,b6=1,b5=0,b4=1,b3=0,b2=0,b1=0, b0=R/W
#define M41T62ADDR  (0x68 << 1) // No other choice

// Register definition
#define M41T62_REG_SSEC         0x0
#define M41T62_REG_SEC          0x1
#define M41T62_REG_MIN          0x2
#define M41T62_REG_HOUR         0x3
#define M41T62_REG_WDAY         0x4
#define M41T62_REG_DAY          0x5
#define M41T62_REG_MON          0x6
#define M41T62_REG_YEAR         0x7
#define M41T62_REG_CALIB        0x8
#define M41T62_REG_WDT          0x9
#define M41T62_REG_ALARM_MON    0xa
#define M41T62_REG_ALARM_DAY    0xb
#define M41T62_REG_ALARM_HOUR   0xc
#define M41T62_REG_ALARM_MIN    0xd
#define M41T62_REG_ALARM_SEC    0xe
#define M41T62_REG_FLAGS        0xf

// Buffer size
#define RTC_BUF_SIZ            (M41T62_REG_FLAGS + 5)

typedef enum {
    RTC_SQW_NONE = 0,
    RTC_SQW_32KHZ,
    RTC_SQW_8KHZ,
    RTC_SQW_4KHZ,
    RTC_SQW_2KHZ,
    RTC_SQW_1KHZ,
    RTC_SQW_512HZ,
    RTC_SQW_256HZ,
    RTC_SQW_128HZ,
    RTC_SQW_64HZ,
    RTC_SQW_32HZ,
    RTC_SQW_16HZ,
    RTC_SQW_8HZ,
    RTC_SQW_4HZ,
    RTC_SQW_2HZ,
    RTC_SQW_1HZ
} sq_wave_t;

/** Interface for RTC (I2C Interface)  STMicroelectronics M41T62
 *
 *  Standalone type RTC via I2C interface
 *
 * @code
 * #include "mbed.h"
 * #include "m41t62_rtc.h"
 *
 * // I2C Communication
 *  M41T62      m41t62(dp5,dp27);   // STmicro RTC(M41T62) SDA, SCL
 * // If you connected I2C line not only this device but also other devices,
 * //     you need to declare following method.
 *  I2C         i2c(dp5,dp27);      // SDA, SCL
 *  M41T62      m41t62(i2c);        // STmicro RTC(M41T62)
 *
 * int main() {
 * tm t;
 * time_t seconds;
 * char buf[40];
 *
 *   m41t62.set_sq_wave(RTC_SQW_NONE); // Stop output for more low current
 *   while(1){
 *      m41t62.read_rtc_std(&t);       // read RTC data
 *      seconds = mktime(&t);
 *      strftime(buf, 40, "%I:%M:%S %p (%Y/%m/%d)", localtime(&seconds));
 *      printf("Date: %s\r\n", buf);
 *   }
 * }
 * @endcode
 */

class M41T62
{
public:

    typedef struct {    // BCD format
        uint8_t rtc_seconds;
        uint8_t rtc_minutes;
        uint8_t rtc_hours;
        uint8_t rtc_weekday;
        uint8_t rtc_date;
        uint8_t rtc_month;
        uint8_t rtc_year_raw;
        uint16_t rtc_year;
    } rtc_time;

    /** Configure data pin (with other devices on I2C line)
      * @param data SDA and SCL pins
      */
    M41T62(PinName p_sda, PinName p_scl);

    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      */
    M41T62(I2C& p_i2c);

    /** Read RTC data with Standard C "struct tm" format
      * @param tm (data save area)
      * @return none but all data in tm
      */
    void read_rtc_std(tm *);
    void get_time_rtc(tm *);

    /** Write data to RTC data with Standard C "struct tm" format
      * @param tm (save writing data)
      * @return none but all data in tm
      */
    void write_rtc_std(tm *);
    void set_time_rtc(tm *);

    /** Set Alarm / IRQ time
      * @param next time (unit: minutes) from now on minimum = 2 minuteｓ!!
      * @return none
      */
    void set_next_IRQ(uint16_t time);

    /** Clear Alarm / IRQ pin interrupt
      * @param none
      * @return none
      */
    void clear_IRQ(void);

    /** Read RTC data with own format
      * @param tm (data save area)
      * @return none but all data in tm
      */
    void read_rtc_direct(rtc_time *);

    /** Read RTC data with own format
      * @param tm (save writing data)
      * @return none but all data in tm
      */
    void write_rtc_direct(rtc_time *);

    /** Set I2C clock frequency
      * @param freq.
      * @return none
      */
    void frequency(int);

    /** Control Square wave output port
      * @param output_mode 
      * @return none
      */
    void set_sq_wave(sq_wave_t);

protected:
    I2C *_i2c_p;
    I2C &_i2c;

    uint8_t bin2bcd(uint8_t);
    uint8_t bcd2bin(uint8_t);
    void set_alarm_reg (uint16_t time);

private:
    uint8_t M41T62_addr;
    uint8_t rtc_buf[RTC_BUF_SIZ];   // buffer for RTC

};

#endif      // M41T62_H
