/*
 * mbed library program
 *  Control M41T62 RTC Module
 *  STMicroelectronics
 *
 * Copyright (c) 2014,'15,'17,'20 Kenji Arai / JH1PJL
 *  http://www7b.biglobe.ne.jp/~kenjia/
 *  https://os.mbed.com/users/kenjiArai/
 *      Created: June       21st, 2014
 *      Revised: August      7th, 2020
 */

#include "mbed.h"
#include "m41t62_rtc.h"

#define RTC_Wk_Sunday          ((uint8_t)0x00)

M41T62::M41T62 (PinName p_sda, PinName p_scl)
    : _i2c_p(new I2C(p_sda, p_scl)), _i2c(*_i2c_p)
{
    M41T62_addr = M41T62ADDR;
    _i2c.frequency(400000);
}

M41T62::M41T62 (I2C& p_i2c)
    : _i2c(p_i2c)
{
    M41T62_addr = M41T62ADDR;
    _i2c.frequency(400000);
}

/////////////// Read RTC data //////////////////////////////////////////////////
void M41T62::get_time_rtc (tm *t)
{
    read_rtc_std(t);
}

void M41T62::read_rtc_std (tm *t)
{
    rtc_time time;

    read_rtc_direct(&time);
    t->tm_sec  = time.rtc_seconds;
    t->tm_min  = time.rtc_minutes;
    t->tm_hour = time.rtc_hours;
    t->tm_mday = time.rtc_date;
    if ( time.rtc_weekday == RTC_Wk_Sunday) {
        t->tm_wday = 0; // Sun is not 7 but 0
    } else {
        t->tm_wday = time.rtc_weekday;
    }
    t->tm_mon  = time.rtc_month - 1;
    t->tm_year = time.rtc_year_raw + 100;
    t->tm_isdst= 0;
}

/////////////// Write data to RTC //////////////////////////////////////////////
void M41T62::set_time_rtc (tm *t)
{
    write_rtc_std(t);
}

void M41T62::write_rtc_std (tm *t)
{
    rtc_time time;

    time.rtc_seconds  = t->tm_sec;
    time.rtc_minutes  = t->tm_min;
    time.rtc_hours    = t->tm_hour;
    time.rtc_date     = t->tm_mday;
    if ( t->tm_wday == 0) {
        time.rtc_weekday = RTC_Wk_Sunday;
    } else {
        time.rtc_weekday = t->tm_wday;
    }
    time.rtc_month    = t->tm_mon + 1;
    time.rtc_year_raw = t->tm_year - 100;
    write_rtc_direct(&time);
}

/////////////// Read/Write RTC another format //////////////////////////////////
void M41T62::read_rtc_direct (rtc_time *tm)
{
    rtc_buf[0] = M41T62_REG_SSEC;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 1, true);
    _i2c.read((int)M41T62_addr, (char *)rtc_buf, 8, false);
    tm->rtc_seconds = bcd2bin(rtc_buf[M41T62_REG_SEC]  & 0x7f);
    tm->rtc_minutes = bcd2bin(rtc_buf[M41T62_REG_MIN]  & 0x7f);
    tm->rtc_hours   = bcd2bin(rtc_buf[M41T62_REG_HOUR] & 0x3f);
    tm->rtc_date    = bcd2bin(rtc_buf[M41T62_REG_DAY]  & 0x3f);
    tm->rtc_weekday = rtc_buf[M41T62_REG_WDAY] & 0x07;
    tm->rtc_month   = bcd2bin(rtc_buf[M41T62_REG_MON]  & 0x1f);
    tm->rtc_year_raw= bcd2bin(rtc_buf[M41T62_REG_YEAR]);
    tm->rtc_year = tm->rtc_year_raw + 100 + 1900;
}

void M41T62::write_rtc_direct (rtc_time *tm)
{
    rtc_buf[M41T62_REG_YEAR + 1] = bin2bcd(tm->rtc_year_raw);
    rtc_buf[M41T62_REG_MON  + 1] = bin2bcd(tm->rtc_month)   & 0x1f;
    rtc_buf[M41T62_REG_DAY  + 1] = bin2bcd(tm->rtc_date)    & 0x3f;
    rtc_buf[M41T62_REG_WDAY + 1] = (tm->rtc_weekday & 0x07);
    rtc_buf[M41T62_REG_HOUR + 1] = bin2bcd(tm->rtc_hours)   & 0x3f;
    rtc_buf[M41T62_REG_MIN  + 1] = bin2bcd(tm->rtc_minutes) & 0x7f;
    rtc_buf[M41T62_REG_SEC  + 1] = bin2bcd(tm->rtc_seconds) & 0x7f;
    rtc_buf[M41T62_REG_SSEC + 1] = 0;
    rtc_buf[0] = M41T62_REG_SSEC;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 9, false);
}

/////////////// Set Alarm / IRQ ////////////////////////////////////////////////
void M41T62::set_alarm_reg (uint16_t time)
{
    tm t;
    uint8_t m, h;

    read_rtc_std(&t);   // read current time
    m = t.tm_min + (uint8_t)(time % 60);
    h = t.tm_hour;
    if (m >= 60) {
        m -= 60;
        h += 1;
    }
    h += (uint8_t)(time / 60);
    if (h >= 24) {
        h -= 24;
    }
    // set OUT = 1
    rtc_buf[0] = M41T62_REG_CALIB;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 1, true);
    _i2c.read((int)M41T62_addr, (char *)rtc_buf, 1, false);
    rtc_buf[1] = rtc_buf[0] & 0x3f;     // keep calbration data
    rtc_buf[1] = rtc_buf[0] | 0x80;     // set OUT
    rtc_buf[0] = M41T62_REG_CALIB;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 2, false);
    // RPT4=1,RPT5=0,RPT3=0,RPT2=0,RPT1=0 & set day,hour,min,sec
    rtc_buf[4] = 0;                 // M41T62_REG_ALARM_SEC ->RPT=1, set 0sec
    rtc_buf[3] = bin2bcd(m) & 0x7f; // M41T62_REG_ALARM_MIN ->RPT2=0
    rtc_buf[2] = bin2bcd(h) & 0x3f; // M41T62_REG_ALARM_HOUR ->RPT3=0
    rtc_buf[1] = 0xc0;              // M41T62_REG_ALARM_DAY ->RPT4=1,RPT5=1
    rtc_buf[0] = M41T62_REG_ALARM_DAY;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 5, false);
    // set AFE(alarm enable flag)
    rtc_buf[0] = M41T62_REG_ALARM_MON;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 1, true);
    _i2c.read((int)M41T62_addr, (char *)rtc_buf, 1, false);
    rtc_buf[1] = rtc_buf[0] & 0x40;     // keep SQWE bit
    rtc_buf[1] |= 0x80;                 // set AFE
    rtc_buf[0] = M41T62_REG_ALARM_MON;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 2, false);
}

void M41T62::set_next_IRQ (uint16_t time)
{
    uint16_t t;

    if (time < 2) {
        // Alarm does not check seconds digit.
        // If 59 to 0 is occured during setting here,
        // 1 minute will have a trouble.
        t = 2;
    } else if (time > 1440) {  // set less than 24 hours
        t = 1440;
    } else {
        t = time;
    }
    set_alarm_reg(t);
}

/////////////// Clear Alarm / IRQ pin interrupt ////////////////////////////////
void M41T62::clear_IRQ ()
{
    for (uint32_t i = 0; i < 40; i++) {
        rtc_buf[0] = M41T62_REG_FLAGS;
        _i2c.write((int)M41T62_addr, (char *)rtc_buf, 1, true);
        _i2c.read((int)M41T62_addr, (char *)rtc_buf, 1, false);
        if ((rtc_buf[0] & 0x40) == 0) {
            break;
        }
    }
    // clear AFE(alarm enable flag)
    rtc_buf[0] = M41T62_REG_ALARM_MON;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 1, true);
    _i2c.read((int)M41T62_addr, (char *)rtc_buf, 1, false);
    rtc_buf[1] = rtc_buf[0] & 0x40;     // keep SQWE bit
    rtc_buf[0] = M41T62_REG_ALARM_MON;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 2, false);
    // set OUT = 1
    rtc_buf[0] = M41T62_REG_CALIB;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 1, true);
    _i2c.read((int)M41T62_addr, (char *)rtc_buf, 1, false);
    rtc_buf[1] = rtc_buf[0] & 0x3f;     // keep calbration data
    rtc_buf[1] = rtc_buf[0] | 0x80;     // set OUT
    rtc_buf[0] = M41T62_REG_CALIB;
}

/////////////// I2C Freq. //////////////////////////////////////////////////////
void M41T62::frequency (int hz)
{
    _i2c.frequency(hz);
}

/////////////// Square wave output /////////////////////////////////////////////
void M41T62::set_sq_wave (sq_wave_t sqw_dt)
{
    // set SQW frequency
    rtc_buf[0] = M41T62_REG_WDAY;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 1, true);
    _i2c.read((int)M41T62_addr, (char *)rtc_buf, 1, false);
    rtc_buf[1] = (rtc_buf[0] & 0x07) | (sqw_dt << 4);
    rtc_buf[0] = M41T62_REG_WDAY;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 2, false);
    // set or clear SQWE
    rtc_buf[0] = M41T62_REG_ALARM_MON;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 1, true);
    _i2c.read((int)M41T62_addr, (char *)rtc_buf, 1, false);
    if (sqw_dt == RTC_SQW_NONE) {   // Clear SQWE
        rtc_buf[1] = rtc_buf[0] & 0xbf;
    } else {                        // Set SQWE
        rtc_buf[1] = rtc_buf[0] | 0x40;
    }
    rtc_buf[0] = M41T62_REG_ALARM_MON;
    _i2c.write((int)M41T62_addr, (char *)rtc_buf, 2, false);
}

/////////////// conversion BCD & BIN ///////////////////////////////////////////
uint8_t M41T62::bin2bcd (uint8_t dt)
{
    uint8_t bcdhigh = 0;

    while (dt >= 10) {
        bcdhigh++;
        dt -= 10;
    }
    return  ((uint8_t)(bcdhigh << 4) | dt);
}

uint8_t M41T62::bcd2bin (uint8_t dt)
{
    uint8_t tmp = 0;

    tmp = ((uint8_t)(dt & (uint8_t)0xf0) >> (uint8_t)0x4) * 10;
    return (tmp + (dt & (uint8_t)0x0f));
}
