/*
 * mbed library program
 *  LSM9DS0 iNEMO inertial module:3D accelerometer,3D gyroscope,3D magnetometer
 *  by STMicroelectronics
 *
 * Copyright (c) 2015,'17 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: Feburary  24th, 2015
 *      Revised: September 21st, 2017
 */
/*
 *---------------- REFERENCE ---------------------------------------------------
 * Original Information
 *  http://www.st.com/web/jp/jp/catalog/sense_power/FM89/SC1448/PF258556
 *  https://strawberry-linux.com/catalog/items?code=12155
 *  DoclD024763 Rev 2 August 2013
 * Sensor board
 *  https://strawberry-linux.com/catalog/items?code=12155
 */

#ifndef LSM9DS0_3D_H
#define LSM9DS0_3D_H

#include "mbed.h"

//  LSM9DS0 iNEMO Address / Acceleration & Magnetic (_XM)
//  7bit address = 0b00111xx(0x1e or 0x1d depends on SA0_XM/SDO_XM)
#define LSM9DS0_XM_G_CHIP_ADDR  (0x1e << 1)    // SA0_XM(=SDO pin) = GND
#define LSM9DS0_XM_V_CHIP_ADDR  (0x1d << 1)    // SA0_XM(=SDO pin) = Vdd

//  LSM9DS0 iNEMO Address / Gyro(Angular) (_G)
//  7bit address = 0b110101x(0x6a or 0x6b depends on SA0_G/SDO_G)
#define LSM9DS0_G_G_CHIP_ADDR   (0x6a << 1)    // SA0_G(=SDO pin) = GND
#define LSM9DS0_G_V_CHIP_ADDR   (0x6b << 1)    // SA0_G(=SDO pin) = Vdd

//  LSM9DS0 iNEMO ID / XM
#define I_AM_LSM9DS0_XM         0x49
//  LSM9DS0 iNEMO ID / G
#define I_AM_LSM9DS0_G          0xd4

//----- Register's definition -----------------------------
//  Angular
#define LSM9DS0_WHO_AM_I_G      0x0f
#define LSM9DS0_CTRL_REG1_G     0x20
#define LSM9DS0_CTRL_REG2_G     0x21
#define LSM9DS0_CTRL_REG3_G     0x22
#define LSM9DS0_CTRL_REG4_G     0x23
#define LSM9DS0_CTRL_REG5_G     0x24
#define LSM9DS0_REFERENCE_G     0x25
#define LSM9DS0_STATUS_REG_G    0x27
#define LSM9DS0_OUT_X_L_G       0x28
#define LSM9DS0_OUT_X_H_G       0x29
#define LSM9DS0_OUT_Y_L_G       0x2a
#define LSM9DS0_OUT_Y_H_G       0x2b
#define LSM9DS0_OUT_Z_L_G       0x2c
#define LSM9DS0_OUT_Z_H_G       0x2d
#define LSM9DS0_FIFO_CTRL_REG_G 0x2e
#define LSM9DS0_FIFO_SRC_REG_G  0x2f
#define LSM9DS0_INT1_CFG_G      0x30
#define LSM9DS0_INT1_SRC_G      0x31
#define LSM9DS0_INT1_TSH_XH_G   0x32
#define LSM9DS0_INT1_TSH_XL_G   0x33
#define LSM9DS0_INT1_TSH_YH_G   0x34
#define LSM9DS0_INT1_TSH_YL_G   0x35
#define LSM9DS0_INT1_TSH_ZH_G   0x36
#define LSM9DS0_INT1_TSH_ZL_G   0x37
#define LSM9DS0_INT1_DURATION_G 0x38

//  Acceleration & Magnetic
#define LSM9DS0_OUT_TEMP_L_XM   0x05
#define LSM9DS0_OUT_TEMP_H_XM   0x06
#define LSM9DS0_STATUS_REG_M    0x07
#define LSM9DS0_OUT_X_L_M       0x08
#define LSM9DS0_OUT_X_H_M       0x09
#define LSM9DS0_OUT_Y_L_M       0x0a
#define LSM9DS0_OUT_Y_H_M       0x0b
#define LSM9DS0_OUT_Z_L_M       0x0c
#define LSM9DS0_OUT_Z_H_M       0x0d
#define LSM9DS0_WHO_AM_I_XM     0x0f
#define LSM9DS0_INT_CTRL_REG_M  0x12
#define LSM9DS0_INT_SRC_M       0x13
#define LSM9DS0_INT_TSH_H_M     0x14
#define LSM9DS0_INT_TSH_L_M     0x15
#define LSM9DS0_OFFSET_X_L_M    0x16
#define LSM9DS0_OFFSET_X_H_M    0x17
#define LSM9DS0_OFFSET_Y_L_M    0x18
#define LSM9DS0_OFFSET_Y_H_M    0x19
#define LSM9DS0_OFFSET_Z_L_M    0x1a
#define LSM9DS0_OFFSET_Z_H_M    0x1b
#define LSM9DS0_REFERENCE_X     0x1c
#define LSM9DS0_REFERENCE_Y     0x1d
#define LSM9DS0_REFERENCE_Z     0x1e
#define LSM9DS0_CTRL_REG0_XM    0x1f
#define LSM9DS0_CTRL_REG1_XM    0x20
#define LSM9DS0_CTRL_REG2_XM    0x21
#define LSM9DS0_CTRL_REG3_XM    0x22
#define LSM9DS0_CTRL_REG4_XM    0x23
#define LSM9DS0_CTRL_REG5_XM    0x24
#define LSM9DS0_CTRL_REG6_XM    0x25
#define LSM9DS0_CTRL_REG7_XM    0x26
#define LSM9DS0_STATUS_REG_A    0x27
#define LSM9DS0_OUT_X_L_A       0x28
#define LSM9DS0_OUT_X_H_A       0x29
#define LSM9DS0_OUT_Y_L_A       0x2a
#define LSM9DS0_OUT_Y_H_A       0x2b
#define LSM9DS0_OUT_Z_L_A       0x2c
#define LSM9DS0_OUT_Z_H_A       0x2d
#define LSM9DS0_FIFO_CTRL_REG   0x2e
#define LSM9DS0_FIFO_SRC_REG    0x2f
#define LSM9DS0_INT_GEN_1_REG   0x30
#define LSM9DS0_INT_GEN_1_SRC   0x31
#define LSM9DS0_INT_GEN_1_THS   0x32
#define LSM9DS0_INT_GEN_1_DURATION  0x33
#define LSM9DS0_INT_GEN_2_REG   0x34
#define LSM9DS0_INT_GEN_2_SRC   0x35
#define LSM9DS0_INT_GEN_2_THS   0x36
#define LSM9DS0_INT_GEN_2_DURATION  0x37
#define LSM9DS0_CLICK_CFG       0x38
#define LSM9DS0_CLICK_SRC       0x39
#define LSM9DS0_CLICK_THS       0x3A
#define LSM9DS0_TIME_LIMIT      0x3B
#define LSM9DS0_TIME_LATENCY    0x3C
#define LSM9DS0_TIME_WINDOW     0x3D
#define LSM9DS0_ACT_THS         0x3E
#define LSM9DS0_ACT_DUR         0x3F

//----- Register bit definition ---------------------------
// CTRL_REG1_G
// Output Data Rate (DR0,1) + Bandwidth (BW0,1)
#define LSM9_G_DR095_12R5       (0 << 4)
#define LSM9_G_DR095_25         (1 << 4)
#define LSM9_G_DR190_12R5       (4 << 4)
#define LSM9_G_DR190_25         (5 << 4)
#define LSM9_G_DR190_50         (6 << 4)
#define LSM9_G_DR190_70         (7 << 4)
#define LSM9_G_DR380_20         (8 << 4)
#define LSM9_G_DR380_25         (9 << 4)
#define LSM9_G_DR380_50         (10 << 4)
#define LSM9_G_DR380_100        (11 << 4)
#define LSM9_G_DR760_30         (12 << 4)
#define LSM9_G_DR760_35         (13 << 4)
#define LSM9_G_DR760_50         (14 << 4)
#define LSM9_G_DR760_100        (15 << 4)

// CTRL_REG1_G
// Power-down mode enable/disable
#define LSM9_G_PD_EN            (0 << 3)
#define LSM9_G_PD_DIS           (1 << 3)

#define LSM9_G_PWR_DWN          (0x0)
#define LSM9_G_SLEEP            (0x8)
#define LSM9_G_NORMAL           (0xf)

// CTRL_REG4_G
// Full Scale
#define LSM9_G_FS_245DPS        (0 << 4)
#define LSM9_G_FS_500DPS        (1 << 4)
#define LSM9_G_FS_2000DPS       (2 << 4)

// CTRL_REG1_XM
// Acceleration data rate (AODR3,2,1,0)
#define LSM9_X_PWRDWN           (0 << 4)
#define LSM9_X_3R125HZ          (1 << 4)
#define LSM9_X_6R25HZ           (2 << 4)
#define LSM9_X_12R5HZ           (3 << 4)
#define LSM9_X_25HZ             (4 << 4)
#define LSM9_X_50HZ             (5 << 4)
#define LSM9_X_100HZ            (6 << 4)
#define LSM9_X_200HZ            (7 << 4)
#define LSM9_X_400HZ            (8 << 4)
#define LSM9_X_800HZ            (9 << 4)
#define LSM9_X_1600HZ           (10 << 4)

#define LSM9_X_CONT_EN_XYZ      (0x7)

// CTRL_REG2_XM
// Accelerometer anti-alias filter bandwidth (ABW1,0)
#define LSM9_X_AT_773HZ         (0 << 6)
#define LSM9_X_AT_194HZ         (1 << 6)
#define LSM9_X_AT_362HZ         (2 << 6)
#define LSM9_X_AT_50HZ          (3 << 6)
// Acceleration full-scale selection (AFS2,1,0)
#define LSM9_X_2G               (0 << 3)
#define LSM9_X_4G               (1 << 3)
#define LSM9_X_6G               (2 << 3)
#define LSM9_X_8G               (3 << 3)
#define LSM9_X_16G              (4 << 3)
// Acceleration self-test enable (AST1,0)
#define LSM9_X_NORMAL_MODE      (0 << 1)
#define LSM9_X_P_SELF_TST       (1 << 1)
#define LSM9_X_N_SELF_TST       (2 << 1)

// CTRL_REG5_XM
// Temperature sensor enable
#define LSM9_M_TMP_ENA          (1 << 7)
#define LSM9_M_TMP_DIS          (0 << 7)
// Magnetic resolution selection (M_RES1,0)
#define LSM9_M_LOW_RESOL        (0 << 5)
#define LSM9_M_HI_RESOL         (3 << 5)
// Magnetic data rate selection (M_ODR2,1,0)
#define LSM9_M_3R125HZ          (0 << 2)
#define LSM9_M_6R25HZ           (1 << 2)
#define LSM9_M_12R5HZ           (2 << 2)
#define LSM9_M_25HZ             (3 << 2)
#define LSM9_M_50HZ             (4 << 2)
#define LSM9_M_100HZ            (5 << 2)

// CTRL_REG6_XM
// Magnetic full-scale selection (MFS1,0)
#define LSM9_M_2GAUSS           (0 << 5)
#define LSM9_M_4GAUSS           (1 << 5)
#define LSM9_M_8GAUSS           (2 << 5)
#define LSM9_M_12GAUSS          (3 << 5)

// CTRL_REG7_XM
// High-pass filter mode selection (AHPM1,0)
#define LSM9_X_NORMAL           (0 << 6)
#define LSM9_X_REF_SIG_FIL      (1 << 6)
#define LSM9_X_AUT_RES          (3 << 6)
// Magnetic sensor mode selection (MD1,0)
#define LSM9_M_CONTINUOUS       (0 << 0)
#define LSM9_M_SINGLE           (1 << 0)
#define LSM9_M_PWRDWN           (2 << 0)

//----- General data definition ---------------------------
#define GRAVITY                 (9.80665f)
//Convert from degrees to radians.
#define toRadians(x)            (x * 0.01745329252f)
//Convert from radians to degrees.
#define toDegrees(x)            (x * 57.2957795f)

////////////// DATA TYPE DEFINITION ///////////////////////
typedef struct {
    uint8_t addr;
    uint8_t range;
    uint8_t data_rate;
    uint8_t anti_filter;
} LSM9DS0_AXL_TypeDef;

typedef struct {
    uint8_t range;
    uint8_t data_rate;
} LSM9DS0_MAG_TypeDef;

typedef struct {
    uint8_t addr;
    uint8_t range;
    uint8_t datarate_and_bandwidth;
} LSM9DS0_GYR_TypeDef;


////////////// DEFAULT SETTING ////////////////////////////
// Standard parameter for easy set-up
const LSM9DS0_AXL_TypeDef acc_std_paramtr = {
    LSM9DS0_XM_G_CHIP_ADDR, // I2C Address, Acc & Magn
    LSM9_X_2G,              // Acc-range slection
    LSM9_X_100HZ,           // Data rate
    LSM9_X_AT_773HZ         // Anti-filter BW
};

const LSM9DS0_MAG_TypeDef mag_std_paramtr = {
    LSM9_M_2GAUSS,          // Mag-range slection
    LSM9_M_25HZ             // Data rate
};

const LSM9DS0_GYR_TypeDef gyr_std_paramtr = {
    LSM9DS0_G_G_CHIP_ADDR,  // I2C Address, Gyro(Angular)
    LSM9_G_FS_245DPS,       // Gyro-range slection
    LSM9_G_DR095_12R5       // Data rate
};

/** Interface for STMicronics inertial module:3D accelerometer, 3D gyroscope, 3D magnetometer
 *      Chip: LSM9DS0
 *
 * @code
 * //--------- Default setting -----------------
 * #include "mbed.h"
 *
 * // I2C Communication
 * LSM9DS0 imu(dp5, dp27, LSM9DS0_XM_G_CHIP_ADDR, LSM9DS0_G_G_CHIP_ADDR);
 * // If you connected I2C line not only this device but also other devices,
 * //     you need to declare following method.
 * I2C i2c(dp5,dp27);              // SDA, SCL
 * LSM9DS0 gyro(i2c, LSM9DS0_XM_G_CHIP_ADDR, LSM9DS0_G_G_CHIP_ADDR);
 *
 * int main() {
 * float fa[3];    // Acc  0:X, 1:Y, 2:Z
 * float fm[3];    // Mag  0:X, 1:Y, 2:Z
 * float fg[3];    // Gyr  0:X, 1:Y, 2:Z
 *
 *    while(true){
 *       imu.get_acc(fa);
 *       imu.get_mag(fm);
 *       imu.get_gyr(fg);
 *       wait(0.5);
 *    }
 * }
 *
 * //--------- Detail setting -----------------
 * #include "mbed.h"
 *
 * const LSM9DS0_AXL_TypeDef acc_prmtr = {
 *     LSM9DS0_XM_V_CHIP_ADDR, // I2C Address, Acc & Magn
 *     LSM9_X_4G,              // Acc-range slection
 *     LSM9_X_100HZ,           // Data rate
 *     LSM9_X_AT_773HZ         // Anti-filter BW
 * };
 *
 * const LSM9DS0_MAG_TypeDef mag_prmtr = {
 *     LSM9_M_4GAUSS,          // Mag-range slection
 *     LSM9_M_25HZ             // Data rate
 * };
 *
 * const LSM9DS0_GYR_TypeDef gyr_prmtr = {
 *     LSM9DS0_G_V_CHIP_ADDR,  // I2C Address, Gyro(Angular)
 *     LSM9_G_FS_245DPS,       // Gyro-range slection
 *     LSM9_G_DR095_12R5       // Data rate
 * };
 *
 * // I2C Communication
 * I2C i2c(dp5,dp27);              // SDA, SCL
 * LSM9DS0 imu(i2c,, &acc_prmtr, &mag_prmtr, &gyr_prmtr );
 *
 * int main() {
 * float fa[3];    // Acc  0:X, 1:Y, 2:Z
 * float fm[3];    // Mag  0:X, 1:Y, 2:Z
 * float fg[3];    // Gyr  0:X, 1:Y, 2:Z
 *
 *    while(true){
 *       imu.get_acc(fa);
 *       imu.get_mag(fm);
 *       imu.get_gyr(fg);
 *       wait(0.5);
 *    }
 * }
 * @endcode
 */

class LSM9DS0
{
public:
    /** Configure data pin
      * @param data SDA and SCL pins
      * @param parameter address for acc (LSM9DS0_AXL_TypeDef)
      * @param parameter address for mag (LSM9DS0_MAG_TypeDef)
      * @param parameter address for gyr (LSM9DS0_GYR_TypeDef)
      */
    LSM9DS0(PinName p_sda, PinName p_scl,
            const LSM9DS0_AXL_TypeDef *acc_parameter,
            const LSM9DS0_MAG_TypeDef *mag_parameter,
            const LSM9DS0_GYR_TypeDef *gyr_parameter);

    /** Configure data pin
      * @param data SDA and SCL pins
      * @param device address
      * @param Other parameters are set default data
      */
    LSM9DS0(PinName p_sda, PinName p_scl, uint8_t acc_mag_addr, uint8_t gyr_addr);

    /** Configure data pin
      * @param data SDA and SCL pins
      * @param Other parameters are set default data
      */
    LSM9DS0(PinName p_sda, PinName p_scl);

    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      * @param parameter address for acc (LSM9DS0_AXL_TypeDef)
      * @param parameter address for mag (LSM9DS0_MAG_TypeDef)
      * @param parameter address for gyr (LSM9DS0_GYR_TypeDef)
      */
    LSM9DS0(I2C& p_i2c,
            const LSM9DS0_AXL_TypeDef *acc_parameter,
            const LSM9DS0_MAG_TypeDef *mag_parameter,
            const LSM9DS0_GYR_TypeDef *gyr_parameter);

    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      * @param Other parameters are set default data
      */
    LSM9DS0(I2C& p_i2c, uint8_t acc_mag_addr, uint8_t gyr_addr);

    /** Configure data pin (with other devices on I2C line)
      * @param I2C previous definition
      * @param Other parameters are set default data
      */
    LSM9DS0(I2C& p_i2c);

    /** Read a float type data from Acceleration
      * @param float type of three arry's address, e.g. float fa[3];
      * @return Acc motion data unit in param array: m/s/s
      * @return fa[0]->x, fa[1]->y, fa[2]->z
      */
    void read_acc(float *dt_usr);
    void get_acc(float *dt_usr);

    /**
      * @return Acc motion data unit: mg
      */
    void read_mg_acc(float *dt_usr);
    void get_mg_acc(float *dt_usr);

    /** Read Data Ready flag
      * @param none
      * @return 1 = Ready
      */
    uint8_t data_ready_acc(void);

    /** Read a float type data from Magnetic
      * @param float type of three arry's address, e.g. float fm[3];
      * @return Mag data unit in param array: gauss
      * @return fm[0]->x, fm[1]->y, fm[2]->z
      */
    void read_mag(float *dt_usr);
    void get_mag(float *dt_usr);

    /** Read Data Ready flag
      * @param none
      * @return 1 = Ready
      */
    uint8_t data_ready_mag(void);

    /** Read a float type data from Gyro
      * @param float type of three arry's address, e.g. float fg[3];
      * @return Gyro motion data unit in param array:dps(degree per second)
      * @return fg[0]->x, fg[1]->y, fg[2]->z
      */
    void read_gyr(float *dt_usr);
    void get_gyr(float *dt_usr);

    /** Read Data Ready flag
      * @param none
      * @return 1 = Ready
      */
    uint8_t data_ready_gyr(void);

    /** Read a Acc & Mag ID number
      * @param none
      * @return ID number
      */
    uint8_t read_acc_mag_id(void);

    /** Read a Gyro ID number
      * @param none
      * @return ID number
      */
    uint8_t read_gyr_id(void);

    /** Set I2C clock frequency
      * @param freq.
      * @return none
      */
    void frequency(int hz);

    /** Read register XM (X=acc, M=mag)(general purpose)
      * @param register's address
      * @return register data
      */
    uint8_t read_reg_acc_mag(uint8_t addr);

    /** Write register XM (X=acc, M=mag)(general purpose)
      * @param register's address
      * @param data
      * @return register data
      */
    uint8_t write_reg_acc_mag(uint8_t addr, uint8_t data);

    /** Read register G (G=gyro)(general purpose)
      * @param register's address
      * @return register data
      */
    uint8_t read_reg_gyr(uint8_t addr);

    /** Write register G (G=gyro)(general purpose)
      * @param register's address
      * @param data
      * @return register data
      */
    uint8_t write_reg_gyr(uint8_t addr, uint8_t data);

protected:
    void initialize(void);
    void read_acc_raw(float *dt_usr);
    void set_initial_dt_to_regs_x(void);
    void set_initial_dt_to_regs_m(void);
    void set_initial_dt_to_regs_g(void);
    void check_id_xm(void);
    void check_id_g(void);

    I2C *_i2c_p;
    I2C &_i2c;

private:
    LSM9DS0_AXL_TypeDef acc_set_data;
    LSM9DS0_MAG_TypeDef mag_set_data;
    LSM9DS0_GYR_TypeDef gyr_set_data;
    float   fsx_factor;  // full scale factor
    float   fsm_factor;  // full scale factor
    float   fsg_factor;  // full scale factor
    char    dt[8];       // working buffer
    uint8_t acc_mag_id;  // acc & mag ID
    uint8_t gyr_id;      // gyr ID
    uint8_t acc_ready;   // Acc is on I2C line = 1, not = 0
    uint8_t gyr_ready;   // Gyr is on I2C line = 1, not = 0
};

#endif      // LSM9DS0_3D_H
