/*
 * mbed library program
 *  LSM9DS0 iNEMO inertial module:3D accelerometer,3D gyroscope,3D magnetometer
 *  by STMicroelectronics
 *
 * Copyright (c) 2015,'17 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: Feburary  24th, 2015
 *      Revised: September 21st, 2017
 */

#include "LSM9DS0.h"

LSM9DS0::LSM9DS0 (PinName p_sda, PinName p_scl,
                  const LSM9DS0_AXL_TypeDef *acc_parameter,
                  const LSM9DS0_MAG_TypeDef *mag_parameter,
                  const LSM9DS0_GYR_TypeDef *gyr_parameter) :
    _i2c_p(new I2C(p_sda, p_scl)), _i2c(*_i2c_p)
{
    _i2c.frequency(400000);
    acc_set_data = *acc_parameter;
    mag_set_data = *mag_parameter;
    gyr_set_data = *gyr_parameter;
    initialize ();
}

LSM9DS0::LSM9DS0 (PinName p_sda, PinName p_scl, uint8_t acc_mag_addr, uint8_t gyr_addr) :
    _i2c_p(new I2C(p_sda, p_scl)), _i2c(*_i2c_p)
{
    _i2c.frequency(400000);
    // Use standard setting
    acc_set_data = acc_std_paramtr;
    mag_set_data = mag_std_paramtr;
    gyr_set_data = gyr_std_paramtr;
    // Change user defined address
    acc_set_data.addr = acc_mag_addr;
    gyr_set_data.addr = gyr_addr;
    initialize ();
}

LSM9DS0::LSM9DS0 (PinName p_sda, PinName p_scl) :
    _i2c_p(new I2C(p_sda, p_scl)), _i2c(*_i2c_p)
{
    _i2c.frequency(400000);
    // Use standard setting
    acc_set_data = acc_std_paramtr;
    mag_set_data = mag_std_paramtr;
    gyr_set_data = gyr_std_paramtr;
    initialize ();
}

LSM9DS0::LSM9DS0 (I2C& p_i2c,
                  const LSM9DS0_AXL_TypeDef *acc_parameter,
                  const LSM9DS0_MAG_TypeDef *mag_parameter,
                  const LSM9DS0_GYR_TypeDef *gyr_parameter) :
    _i2c(p_i2c)
{
    _i2c.frequency(400000);
    acc_set_data = *acc_parameter;
    mag_set_data = *mag_parameter;
    gyr_set_data = *gyr_parameter;
    initialize ();
}

LSM9DS0::LSM9DS0 (I2C& p_i2c, uint8_t acc_mag_addr, uint8_t gyr_addr) :
    _i2c(p_i2c)
{
    _i2c.frequency(400000);
    // Use standard setting
    acc_set_data = acc_std_paramtr;
    mag_set_data = mag_std_paramtr;
    gyr_set_data = gyr_std_paramtr;
    // Change user defined address
    acc_set_data.addr = acc_mag_addr;
    gyr_set_data.addr = gyr_addr;
    initialize ();
}

LSM9DS0::LSM9DS0 (I2C& p_i2c) :
    _i2c(p_i2c)
{
    _i2c.frequency(400000);
    // Use standard setting
    acc_set_data = acc_std_paramtr;
    mag_set_data = mag_std_paramtr;
    gyr_set_data = gyr_std_paramtr;
    initialize ();
}

/////////////// Initialize ////////////////////////////////
void LSM9DS0::initialize (void)
{
    // Check Acc & Mag & Gyro are available of not
    check_id_xm();
    check_id_g();
    if (acc_ready == 0 && gyr_ready == 0) {
        return;
    }
    // Set initial data
    set_initial_dt_to_regs_x(); // Acc
    set_initial_dt_to_regs_m(); // Mag
    set_initial_dt_to_regs_g(); // Gyro
}

////// Set initialize data to related registers ///////////
void LSM9DS0::set_initial_dt_to_regs_x(void)
{
    // Control Reg #1
    dt[0] = LSM9DS0_CTRL_REG1_XM;
    dt[1] = acc_set_data.data_rate | LSM9_X_CONT_EN_XYZ;
    _i2c.write(acc_set_data.addr, dt, 2, false);
    // Control Reg #2
    dt[0] = LSM9DS0_CTRL_REG2_XM;
    dt[1] = acc_set_data.range | acc_set_data.anti_filter | LSM9_X_NORMAL_MODE;
    _i2c.write(acc_set_data.addr, dt, 2, false);
    //  Set Full scal factor
    switch (acc_set_data.range) {
        case LSM9_X_2G:
            fsx_factor = 0.061;
            break;
        case LSM9_X_4G:
            fsx_factor = 0.122;
            break;
        case LSM9_X_6G:
            fsx_factor = 0.183;
            break;
        case LSM9_X_8G:
            fsx_factor = 0.244;
            break;
        case LSM9_X_16G:
            fsx_factor = 0.732;
            break;
        default:
            fsx_factor = 1.0f;
    }
}

void LSM9DS0::set_initial_dt_to_regs_m(void)
{
    // Control Reg #5
    dt[0] = LSM9DS0_CTRL_REG5_XM;
    dt[1] = mag_set_data.data_rate | LSM9_M_HI_RESOL | LSM9_M_TMP_ENA;
    _i2c.write(acc_set_data.addr, dt, 2, false); // acc_set_data is not wrong!
    // Control Reg #6
    dt[0] = LSM9DS0_CTRL_REG6_XM;
    dt[1] = mag_set_data.range;
    _i2c.write(acc_set_data.addr, dt, 2, false); // acc_set_data is not wrong!
    // Control Reg #7
    dt[0] = LSM9DS0_CTRL_REG7_XM;
    dt[1] = LSM9_M_CONTINUOUS | LSM9_X_NORMAL;
    _i2c.write(acc_set_data.addr, dt, 2, false); // acc_set_data is not wrong!
    //  Set Full scal factor
    switch (mag_set_data.range) {
        case LSM9_M_2GAUSS:
            fsm_factor = 0.08;
            break;
        case LSM9_M_4GAUSS:
            fsm_factor = 0.16;
            break;
        case LSM9_M_8GAUSS:
            fsm_factor = 0.32;
            break;
        case LSM9_M_12GAUSS:
            fsm_factor = 0.48;
            break;
        default:
            fsm_factor = 1.0f;
    }
}

void LSM9DS0::set_initial_dt_to_regs_g(void)
{
    // Control Reg #1
    dt[0] = LSM9DS0_CTRL_REG1_G;
    dt[1] = LSM9_G_PD_DIS | LSM9_G_NORMAL | gyr_set_data.datarate_and_bandwidth;
    _i2c.write(gyr_set_data.addr, dt, 2, false);
    // Control Reg #4
    dt[0] = LSM9DS0_CTRL_REG4_G;
    dt[1] = gyr_set_data.range;
    _i2c.write(gyr_set_data.addr, dt, 2, false);
    //  Set Full scal factor
    switch (acc_set_data.range) {
        case LSM9_G_FS_245DPS:
            fsg_factor = 8.75;
            break;
        case LSM9_G_FS_500DPS:
            fsg_factor = 17.5;
            break;
        case LSM9_G_FS_2000DPS:
            fsg_factor = 70;
            break;
        default:
            fsg_factor = 1.0f;
    }
}

/////////////// Check Who am I? ///////////////////////////
void LSM9DS0::check_id_xm(void)
{
    dt[0] = LSM9DS0_WHO_AM_I_XM;
    _i2c.write(acc_set_data.addr, dt, 1, true);
    _i2c.read(acc_set_data.addr, dt, 1, false);
    if (dt[0] == I_AM_LSM9DS0_XM) {
        acc_ready = 1;
    } else {
        acc_ready = 0;
    }
}

void LSM9DS0::check_id_g(void)
{
    dt[0] = LSM9DS0_WHO_AM_I_G;
    _i2c.write(gyr_set_data.addr, dt, 1, true);
    _i2c.read(gyr_set_data.addr, dt, 1, false);
    if (dt[0] == I_AM_LSM9DS0_G) {
        gyr_ready = 1;
    } else {
        gyr_ready = 0;
    }
}

/////////////// Read data & normalize /////////////////////
void LSM9DS0::read_mg_acc(float *dt_usr)
{
    read_acc_raw(dt_usr);
}

void LSM9DS0::get_mg_acc(float *dt_usr)
{
    read_mg_acc(dt_usr);
}

void LSM9DS0::read_acc(float *dt_usr)
{
    read_acc_raw(dt_usr);
    dt_usr[0] = dt_usr[0] * GRAVITY / 1000;
    dt_usr[1] = dt_usr[1] * GRAVITY / 1000;
    dt_usr[2] = dt_usr[2] * GRAVITY / 1000;
}

void LSM9DS0::get_acc(float *dt_usr)
{
    read_acc(dt_usr);
}

void LSM9DS0::read_acc_raw(float *dt_usr)
{
    if (acc_ready == 0) {
        dt_usr[0] = 0;
        dt_usr[1] = 0;
        dt_usr[2] = 0;
        return;
    }
    // X,Y & Z
    // manual said that
    // In order to read multiple bytes, it is necessary to assert the most significant bit
    // of the subaddress field.
    // In other words, SUB(7) must be equal to ‘1’ while SUB(6-0) represents the address
    // of the first register to be read.
    dt[0] = LSM9DS0_OUT_X_L_A | 0x80;
    _i2c.write(acc_set_data.addr, dt, 1, true);
    _i2c.read(acc_set_data.addr, dt, 6, false);
    // data normalization
    dt_usr[0] = float((int16_t)(dt[1] << 8 | dt[0])) * fsx_factor;
    dt_usr[1] = float((int16_t)(dt[3] << 8 | dt[2])) * fsx_factor;
    dt_usr[2] = float((int16_t)(dt[5] << 8 | dt[4])) * fsx_factor;
}

void LSM9DS0::get_mag(float *dt_usr)
{
    read_mag(dt_usr);
}

void LSM9DS0::read_mag(float *dt_usr)
{
    if (acc_ready == 0) {
        dt_usr[0] = 0;
        dt_usr[1] = 0;
        dt_usr[2] = 0;
        return;
    }
    dt[0] = LSM9DS0_OUT_X_L_M | 0x80;
    _i2c.write(acc_set_data.addr, dt, 1, true);
    _i2c.read(acc_set_data.addr, dt, 6, false);
    // data normalization
    dt_usr[0] = float((int16_t)(dt[1] << 8 | dt[0])) * fsm_factor;
    dt_usr[1] = float((int16_t)(dt[3] << 8 | dt[2])) * fsm_factor;
    dt_usr[2] = float((int16_t)(dt[5] << 8 | dt[4])) * fsm_factor;
}

void LSM9DS0::get_gyr(float *dt_usr)
{
    read_gyr(dt_usr);
}

void LSM9DS0::read_gyr(float *dt_usr)
{
    if (gyr_ready == 0) {
        dt_usr[0] = 0;
        dt_usr[1] = 0;
        dt_usr[2] = 0;
        return;
    }
    dt[0] = LSM9DS0_OUT_X_L_G | 0x80;
    _i2c.write(gyr_set_data.addr, dt, 1, true);
    _i2c.read(gyr_set_data.addr, dt, 6, false);
    // data normalization
    dt_usr[0] = float((int16_t)(dt[1] << 8 | dt[0])) * fsg_factor / 1000;
    dt_usr[1] = float((int16_t)(dt[3] << 8 | dt[2])) * fsg_factor / 1000;
    dt_usr[2] = float((int16_t)(dt[5] << 8 | dt[4])) * fsg_factor / 1000;
}

/////////////// Check data ready or not  //////////////////
uint8_t LSM9DS0::data_ready_acc()
{
    if (acc_ready == 1) {
        dt[0] = LSM9DS0_STATUS_REG_A;
        _i2c.write(acc_set_data.addr, dt, 1, true);
        _i2c.read(acc_set_data.addr, dt, 1, false);
        if (!(dt[0] & 0x08)) {
            return 0;
        }
    }
    return 1;;
}

uint8_t LSM9DS0::data_ready_mag()
{
    if (acc_ready == 1) {
        dt[0] = LSM9DS0_STATUS_REG_M;
        _i2c.write(acc_set_data.addr, dt, 1, true);
        _i2c.read(acc_set_data.addr, dt, 1, false);
        if (!(dt[0] & 0x08)) {
            return 0;
        }
    }
    return 1;;
}

uint8_t LSM9DS0::data_ready_gyr()
{
    if (gyr_ready == 1) {
        dt[0] = LSM9DS0_STATUS_REG_G;
        _i2c.write(gyr_set_data.addr, dt, 1, true);
        _i2c.read(gyr_set_data.addr, dt, 1, false);
        if (!(dt[0] & 0x08)) {
            return 0;
        }
    }
    return 1;;
}

/////////////// Read ID ///////////////////////////////////
uint8_t LSM9DS0::read_acc_mag_id()
{
    dt[0] = LSM9DS0_WHO_AM_I_XM;
    _i2c.write(acc_set_data.addr, dt, 1, true);
    _i2c.read(acc_set_data.addr, dt, 1, false);
    return (uint8_t)dt[0];
}

uint8_t LSM9DS0::read_gyr_id()
{
    dt[0] = LSM9DS0_WHO_AM_I_G;
    _i2c.write(gyr_set_data.addr, dt, 1, true);
    _i2c.read(gyr_set_data.addr, dt, 1, false);
    return (uint8_t)dt[0];
}

/////////////// I2C Freq. /////////////////////////////////
void LSM9DS0::frequency(int hz)
{
    _i2c.frequency(hz);
}

/////////////// Read/Write specific register //////////////
uint8_t LSM9DS0::read_reg_acc_mag(uint8_t addr)
{
    if (gyr_ready == 1) {
        dt[0] = addr;
        _i2c.write(acc_set_data.addr, dt, 1, true);
        _i2c.read(acc_set_data.addr, dt, 1, false);
    } else {
        dt[0] = 0xff;
    }
    return (uint8_t)dt[0];
}

uint8_t LSM9DS0::write_reg_acc_mag(uint8_t addr, uint8_t data)
{
    if (gyr_ready == 1) {
        dt[0] = addr;
        dt[1] = data;
        _i2c.write(acc_set_data.addr, dt, 2, false);
    }
    return (uint8_t)dt[0];
}

uint8_t LSM9DS0::read_reg_gyr(uint8_t addr)
{
    if (gyr_ready == 1) {
        dt[0] = addr;
        _i2c.write(gyr_set_data.addr, dt, 1, true);
        _i2c.read(gyr_set_data.addr, dt, 1, false);
    } else {
        dt[0] = 0xff;
    }
    return (uint8_t)dt[0];
}

uint8_t LSM9DS0::write_reg_gyr(uint8_t addr, uint8_t data)
{
    if (gyr_ready == 1) {
        dt[0] = addr;
        dt[1] = data;
        _i2c.write(gyr_set_data.addr, dt, 2, false);
    }
    return (uint8_t)dt[0];
}
