#if 0
//??????????????????????????????????????????????????????????????????????????????
// Due to ROM & RAM size limitation
//??????????????????????????????????????????????????????????????????????????????

/*
 * mbed Application program for the mbed LPC1114FN28
 * Monitor program Ver.2 for only LPC1114FN28 
 *
 * Copyright (c) 2014,'20 Kenji Arai / JH1PJL
 *  http://www7b.biglobe.ne.jp/~kenjia/
 *  https://os.mbed.com/users/kenjiArai/
 *      Created: June       1st, 2014
 *      Revised: August     8th, 2020
 */
/*
 * Function
 *  Show Memory contents, Digital port, Analog input port, CPU clock and others in the mbed board
 *      Connection
 *          uart            USB Vertual com
 */

//  Include ---------------------------------------------------------------------------------------
#include "mbed.h"
#include "AT24C1024.h"		// Own lib. / EEPROM control
#include "mon_hw_config.h"
#include "redirect_stdio.h"

//  Object ----------------------------------------------------------------------------------------
extern I2C    xxi2c;			// SDA, SCL
extern AT24C1024   xat24c1024;	// Atmel 1Mbit EE-PROM

//  Definition ------------------------------------------------------------------------------------
// Define clocks
#define __XTAL				(12000000UL)    // Oscillator frequency
#define __SYS_OSC_CLK		(    __XTAL)    // Main oscillator frequency
#define __IRC_OSC_CLK		(12000000UL)    // Internal RC oscillator frequency

// Range check status
#define ERR_NOTHING         0
#define ERR_MODIFY_SIZ      1
#define ERR_OUT_OF_RANGE    2

// Reg. Size
#define SIZE8				32
#define SIZE16				16
#define SIZE_FULL			32
#define SIZE_X				32

// IO Cinfig
#define IO_ALL				0
#define IO_I2C				1

#define GETC(x)         	getc(x)
#define PUTC(x)				putc(x)
#define PRINTF(...)     	printf(__VA_ARGS__)
#define READABLE(x)     	readable(x)

typedef struct {
    unsigned long mstr;
    unsigned long msiz;
    unsigned long mtmp;
    unsigned long mold;
    unsigned char mflg;
    unsigned char mbhw;
} MEMO;

#define DW_CHAR				sizeof(char)
#define DW_SHORT			sizeof(short)
#define DW_LONG				sizeof(long)

typedef unsigned long	DWORD;
typedef unsigned short	WORD;
typedef unsigned char	BYTE;
typedef unsigned int	UINT;

//  RAM -------------------------------------------------------------------------------------------
//  Memory management
unsigned long SystemFrequency;

#if USE_MEM
//  RAM -------------------------------------------------------------------------------------------
static MEMO mem;

//  ROM / Constant data ---------------------------------------------------------------------------
//  Memory range data
const uint32_t mem_range[][2] = {          // Memory access range
    { 0x00000000, 0x00007fff },            // On-chip non-volatile memory     //32KB Flash memory
    { 0x10000000, 0x10000fff },            // On-chip SRAM                    //4KB local RAM
    { 0x1fff0000, 0x1fff3fff },            // Boot ROM                        //16KB Boot ROM
    { 0x40000000, 0x4007ffff },            // IO area
    { 0x50000000, 0x501fffff }             // IO area
};

static const char *const rmsg0 = "FLASH    ";
static const char *const rmsg1 = "SRAM     ";
static const char *const rmsg2 = "BOOT ROM ";
static const char *const rmsg3 = "IO       ";
static const char *const rmsg4 = "IO       ";
#endif  // USE_MEM

#if USE_EEP
//  RAM -------------------------------------------------------------------------------------------
static MEMO eepmem;

//  ROM / Constant data ---------------------------------------------------------------------------
const uint32_t eepmem_range[2] = { /* Memory access range */
	0x00000, 0x1ffff 	/* EEPROM 1Mbits 128Kbytes */
};
#endif

//  ROM / Constant data ---------------------------------------------------------------------------
static const char *const mon_msg_hw = "HW monitor only for mbed LPC1114FN28";

//  Function prototypes ---------------------------------------------------------------------------
extern char linebuf[];
extern int buf_size;

extern void put_rn ( void );
extern void put_r ( void );
extern void put_lin ( void );
extern void put_spc( uint8_t n);
extern void get_line (char *buff, int len);
extern int xatoi (char **str, unsigned long *res);

//-------------------------------------------------------------------------------------------------
//  Control Program
//-------------------------------------------------------------------------------------------------
//  Help Massage
void msg_hlp_hw (void){
    PRINTF(mon_msg_hw);
    put_rn();
#if USE_MEM
    PRINTF("m  - Entry Memory Mode");
    put_rn();
    PRINTF("m>? -> Help ");
    put_rn();
#endif   // USE_MEM
#if USE_EEP
    PRINTF("e -Enter EEPROM Memory Mode ");
    put_rn();
    PRINTF("e>? -> Help ");
    put_rn();
#endif   // USE_EEP
#if USE_REG_SPI_UART
    PRINTF("r  - Show SPI & UART Reg.");
    put_rn();
    PRINTF("r>? -> Help ");
    put_rn();
#endif   // USE_REG_SPI_UART
#if USE_PORT
    PRINTF("p  - Show port configration");
    put_rn();
#endif   // USE_PORT
#if USE_REG_I2C
    PRINTF("i  - Show I2C Reg.");
    put_rn();
#endif   // USE_REG_I2C
#if USE_SYS
    PRINTF("sf - System Clock");
    put_rn();
    PRINTF("sc - System / CPU information");
    put_rn();
#endif   // USE_SYS
#if USE_RPT
    PRINTF("/    [a],[p] commands every 1 sec  Exit =hit any key (not ENTER key) ");
    put_rn();
#endif
#if DEBUG
    PRINTF("x     Special command for Debug ");
    put_rn();
#endif   // DEBUG
    PRINTF("q  - Quit (back to called routine)");
    put_rn();
}

// No function
static void not_yet_impliment( void ){
	PRINTF( "Not implimented yet" );
	put_rn();
}

const char *const imsg2 = "-->Control Reg.";
const char *const imsg3 = "-->Status Reg.";
const char *const imsg4 = "-->Data Reg.";
//static const char *const imsg5 = "-->Baud rate Reg.";
//static const char *const imsg6 = "-->Own address Reg.";
const char   *const imsg7 = "-->Clock control Reg.";
//static const char *const imsg8 = "-->TRISE Reg.";

static const char *const io_port_name0 = "PIO0_";
static const char *const io_port_name1 = "PIO1_";
static const char *const iomsg0 = "Func->select ";
static const char *const iomsg1 = "IO";
static const char *const iomsg2 = "Reserved";
static const char *const iomsg30 = "B0_MAT";
static const char *const iomsg31 = "B1_MAT";
static const char *const iomsg4 = "Std/F-md I2C";
static const char *const iomsg5 = "func. R";
static const char *const iomsg6 = "D-Mode";
static const char *const iomsg7 = "A-Mode";

//  Show 16bit register contents
void reg_print(uint16_t size, uint16_t reg){
uint16_t i, j, k, n;

    if (size == 8){
        PRINTF(" 7, 6, 5, 4, 3, 2, 1, 0");
        put_rn();
        i = 8;
        n = 0x80;
    } else if (size == 16){
        PRINTF( "15,14,13,12,11,10, 9, 8, 7, 6, 5, 4, 3, 2, 1, 0" );
        put_rn();
        i = 16;
        n = 0x8000;
    } else {
        PRINTF("0x%08x", reg);
        return;
    }
    PUTC(' ');
    for (; i>0; i--){
        k = n >> (size-i);
        j = reg & k;
        if (j){
            PUTC('1');
        } else {
            PUTC('0');
        }
        PUTC(' ');
        PUTC(' ');
    }
    PRINTF("  (0x%04x)", reg);
}

#if USE_PORT
void io_mode(uint32_t reg){
    PRINTF("MODE->");
    switch (reg){
    case 0:
        PRINTF("Inactive");
        break;
    case 1:
        PRINTF("P-DWN");
        break;
    case 2:
        PRINTF("P-UP");
        break;
    case 3:
        PRINTF("Repeater");
        break;
    }
}

void io_hys(uint32_t reg){
    PRINTF("HIS->");
    switch (reg){
    case 0:
        PRINTF("Dis");
        break;
    case 1:
        PRINTF("Ena");
        break;
    }
}

void io_od(uint32_t reg){
    PRINTF("OD->");
    switch (reg){
    case 0:
        PRINTF("no OD");
        break;
    case 1:
        PRINTF("OD");
        break;
    }
}

void io_mode_hys_od(uint32_t reg){
    io_mode ((reg >>  3) & 0x3);
    put_lin();
    io_hys (( reg >>  5) & 0x1); 
    put_lin();
    io_od (( reg >> 10) & 0x1);
}

// I/O Config IO0_x
void io_config0(void){
uint32_t r0;

    // P0_0
    r0 = LPC_IOCON->RESET_PIO0_0;
    PRINTF("RESET_%s0(dp23)", io_port_name0);
    put_spc(1);
    reg_print( SIZE_X, r0 );
    put_spc(2);
    PRINTF( iomsg0 );
    if ((r0 & 0x7) == 0){	PRINTF("RESET");
    } else {				PRINTF( iomsg1 );    }
    put_lin();
    io_mode_hys_od( r0 );
    put_rn();
    // P0_1
    r0 = LPC_IOCON->PIO0_1;
    PRINTF("%s1(dp24)", io_port_name0);
    put_spc(3);
    reg_print( SIZE_X, r0 );
    put_spc(2);
    PRINTF( iomsg0 );
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);				break;
    case 1:	PRINTF("CLKOUT");			break;
    case 2:	PRINTF("32%s2", iomsg30);	break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn(); 
    // P0_2
    r0 = LPC_IOCON->PIO0_2;
    PRINTF("%s2(dp25)",io_port_name0);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);				break;
    case 1:	PRINTF("SSEL0");			break;
    case 2:	PRINTF("16B0_CAP0");		break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_3
    r0 = LPC_IOCON->PIO0_3;
    PRINTF("%s3(dp26)",io_port_name0);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);				break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_4
    r0 = LPC_IOCON->PIO0_4;
    PRINTF("%s4(dp27)",io_port_name0);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);				break;
    case 1:	PRINTF("SCL");				break;
    }
    put_lin();
    switch (( r0 >> 8 ) & 0x3){
    case 0:	PRINTF(iomsg4);				break;
    case 1:	PRINTF(iomsg1);				break;
    case 2:	PRINTF("Fast md");			break;
    case 3:	PRINTF(iomsg2 ;				break;
    }
    put_rn();
    // P0_5
    r0 = LPC_IOCON->PIO0_5;
    PRINTF("%s5(dp5)",io_port_name0);
    put_spc(4);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);				break;
    case 1:	PRINTF("SDA");				break;
    }
    put_lin();
    switch ( ( r0 >> 8 ) & 0x3 ){
    case 0:	PRINTF(iomsg4);	        	break;
    case 1:	PRINTF(iomsg1);	        	break;
    case 2:	PRINTF("Fast md");	        break;
    case 3:	PRINTF(iomsg2);	        	break;
    }
    put_rn();
    // P0_6
    r0 = LPC_IOCON->PIO0_6;
    PRINTF("%s6(dp6)", io_port_name0);
    put_spc(4);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);	        	break;
    case 1:	PRINTF(iomsg2);	        	break;
    case 2:	PRINTF("SCK0");		        break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();    
    // P0_7
    r0 = LPC_IOCON->PIO0_7;
    PRINTF("%s7(dp28)", io_port_name0);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0 ;
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);	        	break;
    case 1:	PRINTF("CTS");		        break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn(); 
    // P0_8
    r0 = LPC_IOCON->PIO0_8;
    PRINTF("%s8(dp1)", io_port_name0);
    put_spc(4);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);	        	break;
    case 1:	PRINTF("MISO0");	        break;
    case 2:	PRINTF("16%s0", iomsg30);	break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_9
    r0 = LPC_IOCON->PIO0_9;
    PRINTF("%s9(dp2)", io_port_name0);
    put_spc(4);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);	        	break;
    case 1:	PRINTF("MOSI0");	        break;
    case 2:	PRINTF("16%s1", iomsg30);	break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_10
    r0 = LPC_IOCON->SWCLK_PIO0_10;
    PRINTF("SWCLK_%s10(dp3)", io_port_name0);
    put_spc(1);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7 {
    case 0:	PRINTF("SWCLK");	        break;
    case 1:	PRINTF(iomsg1);	        	break;
    case 2:	PRINTF("SCK0");		        break;
    case 3:	PRINTF("16%s2", iomsg30);	break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P0_11
    r0 = LPC_IOCON->R_PIO0_11;
    PRINTF("R_%s11(dp4)", io_port_name0);
    put_spc(1);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF( iomsg0 );
    switch ( r0 & 0x7 ){
    case 0:	PRINTF(iomsg5);	        	break;
    case 1:	PRINTF(iomsg1);	        	break;
    case 2:	PRINTF("AD0");		        break;
    case 3:	PRINTF("32%s3", iomsg30);	break;
    }
    put_lin();
    io_mode_hys_od(r0);
    if ( r0 & 0x80 ){	PRINTF(", %s", iomsg6);
    } else {			PRINTF(", %s", iomsg7);
    } 
    put_rn();
}

// I/O Config IO1_x
void io_config1(void){
uint32_t r0;

    // P1_0
    r0 = LPC_IOCON->R_PIO1_0;
    PRINTF("R_%s0(dp9)", io_port_name1);
    put_spc(2);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF( iomsg0 );
    switch ( r0 & 0x7 ){
    case 0:	PRINTF(iomsg5);				break;
    case 1:	PRINTF(iomsg1);				break;
    case 2:	PRINTF("AD1");				break;
    case 3:	PRINTF("32B1_CAP0");		break;
    }
    io_mode_hys_od(r0);
    if (r0 & 0x80){		PRINTF(", %s", iomsg6);
    } else {			PRINTF(", %s", iomsg7);
    } 
    put_rn();
    // P1_1
    r0 = LPC_IOCON->R_PIO1_1;
    PRINTF("R_%s1(dp10)", io_port_name1);
    put_spc(1);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg5);	        	break;
    case 1:	PRINTF(iomsg1);	        	break;
    case 2:	PRINTF("AD2");		        break;
    case 3:	PRINTF("32%s0", iomsg31);	break;
    }
    io_mode_hys_od(r0);
    if (r0 & 0x80){		PRINTF(", %s", iomsg6);
    } else {			PRINTF(", %s", iomsg7);
    } 
    put_rn(); 
    // P1_2
    r0 = LPC_IOCON->R_PIO1_2;
    PRINTF("R_%s2(dp11)", io_port_name1);
    put_spc(1);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg5);	        	break;
    case 1:	PRINTF(iomsg1);	        	break;
    case 2:	PRINTF("AD3");		        break;
    case 3:	PRINTF("32%s1", iomsg31);	break;
    }
    io_mode_hys_od( r0 );
    if (r0 & 0x80){		PRINTF(", %s", iomsg6);
    } else {			PRINTF(", %s", iomsg7);
    }   
    put_rn(); 
    // P1_3
    r0 = LPC_IOCON->SWDIO_PIO1_3;
    PRINTF("SWDIO_%s3(dp12)",io_port_name1);
    put_spc(1);
    reg_print(SIZE_X, r0);
    put_spc(3);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF("SWDIO");	        break;
    case 1:	PRINTF(iomsg1);	        	break;
    case 2:	PRINTF("AD4");		        break;
    case 3:	PRINTF("32%s2", iomsg31);	break;
    }
    io_mode_hys_od(r0);
    if (r0 & 0x80){		PRINTF(", %s", iomsg6);
    } else {			PRINTF(", %s", iomsg7);
    }   
    put_rn();
    // P1_4
    r0 = LPC_IOCON->PIO1_4;
    PRINTF("%s4(dp13)",io_port_name1);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF( iomsg0 );
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);				break;
    case 1:	PRINTF("AD5");				break;
    case 2:	PRINTF("32%s3", iomsg31);	break;
    }
    put_lin();
    io_mode_hys_od(r0);
	put_rn();
    // P1_5
    r0 = LPC_IOCON->PIO1_5;
    PRINTF("%s5(dp14)",io_port_name1);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);				break;
    case 1:	PRINTF("RTS");				break;
    case 2:	PRINTF("32B0_CAP0");		break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P1_6
    r0 = LPC_IOCON->PIO1_6;
    PRINTF("%s6(dp15)", io_port_name1);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);				break;
    case 1:	PRINTF("RXD");				break;
    case 2:	PRINTF( "32%s0", iomsg30 );	break;
    }
    put_lin();
    io_mode_hys_od( r0 );
    put_rn();  
    // P1_7
    r0 = LPC_IOCON->PIO1_7;
    PRINTF("%s7(dp16)", io_port_name1);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);				break;
    case 1:	PRINTF("TXD");				break;
    case 2:	PRINTF("32%s1", iomsg30);	break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P1_8
    r0 = LPC_IOCON->PIO1_8;
    PRINTF("%s8(dp17)", io_port_name1);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);				break;
    case 1:	PRINTF("32%s1", iomsg30);	break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
    // P1_9
    r0 = LPC_IOCON->PIO1_9;
    PRINTF("%s9(dp18)", io_port_name1);
    put_spc(3);
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:	PRINTF(iomsg1);				break;
    case 1:	PRINTF("16%s0", iomsg31);	break;
    case 2:	PRINTF("MOSI1");			break;
    }
    put_lin();
    io_mode_hys_od(r0);
    put_rn();
}
#endif	// USE_PORT

#if USE_REG_SPI_UART
void spi_reg(int no){
uint32_t n, r0, r1 ,r2, r3, r4, r5, r6, r7, r8;

    if (no == SPI_0){
        n = 0;
        r0 = LPC_SSP0->CR1;
        r1 = LPC_SSP0->DR;
        r2 = LPC_SSP0->SR;
        r3 = LPC_SSP0->CPSR;
        r4 = LPC_SSP0->IMSC;
        r5 = LPC_SSP0->RIS;
        r6 = LPC_SSP0->MIS;
        r7 = LPC_SSP0->ICR;
        r8 = LPC_SSP0->CR0;
    } else if (no == SPI_1){
        n = 1;
        r0 = LPC_SSP1->CR1;
        r1 = LPC_SSP1->DR;
        r2 = LPC_SSP1->SR;
        r3 = LPC_SSP1->CPSR;
        r4 = LPC_SSP1->IMSC;
        r5 = LPC_SSP1->RIS;
        r6 = LPC_SSP1->MIS;
        r7 = LPC_SSP1->ICR;
        r8 = LPC_SSP1->CR0;        
    } else {
        return;
    }
    PRINTF("Show SSP%1d(SPI%1d) Registers", n, n);
    put_rn();
    PRINTF("CR0");
    PRINTF(imsg2);
    reg_print(SIZE8, r8);
    put_rn();
    PRINTF("CR1");
    PRINTF(imsg2);
    reg_print(SIZE8, r0);
    put_rn();
    PRINTF("DR");
    PRINTF(imsg4);
    reg_print(SIZE8, r1);
    put_rn();
    PRINTF("SR");
    PRINTF(imsg3);
    reg_print(SIZE8, r2);
    put_rn();
    PRINTF("CPSR");
    PRINTF(imsg7);
    reg_print(SIZE8, r3);
    put_rn();
    PRINTF("IMSC");
    PRINTF(imsg2);
    reg_print(SIZE8, r4);
    put_rn();
    PRINTF("RIS");
    PRINTF(imsg3);
    reg_print(SIZE8, r5);
    put_rn();
    PRINTF("MIS");
    PRINTF(imsg3);
    reg_print(SIZE8, r6);
    put_rn();
    PRINTF("ICR");
    PRINTF(imsg2);
    reg_print(SIZE8, r7);
    put_rn();
}

void uart_reg(void){
uint32_t r0,r1,r2,r3,r4,r5,r6;

    // clear LCR[DLAB] to read registers
    LPC_UART->LCR &= ~(1 << 7);
    r0 = LPC_UART->RBR;
    r1 = LPC_UART->IER;
    r2 = LPC_UART->IIR;
    r3 = LPC_UART->TER;   
    // set LCR[DLAB] to enable writing to divider registers
    LPC_UART->LCR |= (1 << 7);   
    r4 = LPC_UART->DLL;
    r5 = LPC_UART->DLM;
    r6 = LPC_UART->FDR;
    // clear LCR[DLAB]
    LPC_UART->LCR &= ~(1 << 7);
    // Print each register 
    PRINTF("Show UART Registers");   
    put_rn();
    PRINTF("RBR");
    PRINTF(imsg4);
    reg_print(SIZE8, r0);
    put_rn();
    PRINTF("THR--Write only");
    PRINTF(imsg4);
    put_rn();
    PRINTF("DLL");
    PRINTF(imsg2);
    reg_print(SIZE8, r4);
    put_rn();
    PRINTF("DLM");
    PRINTF(imsg2);
    reg_print(SIZE8, r5);
    put_rn();
    PRINTF("IER");
    PRINTF(imsg2);
    reg_print(SIZE8, r1);
    put_rn();
    PRINTF("IIR");
    PRINTF(imsg2);
    reg_print(SIZE8, r2);
    put_rn();
    PRINTF("FCR--Write only");
    PRINTF(imsg2);
    put_rn();
    PRINTF("LCR,MCR,MSR,SCR,ACR--Not support");
    put_rn();
    PRINTF("FDR");
    PRINTF(imsg2);
    reg_print(SIZE8, r6);
    put_rn();
    PRINTF("TER");
    PRINTF(imsg3);
    reg_print(SIZE8, r3);
    put_rn();
    PRINTF("RS485CTRL,485ADRMATCH,485DLY--Not support");
    put_rn();
}
#endif  //USE_REG_SPI_UART

#if USE_SYS
void cpu_inf (void){
    unsigned long m1, m2;

    m1 = SCB->CPUID;
    m2 = (m1 >> 24);
    if (m2 == 0x41) {
        put_r();
        PRINTF("CPU = ARM ");
    } else {
        put_r();
        PRINTF("CPU = NOT ARM ");
    }
    m2 = (m1 >> 4) & 0xfff;
    if (m2 == 0xc23) {
        PRINTF("Cortex-M3");
        put_rn();
    } else if (m2 == 0xc20){
        PRINTF("Cortex-M0");
        put_rn();
    } else {
        PRINTF("NOT Cortex-M3,M0");
        put_rn();
    }   
    m2 = (m1 >> 20) & 0x0f;
    PRINTF("Variant:%x", (int)m2);
    put_rn();
    m2 = m1 & 0x7;
    PRINTF("Revision:%x", (int)m2);
    put_rn();
}

//  Calculate CPU System Clock Frequency /refrence: system_LPCxx.c
static const char *const fmsg0 = "Internal RC Oscillator";
static const char *const fmsg1 = "Xtal Osc Clock";
static const char *const fmsg2 = "Watch dog Osc Clock";
static const char *const fmsg3 = "with PLL";
static const char *const fmsg4 = "System Clock =";
static const char *const fmsg5 = "PLL Post divider ratio =";
static const char *const fmsg6 = "feedback devider =";
static const char *const fmsg7 = "NO Clock ?!";

void get_freq_w_print (void){	// Get Core Clock Frequency
uint32_t wdt_osc = 0, pll_fct0 = 0, pll_fct1 = 0;

    // Determine clock frequency according to clock register values
    switch ((LPC_SYSCON->WDTOSCCTRL >> 5) & 0x0F) {
        case 0:  wdt_osc =  400000; break;
        case 1:  wdt_osc =  500000; break;
        case 2:  wdt_osc =  800000; break;
        case 3:  wdt_osc = 1100000; break;
        case 4:  wdt_osc = 1400000; break;
        case 5:  wdt_osc = 1600000; break;
        case 6:  wdt_osc = 1800000; break;
        case 7:  wdt_osc = 2000000; break;
        case 8:  wdt_osc = 2200000; break;
        case 9:  wdt_osc = 2400000; break;
        case 10: wdt_osc = 2600000; break;
        case 11: wdt_osc = 2700000; break;
        case 12: wdt_osc = 2900000; break;
        case 13: wdt_osc = 3100000; break;
        case 14: wdt_osc = 3200000; break;
        case 15: wdt_osc = 3400000; break;
    }
    wdt_osc /= ((LPC_SYSCON->WDTOSCCTRL & 0x1F) << 1) + 2;
        put_r();
    switch (LPC_SYSCON->MAINCLKSEL & 0x03) {
    case 0:                             // Internal RC oscillator 
        SystemCoreClock = __IRC_OSC_CLK;
        PRINTF("%s = %dHz", fmsg0, SystemCoreClock);
        break;
    case 1:                             // Input Clock to System PLL
        switch (LPC_SYSCON->SYSPLLCLKSEL & 0x03) {
        case 0:                         // Internal RC oscillator
            SystemCoreClock = __IRC_OSC_CLK;
            PRINTF("%s = %dHz", fmsg0, SystemCoreClock);
            break;
        case 1:                         // System oscillator
            SystemCoreClock = __SYS_OSC_CLK;
            PRINTF("%s = %dHz", fmsg1, SystemCoreClock );
            break;
        case 2:                         // WDT Oscillator
            SystemCoreClock = wdt_osc;
            PRINTF("%s = %dHz", fmsg2, wdt_osc );
            break;
        case 3:                         // Reserved
            SystemCoreClock = 0;
            PRINTF(fmsg7);
            break;
        }
        break;
    case 2:                             // WDT Oscillator
        SystemCoreClock = wdt_osc;
        PRINTF("%s = %dHz", fmsg2, wdt_osc );
        break;
    case 3:                             // System PLL Clock Out
        switch (LPC_SYSCON->SYSPLLCLKSEL & 0x03) {
        case 0:                         // Internal RC oscillator
            if (LPC_SYSCON->SYSPLLCTRL & 0x180) {
                SystemCoreClock = __IRC_OSC_CLK;
                PRINTF("%s = %dHz", fmsg0, SystemCoreClock);
            } else {
                pll_fct0 = (LPC_SYSCON->SYSPLLCTRL & 0x01F) + 1;
                pll_fct1 = (LPC_SYSCON->SYSPLLCTRL & 0x060) >> 5UL;
                SystemCoreClock = __IRC_OSC_CLK *  pll_fct0;
                PRINTF("Use Internal RC = %dHz", (int)__IRC_OSC_CLK);
                put_rn();
                PRINTF("%s %s = %dHz", fmsg0, fmsg3, SystemCoreClock);
                put_rn();
                PRINTF("%s %d, %s %d", fmsg5, pll_fct1, fmsg6, pll_fct0);
            }
            break;
        case 1:                         // System oscillator
            if (LPC_SYSCON->SYSPLLCTRL & 0x180) {
                SystemCoreClock = __SYS_OSC_CLK;
                PRINTF("%s = %dHz", fmsg1, SystemCoreClock );
            } else {
                pll_fct0 = (LPC_SYSCON->SYSPLLCTRL & 0x01F) + 1;
                pll_fct1 = (LPC_SYSCON->SYSPLLCTRL & 0x060) >> 5UL;
                SystemCoreClock = __SYS_OSC_CLK * pll_fct0;
                PRINTF("Use XTAL = %dHz", (int)__XTAL);
                put_rn();
                PRINTF("%s %s = %dHz", fmsg1, fmsg3, SystemCoreClock);
                put_rn();
                PRINTF("%s %d, %s %d", fmsg5, pll_fct1, fmsg6, pll_fct0);
            }
            break;
        case 2:                         // WDT Oscillator
            if (LPC_SYSCON->SYSPLLCTRL & 0x180) {
                SystemCoreClock = wdt_osc;
                PRINTF("%s = %dHz", fmsg2, wdt_osc );
            } else {
                pll_fct0 = (LPC_SYSCON->SYSPLLCTRL & 0x01F) + 1;
                pll_fct1 = (LPC_SYSCON->SYSPLLCTRL & 0x060) >> 5UL;
                SystemCoreClock = wdt_osc *  pll_fct0;
                PRINTF("Use WDT OSC = %dHz", wdt_osc);
                put_rn();
                PRINTF("%s %s = %dHz", fmsg2, fmsg3, SystemCoreClock);
                put_rn();
                PRINTF("%s %d, %s %d", fmsg5, pll_fct1, fmsg6, pll_fct0);
            }
            break;
        case 3:                         // Reserved
            SystemCoreClock = 0;
            PRINTF("fmsg7");
            break;
        }
        break;
    }
    SystemCoreClock /= LPC_SYSCON->SYSAHBCLKDIV;
    put_rn();
    PRINTF("%s %dHz", fmsg4, SystemCoreClock);
    put_rn();
}
#endif  //USE_SYS

void get_freq  (void){	// Get Core Clock Frequency
uint32_t wdt_osc = 0, pll_fct0 = 0, pll_fct1 = 0;

    // Determine clock frequency according to clock register values
    switch ((LPC_SYSCON->WDTOSCCTRL >> 5) & 0x0F) {
        case 0:  wdt_osc =  400000; break;
        case 1:  wdt_osc =  500000; break;
        case 2:  wdt_osc =  800000; break;
        case 3:  wdt_osc = 1100000; break;
        case 4:  wdt_osc = 1400000; break;
        case 5:  wdt_osc = 1600000; break;
        case 6:  wdt_osc = 1800000; break;
        case 7:  wdt_osc = 2000000; break;
        case 8:  wdt_osc = 2200000; break;
        case 9:  wdt_osc = 2400000; break;
        case 10: wdt_osc = 2600000; break;
        case 11: wdt_osc = 2700000; break;
        case 12: wdt_osc = 2900000; break;
        case 13: wdt_osc = 3100000; break;
        case 14: wdt_osc = 3200000; break;
        case 15: wdt_osc = 3400000; break;
    }
    wdt_osc /= ((LPC_SYSCON->WDTOSCCTRL & 0x1F) << 1) + 2;
    switch (LPC_SYSCON->MAINCLKSEL & 0x03) {
    case 0:                             // Internal RC oscillator
        SystemCoreClock = __IRC_OSC_CLK;
        break;
    case 1:                             // Input Clock to System PLL
        switch (LPC_SYSCON->SYSPLLCLKSEL & 0x03) {
        case 0:                         // Internal RC oscillator
            SystemCoreClock = __IRC_OSC_CLK;
            break;
        case 1:                         // System oscillator
            SystemCoreClock = __SYS_OSC_CLK;
            break;
        case 2:                         // WDT Oscillator
            SystemCoreClock = wdt_osc;
            break;
        case 3:                         // Reserved
            SystemCoreClock = 0;
            break;
        }
        break;
    case 2:                             // WDT Oscillator
        SystemCoreClock = wdt_osc;
        break;
    case 3:                             // System PLL Clock Out
        switch (LPC_SYSCON->SYSPLLCLKSEL & 0x03) {
        case 0:                         // Internal RC oscillator
            if (LPC_SYSCON->SYSPLLCTRL & 0x180) {
                SystemCoreClock = __IRC_OSC_CLK;
            } else {
                pll_fct0 = (LPC_SYSCON->SYSPLLCTRL & 0x01F) + 1;
                pll_fct1 = (LPC_SYSCON->SYSPLLCTRL & 0x060) >> 5UL;
                SystemCoreClock = __IRC_OSC_CLK *  pll_fct0;
            }
            break;
        case 1:                         // System oscillator
            if (LPC_SYSCON->SYSPLLCTRL & 0x180) {
                SystemCoreClock = __SYS_OSC_CLK;
            } else {
                pll_fct0 = (LPC_SYSCON->SYSPLLCTRL & 0x01F) + 1;
                pll_fct1 = (LPC_SYSCON->SYSPLLCTRL & 0x060) >> 5UL;
                SystemCoreClock = __SYS_OSC_CLK * pll_fct0;
            }
            break;
        case 2:                         // WDT Oscillator
            if (LPC_SYSCON->SYSPLLCTRL & 0x180) {
                SystemCoreClock = wdt_osc;
            } else {
                pll_fct0 = (LPC_SYSCON->SYSPLLCTRL & 0x01F) + 1;
                pll_fct1 = (LPC_SYSCON->SYSPLLCTRL & 0x060) >> 5UL;
                SystemCoreClock = wdt_osc *  pll_fct0;
            }
            break;
        case 3:                         // Reserved
            SystemCoreClock = 0;
            break;
        }
        break;
    }
    SystemCoreClock /= LPC_SYSCON->SYSAHBCLKDIV;
}

#if USE_REG_I2C
void i2c_io_reg (void){
uint32_t r0;

	PRINTF("<Show IO Pin>");
	put_rn();
    // P0_4
    r0 = LPC_IOCON->PIO0_4;
    reg_print(SIZE_X, r0);
    put_spc(2);
    PRINTF("%s4(dp27)",io_port_name0);
	put_spc(2);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:
        PRINTF(iomsg1);
        break;
    case 1:
        PRINTF("SCL");
        break;
    }
    put_lin();
    switch (( r0 >> 8 ) & 0x3){
    case 0:
        PRINTF(iomsg4);
        break;
    case 1:
        PRINTF(iomsg1);
        break;
    case 2:
        PRINTF("Fast md");
        break;
    case 3:
        PRINTF(iomsg2);
        break;
    }
    put_rn();
    // P0_5
    r0 = LPC_IOCON->PIO0_5;
    reg_print( SIZE_X, r0 );
    put_spc(2);
    PRINTF("%s5(dp5)",io_port_name0);
    put_spc(3);
    PRINTF(iomsg0);
    switch (r0 & 0x7){
    case 0:
        PRINTF(iomsg1);
        break;
    case 1:
        PRINTF("SDA");
        break;
    }
    put_lin();
    switch ((r0 >> 8 ) & 0x3){
    case 0:
        PRINTF(iomsg4);
        break;
    case 1:
        PRINTF(iomsg1);
        break;
    case 2:
        PRINTF("Fast md");
        break;
    case 3:
        PRINTF(iomsg2);
        break;
    }
    put_rn();
}
    
void i2c_reg (void){
uint32_t r0,r1,r2,r3,r4,r5,r6,r7;

    r0 = LPC_I2C->CONSET;
    r1 = LPC_I2C->STAT;
    r2 = LPC_I2C->DAT;
    r3 = LPC_I2C->SCLH;
    r4 = LPC_I2C->SCLL;
    r5 = LPC_I2C->CONCLR;
    r6 = LPC_I2C->MMCTRL;
    r7 = LPC_I2C->DATA_BUFFER;

    PRINTF("<Show I2C Registers>");
    put_rn();
    reg_print(SIZE8, r0);
	put_spc(2);
    PRINTF("CONSET");
    PRINTF(imsg2);
    put_rn();
    reg_print(SIZE8, r1);
	put_spc(2);
    PRINTF("STAT");
    PRINTF(imsg3);
    put_rn();
    reg_print(SIZE8, r2);
    put_spc(2);
    PRINTF("DAT");
    PRINTF(imsg4);
    put_rn();
    put_spc(2);
    PRINTF("ADR0--Not support");
    put_rn();
    reg_print(SIZE8, r3);
    put_spc(2);
    PRINTF("SCLH");
    PRINTF(imsg2);
    put_rn();
    reg_print(SIZE8, r4);
    put_spc(2);
    PRINTF("SCLL");
    PRINTF(imsg2);
    put_rn();
    reg_print(SIZE8, r5);
    put_spc(2);
    PRINTF("CONCLR");
    PRINTF(imsg2);
    put_rn();
    reg_print(SIZE8, r6);
    put_spc(2);
    PRINTF("MMCTRL");
    PRINTF(imsg2);
    put_rn();
    put_spc(2);
    PRINTF("ADR1,2,3--Not support");
    put_rn();
    reg_print(SIZE8, r7);
    put_spc(2);
    PRINTF("DATA_BUFFER");
    PRINTF(imsg7);
    put_rn();
    put_spc(2);
    PRINTF("MASK0,1,2,3--Not support");
    put_rn();
}

void i2c_freq (void){
uint32_t r0,r1;

    r0 = LPC_I2C->SCLH;
    r1 = LPC_I2C->SCLL;	
	get_freq();
	PRINTF("<I2C Status>");
	put_rn();
	PRINTF("Freq. = I2CPCLK/(SCLH+SCLL) = %d/(%d+%d) = %d Hz",
			SystemCoreClock, r0, r1, SystemCoreClock/(r0+r1));
	put_rn();
	r0 = LPC_I2C->CONSET;
	PRINTF("I2C I/F ");
	if ( r0 & 0x40 ) {
		PRINTF("Enabled");
	} else {
		PRINTF("disabled");
	}
	put_rn();
}
#endif

#if USE_MEM
//  Range check for Memory dump
static void check_range (MEMO * mem){
uint8_t    i;
uint32_t m;

    mem->mflg = ERR_NOTHING;
    for (i = 0 ; i < 5 ; i++) {
        if (mem->mstr >= mem_range[i][0]) {
            if (mem->mstr < mem_range[i][1]) {
                m = mem->mstr + mem->msiz;
                if (m < mem_range[i][1]) {
                    return;            // no modification
                } else {
                    m = mem_range[i][1];
                    mem->msiz = m - mem->mstr + 1;
                    mem->mflg = ERR_MODIFY_SIZ;
                    return;            // modified size
                }
            }
        }
    }
    mem->mflg = ERR_OUT_OF_RANGE;
    mem->mstr = 0;
    mem->msiz = 0;
    return ;
}

//  Memory dump error massage
void error_print (unsigned char flg){
    switch (flg) {
        case ERR_MODIFY_SIZ :
            put_r();
            PRINTF("Reach to out of range ");
            put_rn();
            break;
        case ERR_OUT_OF_RANGE :
            put_r();
            PRINTF("Not in a memory area ");
            put_rn();
            break;
        case ERR_NOTHING :
        default :
            ;
    }
}

//  Print memory contents
void put_dump (const unsigned char *buff, unsigned long ofs, int cnt){
int n;

    PRINTF("%08lX ", ofs);
    for(n = 0; n < cnt; n++) {	PRINTF(" %02X", buff[n]);    }
    PUTC(' ');
    for(n = 0; n < cnt; n++) {
        if ((buff[n] < 0x20)||(buff[n] >= 0x7F)) {	PUTC('.');
        } else {									PUTC(buff[n]);   }
    }
    put_rn();
}

// dump memory with error check
void dump_w_err_ckeck (char **ptr, MEMO * mem){
    check_range (mem);
    for (*ptr=(char*)mem->mstr; mem->msiz >= 16; *ptr += 16, mem->msiz -= 16) {
        put_r();
        put_dump((unsigned char*)*ptr, (unsigned int)*ptr, 16);
    }
    if (mem->msiz) {	put_dump((unsigned char*)*ptr, (unsigned int)*ptr, mem->msiz);
    }
    error_print(mem->mflg);
}

static void mem_inf (char *ptr){
    put_r();
    PRINTF("Mem. Mode d <address> [<count>], s, <ret> or f, q, ?");
    put_rn();
    mem.mstr = mem_range[0][0];     // default start address = Flash
    mem.msiz =256;
    mem.mold = 0;
    mem.mtmp = 0;
    mem.mflg = 0;
    for (; mem.mflg != 0xff;) {
        PRINTF("m>");
        ptr = linebuf;
        get_line(ptr, buf_size);
        put_r();                   
        switch(*ptr++){
            case 'd' :    // d <address> [<count>] - Dump memory
                mem.mtmp = mem.mstr;
                if (!xatoi(&ptr, &mem.mstr)) {
                    mem.mstr = mem.mtmp;
                }
                if (!xatoi(&ptr, &mem.msiz)) {
                    mem.msiz = 256;
                }
                mem.mtmp = mem.msiz;
                dump_w_err_ckeck(&ptr, &mem);
                mem.mold = mem.mstr;
                mem.mstr += mem.mtmp;
                break;
            case 'f' :        // next
            case 'n' :
            case 0x0d :
                mem.msiz = 256;
                mem.mtmp = mem.msiz;
                dump_w_err_ckeck(&ptr, &mem);
                mem.mold = mem.mstr;
                mem.mstr += 256;
                break;
            case 'q' :        // quit
                mem.mflg = 0xff;
                break;
            case 'b' :        // Back to more
                if (mem.mold == 0){
                    ;
                } else {
                    mem.mold -= 256;
                }
            case 'k' :        // keep previous address
                mem.mstr = mem.mold;
                mem.msiz = 256;
                mem.mtmp = mem.msiz;
                dump_w_err_ckeck(&ptr, &mem);
                mem.mstr += 256;
                break;
            case 'a' :        // start RAM top
                mem.mstr = mem_range[1][0];
                mem.msiz =256;
                mem.mold = 0;
                mem.mtmp = 0;
                mem.mflg = 0;
                dump_w_err_ckeck(&ptr, &mem);
                mem.mstr += 256;
                break;
            case 'o' :        // start ROM top
                mem.mstr = mem_range[0][0];
                mem.msiz =256;
                mem.mold = 0;
                mem.mtmp = 0;
                mem.mflg = 0;
                dump_w_err_ckeck(&ptr, &mem);
                mem.mstr += 256;
                break;
            case 's' :
                PRINTF("Memory Configuration");
                put_rn();
                PRINTF("%s0x%08lx to 0x%08lx ", rmsg0, mem_range[0][0], mem_range[0][1]);
                put_rn();
                PRINTF("%s0x%08lx to 0x%08lx ", rmsg1, mem_range[1][0], mem_range[1][1]);
                put_rn();
                PRINTF("%s0x%08lx to 0x%08lx ", rmsg2, mem_range[2][0], mem_range[2][1]);
                put_rn();
                PRINTF("%s0x%08lx to 0x%08lx ", rmsg3, mem_range[3][0], mem_range[3][1]);
                put_rn();
                PRINTF("%s0x%08lx to 0x%08lx ", rmsg4, mem_range[4][0], mem_range[4][1]);
                put_rn();
                break;
            case '?' :
                PRINTF("d <address> [<count>] - Dump memory");
                put_rn();
                PRINTF("s  - Show memory structure ");
                put_rn();
                PRINTF("o  - Dump memory / start from ROM top");
                put_rn();
                PRINTF("a  - Dump memory / start from RAM top");
                put_rn();
                PRINTF("k  - Dump memory / keep same 256bytes");
                put_rn();
                PRINTF("b  - Dump memory / before 256bytes");
                put_rn();
                PRINTF("<RET> or f, n - Dump memory / next 256bytes");
                put_rn();
                PRINTF("q  - Exit memory mode");
                put_rn();
                break;
            default:
                PUTC('?');
                put_rn();
        }
    }
    PRINTF("Return to Normal Mode");
}
#endif      //USE_MEM

#if USE_EEP
static void mem_chk_msg (char mflg){
	switch (mflg) {
		case ERR_MODIFY_SIZ :
			PRINTF(" Reached out of range");
			put_rn();
			break;
		case ERR_OUT_OF_RANGE :
			PRINTF(" Not in a memory area");
			put_rn();
			break;
		case ERR_NOTHING :
		default :
			;
	}
}

void put_eep_dump (
	uint32_t addr,		// Heading address value
	uint8_t len			// Number of items to be dumped
){
uint32_t i;
uint8_t	eep[16];
	//?????????????????????????????????????????????????????????????????????????
	#warning "?????"
	PRINTF("%08x ", (int)addr);		// address
	for (i = 0; i < len; i++) {		// Hexdecimal dump
		eep[i] = xat24c1024.read(addr++);
		PRINTF(" %02x", eep[i]);
	}
	PUTC(' ');
	for (i = 0; i < len; i++){		// ASCII dump
		PUTC((eep[i] >= ' ' && eep[i] <= '~') ? eep[i] : '.');
	}
	put_rn();
}

static void check_eeprange (MEMO * eepmem){
uint32_t m;

	m = sizeof(eepmem_range) / sizeof(*eepmem_range);
	eepmem->mflg = ERR_NOTHING;
	if (eepmem->mstr >= eepmem_range[0]){
		if (eepmem->mstr < eepmem_range[1]){
			m = eepmem->mstr + eepmem->msiz;
			if (m < eepmem_range[1]){
				return;			// no modification
			} else {
				m = eepmem_range[1];
				eepmem->msiz = m - eepmem->mstr + 1;
				eepmem->mflg = ERR_MODIFY_SIZ;
				return;			// modified size
			}
		}
	}
	eepmem->mflg = ERR_OUT_OF_RANGE;
	eepmem->mstr = 0;
	eepmem->msiz = 0;
	return ;
}

static void eepmem_inf (char *ptr){
uint8_t	c, dt;
unsigned long p1, p2, p3;

	put_r();
	PRINTF("Enter EEPROM 1) d <address> 256bytes, 2) s, 3) <ret>, 4) q 5) ?");
	put_rn();
	eepmem.mstr = eepmem_range[0];		// start head of table
	eepmem.msiz =256;
	eepmem.mold = 0;
	eepmem.mtmp = 0;
	eepmem.mflg = 0;
	eepmem.mbhw = DW_CHAR;
    for (; eepmem.mflg != 0xff;) {
		PRINTF("e>");
        ptr = linebuf;
        get_line(ptr, buf_size);
        put_r();                   
        switch(*ptr++){
			case 'd' :	// d <address> - Dump memory
				eepmem.mtmp = eepmem.mstr;		// save defult start addres
				if (!xatoi(&ptr, &eepmem.mstr)){
					eepmem.mstr = eepmem.mtmp;	// recover defult address
				}
				eepmem.msiz = 256;
				eepmem.msiz /= eepmem.mbhw;
				eepmem.mtmp = eepmem.msiz * eepmem.mbhw;
				check_eeprange (&eepmem);
				for (ptr=(char*)eepmem.mstr;
					 eepmem.msiz >= 16/eepmem.mbhw;
					 ptr += 16, eepmem.msiz -= 16/eepmem.mbhw)
				{
					put_eep_dump((UINT)ptr, 16/eepmem.mbhw);
				}
				if (eepmem.msiz){
					put_eep_dump((UINT)ptr, eepmem.msiz);
				}
				mem_chk_msg( eepmem.mflg );
				eepmem.mold = eepmem.mstr;
				eepmem.mstr += eepmem.mtmp;
				break;
            case 'f' :  // next
            case 'n' :
			case 0x0d :	// CR
				eepmem.msiz = 256/eepmem.mbhw;
				eepmem.mtmp = eepmem.msiz;
				check_eeprange (&eepmem);
				for (ptr=(char*)eepmem.mstr;
					 eepmem.msiz >= 16/eepmem.mbhw;
					 ptr += 16, eepmem.msiz -= 16/eepmem.mbhw)
				{
					put_eep_dump((UINT)ptr, 16/eepmem.mbhw);
				}
				mem_chk_msg(eepmem.mflg);
				eepmem.mold = eepmem.mstr;
				eepmem.mstr += 256;
				break;
			case 'q' :	// quit
                eepmem.mflg = 0xff;
                break;
			case 'b' :	// Back to more
                if (eepmem.mold == 0){
                    ;
                } else {
                    eepmem.mold -= 256;
                }
            case 'k' :	// keep previous address
				eepmem.mstr = eepmem.mold;
				eepmem.msiz = 256/eepmem.mbhw;
				eepmem.mtmp = eepmem.msiz;
				check_eeprange (&eepmem);
				for (ptr=(char*)eepmem.mstr; eepmem.msiz >= 16/eepmem.mbhw; ptr += 16,
					eepmem.msiz -= 16/eepmem.mbhw){
					put_eep_dump((UINT)ptr, 16/eepmem.mbhw);
				}
				mem_chk_msg(eepmem.mflg);
				eepmem.mstr += 256;
				break;
			case 'e' :	// e <address> [<value> ...] - Edit memory
				ptr = linebuf;
        		get_line(ptr, buf_size);
				p3 = DW_CHAR;
				if (!xatoi(&ptr, &p1)){
					PRINTF("? ->e <address> [<value> ...]");
					put_rn();
					break;	// Get start address
				}
				if (xatoi(&ptr, &p2)) { // 2nd parameter is given
					do {
						xat24c1024.write((uint16_t)p1, (uint8_t)p2);
						ThisThread::sleep_for(10ms);
						dt = xat24c1024.read((uint16_t)p1);
						if (dt != (uint8_t)p2){
							PRINTF("?\r\n");
						}
						p1 += p3;
					} while (xatoi(&ptr, &p2));	// Get next value
					break;
				}
				for (;;) {// 2nd parameter is not given (interactive mode)
					PRINTF("%08x 0x%02x-", (int)p1, *(BYTE*)p1);
					ptr = linebuf;
	        		get_line(ptr, buf_size);
					if (*ptr == '.'){ break;}
					if (*ptr == 0x1b){ break;}
					if ((BYTE)*ptr >= ' ') {
						if (!xatoi(&ptr, &p2)){ continue;}
						xat24c1024.write((uint16_t)p1, (uint8_t)p2);
						ThisThread::sleep_for(10ms);
						dt = xat24c1024.read((uint16_t)p1);
						if (dt != (uint8_t)p2){
							PRINTF("?\r\n");
						}
					}
					p1 += p3;
				}
				break;
			case 's' :
				PRINTF("AT24C1024B");
				put_rn();
	//?????????????????????????????????????????????????????????????????????????
	#warning "?????"
				PRINTF("EEPROM  0x%08lx to 0x%08lx", (int)eepmem_range[0], (int)eepmem_range[1]);
				put_rn();
				break;
			case '?' :
				PRINTF("d <address> -Dump memory");
				PRINTF(" (e.g. >d 0x10f00<RET>)");
				put_rn();
				PRINTF("e <address> [<value> ...] - Edit memory");
				put_rn();
				PRINTF("s -Show memory structure");
				put_rn();
				PRINTF("<RET>-Dump memory, next 256 bytes (also 'f','n')");
				put_rn();
				PRINTF("k -Dump memory, same as before 256 bytes ('b' -256)");
				put_rn();
				PRINTF("q -Exit EEPROM mode" );
				put_rn();
				break;
			default:
                PUTC('?');
                put_rn();
		}
	}
    PRINTF("Return to Normal Mode");
}
#endif

//-----------------------------------------------------------------------------
//  Monitor Main Program
//-----------------------------------------------------------------------------
int mon_hw (void) {
char *ptr;

	put_r();
    PRINTF("%s [Help:'?' key]", mon_msg_hw);
    put_rn();
#if USE_SYS
    get_freq();
#endif      //USE_SYS
    for (;;) {
        put_r();
        PUTC('>');
        ptr = linebuf;
        get_line(ptr, buf_size);
        switch (*ptr++) {
#if USE_MEM
    //---------------------------------------------------------------------------------
    //    Memory
    //---------------------------------------------------------------------------------
            case 'm' :
				mem_inf(ptr);
                put_rn();
                break;
#endif   // USE_MEM
#if USE_EEP
    //---------------------------------------------------------------------------------
    //    EEPROM Memory
    //---------------------------------------------------------------------------------
		/*	EEPROM Memory														*/
			case 'e' :
				eepmem_inf(ptr);
				put_rn();
				break;
#endif   // USE_EEP	
#if USE_REG_SPI_UART
    //--------------------------------------------------------------------------------------
    //    Register
    //--------------------------------------------------------------------------------------
            case 'r' :
uint8_t  r_flg;
                put_r();
                PRINTF("Reg. Mode p,u,i,s,t,a,d,l,w,c & ?");
                put_rn();
                r_flg = 0;
                for (; r_flg != 0xff;) {
                    PRINTF("r>");
                    ptr = linebuf;
                    get_line(ptr, sizeof(linebuf));
        			put_r();
                    switch(*ptr++){
                        case 'p' :	
	                    switch(*ptr++){
	                        case '0' :
                                io_config0(); 
	                            break;
	                        case '1' :
                                io_config1();                      
	                            break;
	                        case '*' :
                                io_config0(); 
                                put_rn();
                                io_config1(); 
	                            break;
	                        case '?' :
	                        default:
								PRINTF("Enter p0,p1 and p* for all");
								put_rn();
                            }
                            break;; 
                        case 'u' :
                            uart_reg();
                            break;
                        //case 'i' :
                        //    i2c_reg();
                        //    break;
                        case 's' :
	                    switch(*ptr++){
	                        case '0' :
                                spi_reg(SPI_0);
	                            break;
	                        case '1' :
                                spi_reg(SPI_1);                      
	                            break;
	                        case '*' :
                                spi_reg(SPI_0);
                                put_rn();
                                spi_reg( SPI_1 );                        
	                            break;
	                        case '?' :
	                        default:
								PRINTF("Enter s0,s1 and s* for all");
								put_rn();
                            }
                            break;
                        case 't' :		// 
                        	not_yet_impliment();                        
                            break;                            
                        case 'a' :		// 
                        	not_yet_impliment();                        
                            break;
                        case 'd' :		// 
                        	not_yet_impliment();                        
                            break;
                        case 'w' :		// 
                        	not_yet_impliment();                       
                            break;
                        case 'l' :        //
                        	not_yet_impliment();
                            break;
                        case 'c' :		// 
                        	not_yet_impliment();                        
                            break;
                        case 'x' :		// 
                        	not_yet_impliment();                        
                            break;
                        case 'y' :		// 
                        	not_yet_impliment();                        
                            break;
                        case '?' :
                            PRINTF("p - I/O Pin Config");
                            put_rn();
                            PRINTF("u - UART");
                            put_rn();
                            PRINTF("i - I2C -> separate command 'i'");
                            put_rn();
                            PRINTF("s - SPI");
                            put_rn();
                            PRINTF("t - TIMER");
                            put_rn();
                            PRINTF("a - ADC");
                            put_rn();
                            PRINTF("d - DAC");
                            put_rn();
                            PRINTF("l - LDC");
                            put_rn();
                            PRINTF("w - WWDG");
                            put_rn();
                            PRINTF("c - COMP");
                            put_rn();
                            break;
                        case 'q' :        // quit
                            r_flg = 0xff;
                            break;
                        default:
                        	PUTC('?');
                            put_rn();
                    }
                }
				PRINTF("Return to Normal Mode");
                put_rn();
                break;            
#endif   // USE_REG_SPI_UART
#if USE_PORT
    //---------------------------------------------------------------------------------
    //    Port configration
    //---------------------------------------------------------------------------------
            case 'p' :    //
            	put_r();
                switch(*ptr++){
                    case '0' :
                        io_config0(); 
                        break;
                    case '1' :
                        io_config1();                      
                        break;
                    case '*' :
                        io_config0(); 
                        put_rn();
                        io_config1(); 
                        break;
                    case '?' :
                    default:
						PRINTF("Enter p0,p1 and p* for all");
						put_rn();
                }              
                break;
#endif	// USE_PORT
#if USE_REG_I2C
    //---------------------------------------------------------------------------------
    //    I2C register
    //---------------------------------------------------------------------------------
            case 'i' :    // 
                put_r();
            	i2c_io_reg();
                i2c_reg();
                i2c_freq();               
                break;
#endif	// USE_REG_I2C
#if USE_SYS
    //---------------------------------------------------------------------------------
    //    System
    //---------------------------------------------------------------------------------
            case 's' :    // System related information
                switch (*ptr++) {
                    case 'f' :    // sc - show system clock frequency
                        get_freq_w_print();
                        break;
                    case 'c' :    // sc - show system CPU information
                        cpu_inf();
                        break;
                    case '?' :
                    default:
                    	put_r();
                        PRINTF("sc - System CPU information");
                        put_rn();
                        PRINTF("sf - System Clock");
                        put_rn();
                        break;
                }
                break;
#endif   // USE_SYS
    //---------------------------------------------------------------------------------
    //    Help
    //---------------------------------------------------------------------------------
            case '?' :
                put_r();
                msg_hlp_hw();
                break;
    //---------------------------------------------------------------------------------
    //    Return to main routine
    //---------------------------------------------------------------------------------
            case 'q' :        // Quit
                PRINTF("\rReturn to monitor ");
                return 0;
    //---------------------------------------------------------------------------------
    //    Special command for DEBUG
    //---------------------------------------------------------------------------------
#if DEBUG
            case 'x' :
				not_yet_impliment();
                break;
#endif
        }
    }
}

#endif