/*
 * Mbed Library
 *  example:
 *      Akizuki AE-KIT45-KEYPAD4X3
 *      http://akizukidenshi.com/catalog/g/gK-12229/
 *
 * Copyright (c) 2017,'20 Kenji Arai / JH1PJL
 *  http://www7b.biglobe.ne.jp/~kenjia/
 *  https://os.mbed.com/users/kenjiArai/
 *      Created:    September 27th, 2017
 *      Revised:    April     13th, 2020
 */
// ---- Merged below and added 4x5=20, 5x5=25 library --------------------------

/*
 * 4x4 Fork: Henri Clarke
 *          May 17th, 2018
 *      https://os.mbed.com/users/microHenri/code/Keypad/
 */

/*
 *  Performance
 *      every 2ms runs XuS for key detection(5x5=25 keys)
 *          on Nucleo-F446RE ->X=10uS, CPU occupancy is around 0.5%
 *          on Nucleo-L152RE ->X=50uS, CPU occupancy is around 2.5%
 *          on FRDM-K64F     ->X=28uS, CPU occupancy is around 1.4%
 */

#ifndef KEYPAD_H
#define KEYPAD_H

#include "mbed.h"

#define  CHNG_CNT       4

//------------- ASCII CODE ---------------
#define NUL     0x00
#define SOH     0x01
#define STX     0x02
#define ETX     0x03
#define EOT     0x04
#define ENQ     0x05
#define ACK     0x06
#define BEL     0x07
#define BS      0x08
#define HT      0x09
#define LF      0x0a
#define VT      0x0b
#define FF      0x0c
#define CR      0x0d
#define SD      0x0e
#define SI      0x0f
#define DLE     0x10
#define DC1     0x11
#define DC2     0x12
#define DC3     0x13
#define DC4     0x14
#define NAK     0x15
#define SYN     0x16
#define ETB     0x17
#define CAN     0x18
#define EM      0x19
#define SUB     0x1a
#define ESC     0x1b
#define FS      0x1c
#define GS      0x1d
#define RS      0x1e
#define US      0x1f
#define SPC     0x20

/**
 * @code
 * #include "mbed.h"
 * #include "Keypad.h"
 *
 * //       output port  X  Y  Z
 * //       Input A      *  0  #
 * //       Input B      7  8  9
 * //       Input C      4  5  6
 * //       Input D      1  2  3
 * //          X    Y   Z   A   B   C   D   OUT(XYZ), IN(ABCD)
 * Keypad key(D10, D9, D8, D7, D6, D5, D4);
 *
 * // define key number at main routine
 * char *const key_table = "?*7410852#963";  // key_table[0]=? is not used!
 *
 * int main() {
 *     uint32_t key_num;
 *     while(true) {
 *         while ((key_num = key.read()) != 0){
 *             printf("%c\r\n", *(key_table + key_num));
 *         }
 *         wait(1.0);
 *     }
 * }
 * @endcode
 */

class Keypad
{
public:
    /** 4x3 keypad interface:
     *  @param key input port           A,B,C,D
     *  @param key output(scan) port    X,Y,Z
     */
    Keypad(PinName kx, PinName ky, PinName kz,
           PinName ka, PinName kb, PinName kc, PinName kd);

    /** 4x4 keypad interface:
     *  @param key input port           A,B,C,D
     *  @param key output(scan) port    X,Y,Z,W
     */
    Keypad(PinName kx, PinName ky, PinName kz, PinName kw,
           PinName ka, PinName kb, PinName kc, PinName kd);

    /** 5x4 keypad interface:
     *  @param key input port           A,B,C,D,E
     *  @param key output(scan) port    X,Y,Z,W
     */
    Keypad(PinName kx, PinName ky, PinName kz, PinName kw,
           PinName ka, PinName kb, PinName kc, PinName kd, PinName ke);

    /** 5x5 keypad interface:
     *  @param key input port           A,B,C,D,E
     *  @param key output(scan) port    X,Y,Z,W,V
     */
    Keypad(PinName kx, PinName ky, PinName kz, PinName kw, PinName kv,
           PinName ka, PinName kb, PinName kc, PinName kd, PinName ke);

    /** Read key data into buffer
     *  @param none
     *  @return key number by ASCII code
     */
    uint8_t read(void);

    /** Read key ON/OFF state
     *  @param none
     *  @return ON(true) or OFF(false)
     */
    bool read_state(uint8_t key_num);

protected:
#define BF_SIZE     32
#define NUM         5

    DigitalIn     *k_in[NUM];
    DigitalInOut  *k_out[NUM];
    Ticker        tk;

    // key control
    enum State {OFF_state, OFF_to_ON_transient, ON_state, ON_to_OFF_transient};
    volatile State  key_state[NUM][NUM];
    volatile int8_t key_transent_cntr[NUM][NUM];
    void key_scan(void);
    void initialize(void);

    // mode control
    uint8_t key_mode;
    uint8_t key_in_num;
    uint8_t key_out_num;

    // buffer control
    uint8_t read_addr;
    uint8_t write_addr;
    uint8_t buf[BF_SIZE];
    void bf_put(char dat);
    int8_t bf_get(void);

};

#endif // KEYPAD_H
