/*
    for u-blox UBX-M8030-KT
    https://www.u-blox.com/en/product/ubx-m8030-series

    original
        https://os.mbed.com/teams/Multi-Hackers/code/GpsParser/
    modified by Kenji Arai
        http://www.page.sannet.ne.jp/kenjia/index.html
        http://mbed.org/users/kenjiArai/
            Created:    March      7th, 2019
            Revised:    March     16th, 2019

    tested on Nulceo-F446RE and GT-902PMGG(YUECHUNG INTERNATIONAL CORP.)
    http://akizukidenshi.com/catalog/g/gM-12905/
 */

#ifndef GPSPARSER_H
#define GPSPARSER_H

#include "mbed.h"

#define START_THREAD 1

#define GPS_DATA_BUFF_SIZE  128

/**
 * @typedef struct latitude
 * @brief latitude degrees, minutes, seconds structure + decimal
 * range of degrees : 0 to 90 as signed (negative degrees is South)
 * range of minutes : 0 to 60 as unsigned
 * Range of seconds : 0 to 60 as float
 * Range of decimal : deg + min +sec as float
 */
typedef struct {
    int8_t degrees;
    uint8_t minutes;
    float seconds;
    float decimal;
} latitude;

/**
 * @typedef struct longitude
 * @brief longitude degrees, minutes, seconds structure negative degrees is West
 * range of degrees : 0 to 180 as signed (negative degrees is West)
 * range of minutes : 0 to 60 as unsigned
 * Range of seconds : 0 to 60 as float
 * Range of decimal : deg + min +sec as float
 */
typedef struct {
    int16_t degrees;
    uint8_t minutes;
    float seconds;
    float decimal;
} longitude;

/**
 * @typdef satellite_prn
 * @brief satellite prn list, up to 12 valid
 */
typedef uint8_t satellite_prn[12];

/**
 * @typedef struct movement
 * @brief movement of object (speed and direction)
 * range of speed : km/h
 * range of direction : degrees
 */
typedef struct {
    float speed;
    float direction;
    bool  suspend;
} movement;

class GPSPARSER
{
public:
    /** Create the GPSPARSER object
     *  @param PinName GPS TX & RX
     *  @param Baud for GPS
     */
    GPSPARSER(PinName tx, PinName rx, int Baud);

    /** destroy the GPSPARSER object
     */
    ~GPSPARSER(void);

    /** GPS device detected
     *  @return boolean whether GPS device detected
     */
    bool gpsDetected(void);

    /** get each raw pakect
     *  @return last valid NMEA string data
     */
    bool getGGA(char *gga);
    bool getGLL(char *gll);
    bool getGSA(char *gsa);
    bool getRMC(char *rmc);
    bool getVTG(char *vtg);

    /** get longitude structure
     *  @return last valid NMEA string longitude data
     */
    void getLongitude(longitude *lng);

    /** get latitude structure
     *  @return last valid NMEA string latitude data
     */
    void getLatitude(latitude *lat);

    /** get timestamp structure
     *  @return last valid NMEA string time and date data
     *  uses tm struc from ctime standard library
     */
    struct tm getTimestamp(void);

    /** Mbed RTC (inside CPU chip) interface
     */
    void set_gps_time_to_mbed_rtc(float local_offser_hr);
    time_t check_rtc_and_gps_time();

    /** get GPS Lock status
     *  @return boolean of last valid NMEA string lock status data
     */
    bool getLockStatus(void);

    /** get GPS Fix status
     *  @return last valid NMEA Fix status data
     *  1 = no fix
     *  2 = 2D fix - latitude, longitude, time only valid
     *  3 = 3D fix - all data valid
     */
    uint8_t getFixStatus(void);

    /** get GPS Fix qualty
     *  @return last valid NMEA string Fix Quality data
     *  0 = invalid
     *  1 = GPS fix (SPS)
     *  2 = DGPS fix
     *  3 = not defined
     *  4 = Real Time Kinematic
     *  5 = Float RTK
     *  6 = estimated (dead reckoning) (2.3 feature)
     *  7 = not defined
     *  8 = not defined
     *  9 = not defined
     */
    uint8_t getFixQuality(void);

    /** get number of visible satellites
     *  @return last valid NMEA string number of satellites visible
     */
    uint8_t getNumSatellites(void);

    /** get calculated altitude
     *  @return last valid NMEA string altitude data in meters
     */
    int16_t getAltitude(void);

    /** get speed and direction
     *  @return last valid NMEA string speed[km/h] and direction[degrees]
     */
    void getMovement(movement *mv);

    /** get available satellite ID list
     *  @return last valid NMEA string satellite id
     */
    void getSat_id_list(satellite_prn list);

private:
    RawSerial _gps;
    Thread _getSentenceThread;
    Mutex _mutex;

    void rx_handler(void);
    bool chk_checksum(char *ptr, uint32_t len);
    void parse_main(char *ptr);
    bool copy_to_buf(char *p1, char *p2, uint32_t len);

    static void startSentenceThread (void const *p);
    void readNemaSentence (void);
    void parseGGA(char *nema_buf);
    void parseGSA(char *nema_buf);
    void parseGSV(char *nema_buf);
    void parseRMC(char *nema_buf);
    void parseVTG(char *nema_buf);
    void parseGLL(char *nema_buff);
    void parseZDA(char *nema_buff);

    char *find_comma(char *p, uint32_t n);
    char *next_period(char *p);

    bool _gps_status;
    uint8_t _fix_status;
    uint8_t _fix_quality;
    uint8_t _num_satellites[5];
    int8_t _taker_id;
    satellite_prn _satellite_prn;
    int16_t _msl_altitude;
    uint32_t _message_count;
    time_t _time_offset;
    latitude _gps_latitude;
    longitude _gps_longitude;
    struct tm _timestamp;
    movement _move;
    char _nmea_buff[2][256];
    char _temp_str[6];
    char _gga_buff[GPS_DATA_BUFF_SIZE];
    char _gll_buff[GPS_DATA_BUFF_SIZE];
    char _gsa_buff[GPS_DATA_BUFF_SIZE];
    char _rmc_buff[GPS_DATA_BUFF_SIZE];
    char _vtg_buff[GPS_DATA_BUFF_SIZE];

    bool _gps_detected;
    CircularBuffer<char, 256> rxbuf;
    uint32_t _rx_count;
    uint32_t _nmea_buff_count;
    bool _one_paket_full;
    bool _rtc_set_flag;
};
#endif

/***************** REFRENCE Program *******************************************/
//  https://os.mbed.com/teams/Multi-Hackers/code/GpsParser/
/**
 * @file    GpsParser.h
 * @brief   NMEA String to Packet Parser - NMEA strings to compact packet data
 * @author  Tim Barr
 * @version 1.0
 * @see     http://www.kh-gps.de/nmea.faq
 * @see     http://www.catb.org/gpsd/NMEA.html
 * @see     http://www.skytraq.com.tw/products/Venus638LPx_PB_v3.pdf
 *
 * Copyright (c) 2015
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
