/*
    for u-blox UBX-M8030-KT
    https://www.u-blox.com/en/product/ubx-m8030-series

    original
        https://os.mbed.com/teams/Multi-Hackers/code/GpsParser/
    modified by Kenji Arai
        http://www.page.sannet.ne.jp/kenjia/index.html
        http://mbed.org/users/kenjiArai/
            Created:    March      7th, 2019
            Revised:    March     16th, 2019

    tested on Nulceo-F446RE and GT-902PMGG(YUECHUNG INTERNATIONAL CORP.)
    http://akizukidenshi.com/catalog/g/gM-12905/
 */

#include "GpsParser.h"

//    RUN ONLY ON mbed-os5
#if (MBED_MAJOR_VERSION != 5)
#error "Please use Mbed-os5"
#endif

GPSPARSER::GPSPARSER(PinName tx, PinName rx, int Baud)
    : _getSentenceThread(&GPSPARSER::startSentenceThread,this),
      _gps(tx, rx)
{
    _gps.baud(Baud);
    _gps.attach(this,&GPSPARSER::rx_handler, Serial::RxIrq);

    _gps_latitude.degrees = 0;
    _gps_latitude.minutes = 0;
    _gps_latitude.seconds = 0;
    _gps_longitude.degrees = 0;
    _gps_longitude.minutes = 0;
    _gps_longitude.seconds = 0;
    _move.speed = 0;
    _move.direction = 0;
    _move.suspend = true;
    _timestamp.tm_sec = 0;
    _timestamp.tm_min = 0;
    _timestamp.tm_hour = 0;
    _timestamp.tm_mday = 0;
    _timestamp.tm_mon = 0;
    _timestamp.tm_year = 0;
    _timestamp.tm_wday = 0;
    _timestamp.tm_yday = 0;
    _timestamp.tm_isdst = -1; // time is UTC so no Daylight savings time

    _gps_status = false;
    _fix_status = 1;
    for (uint32_t i=0; i <5; i++) {
        _num_satellites[i] = 0;
    }
    _gps_detected = false;
    _message_count = 0;

    _rx_count = 0;
    _nmea_buff_count = 0;
    _one_paket_full = false;

    _getSentenceThread.signal_set(START_THREAD);

    return;
}

GPSPARSER::~GPSPARSER(void)
{
    if (_gps_detected) {
        _getSentenceThread.terminate();
    }
}

// Interrupt Hander
void GPSPARSER::rx_handler(void)
{
    static uint32_t rx_isr_count = 0;

    while(_gps.readable()) {
        rxbuf.push(_gps.getc());
        ++rx_isr_count;
    }
    if ((rx_isr_count % 16) == 0) {
        _getSentenceThread.signal_set(START_THREAD);
    }
}

void GPSPARSER::startSentenceThread(void const *p)
{
    GPSPARSER *instance = (GPSPARSER*)p;
    instance->readNemaSentence();
}

void GPSPARSER::readNemaSentence (void)
{
    char c;
    uint32_t pointer;

    while(true) {
        _getSentenceThread.signal_wait(START_THREAD);
        while(rxbuf.empty() == false) {
            rxbuf.pop(c);
            if (c == '\n') {
                _nmea_buff[_rx_count % 2][_nmea_buff_count] = c;
                ++_nmea_buff_count;
                _nmea_buff[_rx_count % 2][_nmea_buff_count] = 0;
                ++_rx_count;
                _nmea_buff_count = 0;
                _one_paket_full = true;
                //printf("\r\n");
                break;
            }
            _nmea_buff[_rx_count % 2][_nmea_buff_count] = c;
            if (++_nmea_buff_count >= 256) {
                _nmea_buff_count = 254;
            }
            //printf("%c",c);
        }
        if (_one_paket_full == true) {
            _one_paket_full = false;
            pointer = (_rx_count - 1) % 2;
            if (chk_checksum(_nmea_buff[pointer], sizeof(_nmea_buff[pointer]))
                    == true) {
                _gps_detected = true;
                parse_main(_nmea_buff[pointer]);
            } else {
                _gps_detected = false;
            }
        }
    }
}

void GPSPARSER::parse_main(char *ptr)
{
    //printf("%s",ptr);
    switch (*(ptr+2)) {
        case 'P':
            _taker_id = 0;
            break;
        case 'L':
            _taker_id = 1;
            break;
        case 'A':
            _taker_id = 2;
            break;
        case 'B':
            _taker_id = 3;
            break;
        case 'N':
            _taker_id = 4;
            break;
        default:
            _taker_id = -1;
            break;
    }
    char *p = ptr + 3;  // skip $GP, $GN and so on
    //printf("%s",p);
    if (strncmp (p,"GGA",3) == 0) {
        copy_to_buf(ptr, _gga_buff, sizeof(_gga_buff));
        parseGGA(p);
    } else if (strncmp (p,"GSA",3) == 0) {
        copy_to_buf(ptr, _gsa_buff, sizeof(_gsa_buff));
        parseGSA(p);
    } else if (strncmp (p,"GSV",3) == 0) {
        parseGSV(p);
    } else if (strncmp (p,"GLL",3) == 0) {
        copy_to_buf(ptr, _gll_buff, sizeof(_gll_buff));
        parseGLL(p);
    } else if (strncmp (p,"RMC",3) == 0) {
        ++_message_count;
        copy_to_buf(ptr, _rmc_buff, sizeof(_rmc_buff));
        parseRMC(p);
    } else if (strncmp (p,"VTG",3) == 0) {
        copy_to_buf(ptr, _vtg_buff, sizeof(_vtg_buff));
        parseVTG(p);
    } else if (strncmp (p,"ZDA",3) == 0) {
        parseZDA(p);
    } else {
        printf("Unknown NMEA String Type\r\n");
    }
}

#define c2h(x) (((x) & 64 ? (x) + 9 : (x)) & 15)

bool GPSPARSER::chk_checksum(char *ptr, uint32_t len)
{
    char *p = ptr;
    if (*p++ != '$') {
        return false;
    }
    uint8_t sum = 0;
    for(uint32_t n = 0; n < len; n++) {
        if (*p == '*') {
            break;
        } else {
            sum = sum ^ *p++;
        }
    }
    uint8_t x = c2h(*(p+1)) * 16 + c2h(*(p+2));
    if (x == sum) {
        return true;
    } else {
        printf("Check sum error\r\n");
        return false;
    }
}

bool GPSPARSER::copy_to_buf(char *p1, char *p2, uint32_t len)
{
    char *ps = p1;
    char *pd = p2;
    bool status = false;

    _mutex.lock();
    for (uint32_t i=0;i < len; i++){
        if (*ps == 0){
            *pd = 0;
            if (i != 0){
                status = true;
            } 
            break;
        }
        *pd++ = *ps++;
    }
    _mutex.unlock();
    return status;
    
}

void GPSPARSER::parseGGA(char *ptr)
{
    //printf("$GN%s",ptr);
    // read fix quality
    char* tp = find_comma(ptr, 6);
    _fix_quality = atoi(tp+1);
    // read msl altitude in meters
    tp = find_comma(tp, 3);
    _msl_altitude = atoi(tp+1);
}

void GPSPARSER::parseGSA(char *ptr)
{
    satellite_prn temp;
    static uint32_t old_msg_cnt;

    //printf("+++%s",ptr);
    char* tp = find_comma(ptr, 2);
    _fix_status = atoi(tp+1);
    if (_message_count != old_msg_cnt) {
        old_msg_cnt = _message_count;
        // read satellite PRNs
        for (uint32_t i=0; i<12; i++) {
            tp = find_comma(tp, 1);
            _satellite_prn[i] = atoi(tp+1);
        }
    } else {
        // read satellite PRNs
        for (uint32_t i=0; i<12; i++) {
            tp = find_comma(tp, 1);
            temp[i] = atoi(tp+1);
        }
        uint32_t j=0;
        for (uint32_t i=0; i<12; i++) {
            if (_satellite_prn[i] == 0) {
                if (temp[j] != 0) {
                    _satellite_prn[i] = temp[j++];
                }
            }
        }
    }
}

void GPSPARSER::parseGSV(char *ptr)
{
    //printf("+++%s",ptr);
    char* tp = find_comma(ptr, 3);
    if ((_taker_id >= 0) && (_taker_id < 5)) {
        _num_satellites[_taker_id] = atoi(tp+1);
        //printf("num %d, %d\r\n", _taker_id, _num_satellites[_taker_id]);
    }
}

void GPSPARSER::parseRMC(char *ptr)
{
    //printf("$GN%s",ptr);
    // read timestamp
    char* tp = find_comma(ptr, 1);
    memset(_temp_str,0x00,6);
    strncpy(_temp_str,tp+1,2);
    _mutex.lock();
    _timestamp.tm_hour = atoi(_temp_str);
    memset(_temp_str,0x00,6);
    strncpy(_temp_str,tp+3,2);
    _timestamp.tm_min = atoi(_temp_str);
    memset(_temp_str,0x00,6);
    strncpy(_temp_str,tp+5,2);
    _timestamp.tm_sec = atoi(_temp_str);
    _mutex.unlock();
    // set gps_status  true = active
    tp = find_comma(tp, 1);
    if (*(tp+1) == 'A') {
        _gps_status = true;
    } else {
        _gps_status = false;
    }
    // read latitude degrees minutes
    tp = find_comma(tp, 1);
    memset(_temp_str,0x00,6);
    strncpy(_temp_str,tp+1,2);
    _gps_latitude.degrees = atoi(_temp_str);
    memset(_temp_str,0x00,6);
    strncpy(_temp_str,tp+3,2);
    _gps_latitude.minutes = atoi(_temp_str);
    // read fractional minutes
    tp = next_period(tp);
    _gps_latitude.seconds = (float)atoi(tp+1) * 60.0f / 100000.0f ;
    // read latitude hemisphere change sign if 'S'
    tp = find_comma(tp, 1);
    if (*(tp+1) == 'S') {
        _gps_latitude.degrees *= -1;
    }
    // read longitude degree minutes
    tp = find_comma(tp, 1);
    memset(_temp_str,0x00,6);
    strncpy(_temp_str,tp+1,3);
    _gps_longitude.degrees = atoi(_temp_str);
    memset(_temp_str,0x00,6);
    strncpy(_temp_str,tp+4,2);
    _gps_longitude.minutes = atoi(_temp_str);
    // read fractional minutes
    tp = next_period(tp);
    _gps_longitude.seconds = (float)atoi(tp+1) * 60.0f / 100000.0f ;
    // read longitude hemisphere change sign if 'W'
    tp = find_comma(tp, 1);
    if (*(tp+1) == 'W') {
        _gps_longitude.degrees *= -1;
    }
    // speed and direction
    tp = find_comma(tp, 1);
    _move.speed = atof(tp+1) * 1.852f;
    //printf("speed: %f\r\n", _move.speed);
    tp = find_comma(tp, 1);
    if (*(tp+1) == ',') {
        _move.suspend = true;
        //printf("direction: stop\r\n");
    } else {
        _move.suspend = false;
        _move.direction = atof(tp+1);
        //printf("direction: %f\r\n", _move.direction);
    }
    // read date
    tp = find_comma(tp, 1);
    memset(_temp_str,0x00,6);
    strncpy(_temp_str,tp+1,2);
    _mutex.lock();
    _timestamp.tm_mday = atoi(_temp_str);
    memset(_temp_str,0x00,6);
    strncpy(_temp_str,tp+3,2);
    _timestamp.tm_mon = atoi(_temp_str) - 1;
    memset(_temp_str,0x00,6);
    strncpy(_temp_str,tp+5,2);
    _timestamp.tm_year = atoi(_temp_str) + 100;
    _mutex.unlock();

    _gps_latitude.decimal  = (float)_gps_latitude.degrees +
                             (float)_gps_latitude.minutes / 60.0f +
                             _gps_latitude.seconds / 3600.0f;
    _gps_longitude.decimal = (float)_gps_longitude.degrees +
                             (float)_gps_longitude.minutes / 60.0f +
                             _gps_longitude.seconds / 3600.0f;

    if (_rtc_set_flag == true) {
        time_t seconds = mktime(&_timestamp);
        seconds += _time_offset;
        set_time(seconds);
        _rtc_set_flag = false;
    }
}

void GPSPARSER::parseVTG(char *ptr)
{
    //printf("ParseVTG****\r\n");
    //printf(ptr);
    //printf("\r\n");
}

void GPSPARSER::parseGLL(char *ptr)
{
    //printf("ParseGLL*****\r\n");
    //printf(ptr);
    //printf("\r\n");
}

void GPSPARSER::parseZDA(char *ptr)
{
    //printf("ParseZDA******\r\n");
    //printf(ptr);
    //printf("\r\n");
}

bool GPSPARSER::gpsDetected(void)
{
    return _gps_detected;
}

void GPSPARSER::getLongitude(longitude *lng)
{
    *lng = _gps_longitude;
}

void GPSPARSER::getLatitude(latitude *lat)
{
    *lat = _gps_latitude;
}

void GPSPARSER::getMovement(movement *mv)
{
    *mv = _move;
}

struct tm GPSPARSER::getTimestamp(void)
{
    return _timestamp;
}

bool GPSPARSER::getLockStatus(void)
{
    return _gps_status;
}

uint8_t GPSPARSER::getFixStatus(void)
{
    return _fix_status;
}

uint8_t GPSPARSER::getFixQuality(void)
{
    return _fix_quality;
}

uint8_t GPSPARSER::getNumSatellites(void)
{
    uint32_t n = 0;
    for (uint32_t i=0; i <5; i++) {
        n += _num_satellites[i];
    }
    return (uint8_t)n;
}

int16_t GPSPARSER::getAltitude(void)
{
    return _msl_altitude;
}

bool GPSPARSER::getGGA(char *p)
{
    return copy_to_buf(_gga_buff, p, sizeof(_gga_buff));
}

bool GPSPARSER::getGLL(char *p)
{
    return copy_to_buf(_gll_buff, p, sizeof(_gll_buff));
}

bool GPSPARSER::getGSA(char *p)
{
    return copy_to_buf(_gsa_buff, p, sizeof(_gsa_buff));
}

bool GPSPARSER::getRMC(char *p)
{
    return copy_to_buf(_rmc_buff, p, sizeof(_rmc_buff));
}

bool GPSPARSER::getVTG(char *p)
{
    return copy_to_buf(_vtg_buff, p, sizeof(_vtg_buff));
}

void GPSPARSER::set_gps_time_to_mbed_rtc(float local_offser_hr)
{
    _time_offset = (time_t)(local_offser_hr * 3600.0f);
    _rtc_set_flag = true;
}

time_t GPSPARSER::check_rtc_and_gps_time()
{
    time_t gps_time = mktime(&_timestamp);
    time_t rtc_time = time(NULL);
    return gps_time - rtc_time;
}

void GPSPARSER::getSat_id_list(satellite_prn list)
{
    for (uint32_t i=0; i<12; i++) {
        list[i] = _satellite_prn[i];
    }
}

char * GPSPARSER::find_comma(char *p, uint32_t n)
{
    for (char *px = p;;) {
        px++;
        if ((*px == ',') && (--n == 0)) {
            return px;
        } else if (*px == 0) {
            return NULL;
        }
    }
}

char * GPSPARSER::next_period(char *p)
{
    for (char *px = p;;) {
        px++;
        if (*px == '.') {
            return px;
        } else if (*px == 0) {
            return NULL;
        }
    }
}

/***************** REFRENCE Program *******************************************/
//  https://os.mbed.com/teams/Multi-Hackers/code/GpsParser/
/**
 * @file    NmeaParser.cpp
 * @brief   NMEA String to Packet Parser - NMEA strings to compact packet data
 * @author  Tim Barr
 * @version 1.01
 * @see     http://www.kh-gps.de/nmea.faq
 * @see     http://www.catb.org/gpsd/NMEA.html
 *
 * Copyright (c) 2015
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * 09/16/15 TAB V1.01 Changed report rate of GGA and GSA NMEA sentences
 * 09/22/15 TAB V1.02 Fixed status value for no GPS detected.
 *                    Increased report rate of GSV, GGA, and GSA NMEA Sentences
 * Mar.7th,'19        modified for not only GPS but GNSS
 *                    NEMA -> NMEA
 *
 * TODO: Add speed, compass direction, error (DOP) data. Make init more generic
 */
