/*
 * mbed Application program
 *
 *  Copyright (c) 2010-2014 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created:  May       15th, 2010
 *      Spareted: June      25th, 2014      mon() & mon_hw()
 *      Ported:   July      12th, 2014      from L152RE
 *      Revised:  Feburary   8th, 2015
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
 
//  Include ---------------------------------------------------------------------------------------
#include "mbed.h"
#include "rtos.h"

//  Object ----------------------------------------------------------------------------------------
Serial pc(USBTX, USBRX);

//  Definition ------------------------------------------------------------------------------------
#define BAUD(x)                 pc.baud(x)
#define GETC(x)                 pc.getc(x)
#define PUTC(x)                 pc.putc(x);
#define PRINTF(...)             pc.printf(__VA_ARGS__)
#define READABLE(x)             pc.readable(x)

//  RAM -------------------------------------------------------------------------------------------
static char linebuf[64];

extern float fa[3];    // Acc  0:X, 1:Y, 2:Z
extern float fg[3];    // Gyro 0:X, 1:Y, 2:Z
extern uint8_t show_flag0;
extern uint8_t show_flag1;

//  ROM / Constant data ---------------------------------------------------------------------------
static char *const mon_msg = "Monitor for mbed system, created on "__DATE__"";

//  Function prototypes ---------------------------------------------------------------------------
extern void debug_interface(void);
extern int read_sw(uint8_t n);

//-------------------------------------------------------------------------------------------------
//  Control Program
//-------------------------------------------------------------------------------------------------
//  Put \r\n
static void put_rn ( void ){
    Thread::yield(); // change thread
    PUTC('\r');
    Thread::yield(); // change thread
    PUTC('\n');
}

//  Put \r
static void put_r ( void ){
    Thread::yield(); // change thread
    PUTC('\r');
}

//  Change string -> integer
//int xatoi (char **str, unsigned long *res){
static int xatoi (char **str, int32_t *res){
unsigned long val;
unsigned char c, radix, s = 0;

    while ((c = **str) == ' ') (*str)++;
    if (c == '-') {
        s = 1;
        c = *(++(*str));
    }
    if (c == '0') {
        c = *(++(*str));
        if (c <= ' ') { *res = 0;   return 1; }
        if (c == 'x') {
            radix = 16;
            c = *(++(*str));
        } else {
            if (c == 'b') {
                radix = 2;
                c = *(++(*str));
            } else {
                if ((c >= '0')&&(c <= '9')){    radix = 8;
                }   else {   return 0;}
            }
        }
    } else {
        if ((c < '1')||(c > '9')){  return 0;}
        radix = 10;
    }
    val = 0;
    while (c > ' ') {
        if (c >= 'a') c -= 0x20;
        c -= '0';
        if (c >= 17) {
            c -= 7;
            if (c <= 9) return 0;
        }
        if (c >= radix) return 0;
        val = val * radix + c;
        c = *(++(*str));
    }
    if (s) val = -val;
    *res = val;
    return 1;
}

//-------------------------------------------------------------------------------------------------
//  Monitor
//-------------------------------------------------------------------------------------------------
//  Help Massage
static void msg_hlp (void){
    PRINTF(mon_msg);                        put_rn();
    PRINTF("d - Show control data");        put_rn();
    PRINTF("m - Show mail data");           put_rn();
    PRINTF("s - Show USER Button");         put_rn();
    PRINTF("t - Check and set RTC");        put_rn();
    PRINTF("r - Show RTC date & time");     put_rn();
    PRINTF("x - Goto HW monitor");          put_rn();
    PRINTF("q - Return to main");           put_rn();
}

//  Get key input data
static void get_line (char *buff, int len){
char c;
int idx = 0;

    for (;;) {
        c = GETC();
        //    Added by Kenji Arai / JH1PJL   May 9th, 2010
        if (c == '\r') {
            buff[idx++] = c;
            break;
        }
        if ((c == '\b') && idx) {
            idx--;
            PUTC(c);
            PUTC(' ');
            PUTC(c);
        }
        if (((uint8_t)c >= ' ') && (idx < len - 1)) {
            buff[idx++] = c;
            PUTC(c);
        }
        Thread::yield(); // change thread
        Thread::wait(10); // Wait 10mS
    }
    buff[idx] = 0;
    PUTC('\n');
}

// RTC related subroutines
static void chk_and_set_time(char *ptr){
//unsigned long p1;
int32_t p1;
struct tm t;
time_t seconds;
char buf[40];

    if (xatoi(&ptr, &p1)){
        t.tm_year       = (uint8_t)p1 + 100;
        PRINTF("Year:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_mon        = (uint8_t)p1 - 1;
        PRINTF("Month:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_mday       = (uint8_t)p1;
        PRINTF("Day:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_hour       = (uint8_t)p1;
        PRINTF("Hour:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_min        = (uint8_t)p1;
        PRINTF("Min:%d ",p1);
        xatoi( &ptr, &p1 );
        t.tm_sec        = (uint8_t)p1;
        PRINTF("Sec: %d \r\n",p1);
        seconds = mktime(&t);
        set_time(seconds);
    }
    seconds = time(NULL);
    strftime(buf, 40, "%B %d,'%y, %H:%M:%S", localtime(&seconds));
    PRINTF("Date: %s\r\n", buf);
}

// ---------- Program starts here! ---------------------------------------------------------------
int mon(void) {
char *ptr;

    BAUD(9600);
    put_rn();
    put_rn();
    PRINTF("%s [Help:'?' key]", mon_msg);
    put_rn();
    for (;;) {
        put_r();
        PUTC('>');
        ptr = linebuf;
        get_line(ptr, sizeof(linebuf));
        switch (*ptr++) {
    //---------------------------------------------------------------------------------------------
    //  Check control data
    //---------------------------------------------------------------------------------------------
        case 'd' :
            while(true) {
                put_r();
                PRINTF("G:%+6.1f,%+6.1f,%+6.1f, ", fg[0], fg[1], fg[2]);
                Thread::yield(); // change thread               
                PRINTF("A:%+6.1f,%+6.1f,%+6.1f ", fa[0], fa[1], fa[2]);
                put_rn();
                if ( READABLE() ) {
                    break;
                }
                Thread::wait(200); // Wait 200mS
            }
            break;
    //---------------------------------------------------------------------------------------------
    //  Show Mail data
    //---------------------------------------------------------------------------------------------
        case 'm' :
            put_rn();
            while(true) {
                show_flag0 = 1;
                if ( READABLE() ) {
                    break;
                }
                Thread::wait(100); // Wait 100mS
            }
            show_flag0 = 0;
            break;
    //---------------------------------------------------------------------------------------------
    //  Show switch status
    //---------------------------------------------------------------------------------------------
        case 's' :
            put_r();
            PRINTF("Show USER_Button status");
            put_rn();
            PRINTF("every 500mS  - hit any key for stop");
            put_rn();
            while (true){
                PRINTF("SW0 = ");
                if (read_sw(0) == 1) {  PRINTF("ON ");
                } else {                PRINTF("OFF");
                }
                PRINTF(", SW1 = ");
                if (read_sw(1) == 1) {  PRINTF("ON");
                } else {                PRINTF("OFF");
                }
                put_rn();
                if (READABLE()){ GETC(); break;}
                wait(0.5);
            }
            break;
    //---------------------------------------------------------------------------------------------
    //  check and set RTC
    //---------------------------------------------------------------------------------------------
        case 't' :
            put_r(); 
            chk_and_set_time(ptr);               
            break;
    //---------------------------------------------------------------------------------------------
    //  check and set RTC
    //---------------------------------------------------------------------------------------------
        case 'r' :
            put_r();
            while(true) {
                show_flag1 = 1;
                if ( READABLE() ) {
                    break;
                }
                Thread::wait(50); // Wait 100mS
            }
            show_flag1 = 0;             
            break;
    //---------------------------------------------------------------------------------------------
    //  help
    //---------------------------------------------------------------------------------------------
        case '?' :
            put_r(); 
            msg_hlp();                 
            break;
    //---------------------------------------------------------------------------------------------
    //  Go back to main()
    //---------------------------------------------------------------------------------------------
        case 'q' :        // Quit
            PRINTF("\rReturn to main\r\n");
            PRINTF("cannot control anymore from here\r\n");
            return 0;
        }
    }
}
