/*
 * mbed Application program for the mbed
 *      Test program for GR-PEACH
 *
 * Copyright (c) 2014,'15 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Created: November  29th, 2014
 *      Revised: Feburary   8th, 2015
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

//  Include ---------------------------------------------------------------------------------------
#include    "mbed.h"
#include    "rtos.h"
#include    "L3GD20.h"
#include    "LIS3DH.h"
#include    "TextLCD.h"

//  Definition ------------------------------------------------------------------------------------
#define USE_COM         // use Communication with PC(UART)
#define USE_I2C_LCD
#define USE_I2C_SENSOR
 
// Com
#ifdef  USE_COM
#define BAUD(x)             pcx.baud(x)
#define GETC(x)             pcx.getc(x)
#define PUTC(x)             pcx.putc(x)
#define PRINTF(...)         pcx.printf(__VA_ARGS__)
#define READABLE(x)         pcx.readable(x)
#else
#define BAUD(x)             {;}
#define GETC(x)             {;}
#define PUTC(x)             {;}
#define PRINTF(...)         {;}
#define READABLE(x)         {;}
#endif

#define TIMEBASE        12000
#define FIXED_STEPS     100

#define PI              3.1415926536
#define RAD_TO_DEG      57.29578
#define TIME_BASE_S     0.02
#define TIME_BASE_MS    ( TIME_BASE_S * 1000)
#define RATE            0.1

//  Object ----------------------------------------------------------------------------------------
// LED's
DigitalOut LEDs[4] = {
    DigitalOut(LED1), DigitalOut(LED2), DigitalOut(LED3), DigitalOut(LED4)
};
// Swiches
DigitalIn   USER_SWITCH[2] = {
#if defined(TARGET_RZ_A1H)
    DigitalIn(P6_0),  DigitalIn(P6_1)
#elif defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F411RE)
    DigitalIn(PC_13),  DigitalIn(A0)
#endif
};

// OS check
#if defined(TARGET_RZ_A1H)
DigitalOut task0(P1_8);
DigitalOut task1(P1_9);
#elif defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F411RE)
DigitalOut task0(A0);
DigitalOut task1(A1);
#endif

// com
#ifdef USE_COM
Serial      pcx(USBTX, USBRX);      // Communication with Host
#endif
I2C i2c(D14,D15);
// Gyro
L3GX_GYRO   gyro(i2c, L3GD20_V_CHIP_ADDR, L3GX_DR_95HZ, L3GX_BW_HI, L3GX_FS_250DPS);
// Acc
LIS3DH      acc(i2c, LIS3DH_G_CHIP_ADDR, LIS3DH_DR_NR_LP_50HZ, LIS3DH_FS_8G);
#ifdef USE_I2C_LCD
// LCD
TextLCD_I2C_N lcd0(&i2c, 0x7c, TextLCD::LCD16x2);  // LCD(Akizuki AQM0802A)
#endif
// Mutex
Mutex       i2c_mutex; 

//  RAM -------------------------------------------------------------------------------------------
Queue<uint32_t, 2> queue0;
float fa[3];    // Acc  0:X, 1:Y, 2:Z
float fg[3];    // Gyro 0:X, 1:Y, 2:Z

/* Mail */
typedef struct {
  float    voltage; /* AD result of measured voltage */
  float    current; /* AD result of measured current */
  uint32_t counter; /* A counter value               */
} mail_t;
 
Mail<mail_t, 16> mail_box;

uint8_t show_flag0;
uint8_t show_flag1;
uint32_t count;

#if defined(TARGET_RZ_A1H)
uint8_t big_data[4096*1024];
uint32_t big_data_index = 0;
#endif

//  ROM / Constant data ---------------------------------------------------------------------------

//  Function prototypes ---------------------------------------------------------------------------

//  Function prototypes ---------------------------------------------------------------------------
extern int mon( void);

//-------------------------------------------------------------------------------------------------
//  Control Program
//-------------------------------------------------------------------------------------------------
void send_thread (void const *args) {
    uint32_t i = 0;
    while (true) {
        i++; // fake data update
        mail_t *mail = mail_box.alloc();
        mail->voltage = (i * 0.1) * 33; 
        mail->current = (i * 0.1) * 11;
        mail->counter = i;
        mail_box.put(mail);
        Thread::wait(1000);
    }
}

void blink(void const *n) {
    LEDs[(int)n] = !LEDs[(int)n];
}

// Read switch status
int read_sw(uint8_t n){
    if (USER_SWITCH[n] == 0){   return 1;
    } else {                    return 0;}
}

// Monitor program
void monitor(void const *args){
    Thread::wait(1000);
    while (true){
        mon();
    }
}

void watch_time (void const *args) {
    uint32_t i = 0;
    time_t seconds;
    char buf[64];
#if 0
    struct tm t;

    t.tm_year       = 15 + 100;
    t.tm_mon        = 2 - 1;
    t.tm_mday       = 7;
    t.tm_hour       = 22;
    t.tm_min        = 21;
    t.tm_sec        = 20;
    seconds = mktime(&t);
    set_time(seconds);
#endif
    while (true) {
        seconds = time(NULL);
        strftime(buf, 40, "%B %d,'%y, %H:%M:%S", localtime(&seconds));
        if (show_flag1){
            printf("[TASK1] %s, No:%5d, Ticker:%10u\r\n",buf, i++,  us_ticker_read());
        }
        Thread::wait(1000);
    }
}

// Interrupt routine
void queue_isr0() {
    queue0.put((uint32_t*)1);
}

// Update sensor data
void update_angle(void const *args){
    while (true) {
        osEvent evt = queue0.get();
        // ---->lock
        i2c_mutex.lock();
#ifdef USE_I2C_SENSOR
        // read acceleration data from sensor
        acc.read_data(fa);
        // read gyroscope data from sensor
        gyro.read_data(fg);
#else
        fa[0] = fa[1] = fa[2] = 1.11f;
        fg[0] = fg[1] = fg[2] = 1.11f;
#endif
        // <----unlock
        i2c_mutex.unlock();
        // debug
        task0 = !task0;
    }
}

// Update sensor data
void display(void const *args){
    uint32_t n = 0;

    while (true) {
#ifdef USE_I2C_LCD
        i2c_mutex.lock();
        lcd0.locate(0, 0);    // 1st line top
        lcd0.printf(" G=%4.1f ", sqrt(fa[0]*fa[0] + fa[1]*fa[1] + fa[2]*fa[2]));
        lcd0.locate(0, 1);    // 2nd line top
        lcd0.printf("%8d",n++);
        i2c_mutex.unlock();
#endif
        Thread::wait(500);
        // debug
        task1 = !task1;
    }
}

// Thread definition
osThreadDef(update_angle, osPriorityNormal, 1024);
osThreadDef(monitor, osPriorityNormal, 1024);
osThreadDef(display, osPriorityAboveNormal, 1024);
osThreadDef(watch_time, osPriorityNormal, 1024);

int main(void) {
    PRINTF("\r\nstep1\r\n");
    
    RtosTimer led_1_timer(blink, osTimerPeriodic, (void *)0);
    RtosTimer led_2_timer(blink, osTimerPeriodic, (void *)1);
    RtosTimer led_3_timer(blink, osTimerPeriodic, (void *)2);
    RtosTimer led_4_timer(blink, osTimerPeriodic, (void *)3);

    PRINTF("step2\r\n");    
    led_1_timer.start(2000);
    led_2_timer.start(1000);
    led_3_timer.start(500);
    led_4_timer.start(250);

    PRINTF("step3\r\n");    
    Thread thread(send_thread);

    PRINTF("step4\r\n");

    // IRQ
    Ticker ticker0;
    Ticker ticker1;
    ticker0.attach(queue_isr0, TIME_BASE_S);

 
    PRINTF("step5\r\n");
    // Starts threads
    if (osThreadCreate(osThread(display), NULL) == NULL){
        PRINTF("ERROR4\r\n");
    }
    if (osThreadCreate(osThread(monitor), NULL) == NULL){
        PRINTF("ERROR3\r\n");
    }
    if (osThreadCreate(osThread(update_angle), NULL) == NULL){
        PRINTF("ERROR1\r\n");
    }
    if (osThreadCreate(osThread(watch_time), NULL) == NULL){
        printf("ERROR5\r\n");
    }
    // I2C LCD
#ifdef USE_I2C_LCD
    // ---->lock
    i2c_mutex.lock();
    lcd0.locate(0, 0);    // 1st line top
    lcd0.printf("I2C test");
    lcd0.locate(0, 1);    // 2nd line top
    lcd0.puts(" JH1PJL ");
    lcd0.setContrast(0x14);
    // <----unlock
    i2c_mutex.unlock();
#endif
    count = 0;
    PRINTF("step6\r\n");
    while (true) {
        osEvent evt = mail_box.get();
        if (evt.status == osEventMail) {
            mail_t *mail = (mail_t*)evt.value.p;
            if (show_flag0){
                PRINTF("[MAIN]\r\n");
                PRINTF("This is dummy!, ");
                PRINTF("Volt: %.2f V, "   , mail->voltage);
                PRINTF("Current: %.2f A, "     , mail->current);
                PRINTF("# of cycles: %u\r\n", mail->counter);
            }
            mail_box.free(mail);
        }
#if defined(TARGET_RZ_A1H)
        for (uint32_t i = 0; i < 100; i++){
            big_data[big_data_index] = ++big_data_index;
            if (big_data_index == (4096 * 1024)){
                big_data_index = 0;
            }
        }
#endif
    }
}

void mbed_die(void) {
    PRINTF("Error, came from os_error()!\r\n");
    gpio_t led_1; gpio_init_out(&led_1, LED1);
    gpio_t led_2; gpio_init_out(&led_2, LED2);
    gpio_t led_3; gpio_init_out(&led_3, LED3);
    gpio_t led_4; gpio_init_out(&led_4, LED4);

    while (1) {
        gpio_write(&led_1, 1);
        gpio_write(&led_2, 0);
        gpio_write(&led_3, 0);
        gpio_write(&led_4, 1);
        wait_ms(100);
        gpio_write(&led_1, 0);
        gpio_write(&led_2, 1);
        gpio_write(&led_3, 1);
        gpio_write(&led_4, 0);
        wait_ms(100);
    }
}
