/*
 * mbed Application program for the ST NUCLEO Board
 * Monitor program Ver.3 for only for Renesas GR-PEACH RZ/A1H
 *
 *  Copyright (c) 2010-2014 Kenji Arai / JH1PJL
 *  http://www.page.sannet.ne.jp/kenjia/index.html
 *  http://mbed.org/users/kenjiArai/
 *      Started:  May        9th, 2010
 *      Created:  May       15th, 2010
 *          release as "monitor_01" http://mbed.org/users/kenjiArai/code/monitor_01/
 *      Spareted: June      25th, 2014      mon() & mon_hw()
 *      restart:  July      12th, 2014
 *      Revised:  December  28th, 2014
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 * INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#if defined(TARGET_RZ_A1H)

//  Include ---------------------------------------------------------------------------------------
#include "mbed.h"

//  Object ----------------------------------------------------------------------------------------
extern Serial pch(USBTX, USBRX);

//  Definition ------------------------------------------------------------------------------------
#define BAUD_RATE 9600

#define BAUD(x)             pch.baud(x)
#define GETC(x)             pch.getc(x)
#define PUTC(x)             pch.putc(x)
#define PRINTF(...)         pch.printf(__VA_ARGS__)
#define READABLE(x)         pch.readable(x)

// Range check status
#define ERR_NOTHING         0
#define ERR_MODIFY_SIZ      1
#define ERR_OUT_OF_RANGE    2

// System CLOCK
#define EXTERNAL_XTAL       (13.333f)
#define USB_XTAL            (48.000f)
#define VIDEO_XTAL          (27.000f)

//  RAM -------------------------------------------------------------------------------------------
char linebuf[128];
int buf_size = sizeof(linebuf);

typedef struct {
    int32_t mstr;
    int32_t msiz;
    int32_t mtmp;
    int32_t mold;
    uint8_t  mflg;
    uint8_t  mbhw;
    uint8_t  mmode;
} MEMO;
static MEMO mem;

uint8_t quitflag;

//  ROM / Constant data ---------------------------------------------------------------------------
char *const mon_msg =
    "Monitor only for mbed Renesas GR-PEACH RZ/A1H, created on UTC:"__DATE__"("__TIME__")";

const uint32_t mem_range0[10][2] = {   // Memory access range
    { 0x20000000, 0x200fffff },     // RAM page 0u
    { 0x20100000, 0x201fffff },     // RAM page 1u
    { 0x20200000, 0x202fffff },     // RAM page 2u
    { 0x20300000, 0x203fffff },     // RAM page 3u
    { 0x20400000, 0x204fffff },     // RAM page 4u
    { 0x20500000, 0x205fffff },     // RAM page 0l
    { 0x20600000, 0x206fffff },     // RAM page 1l
    { 0x20700000, 0x207fffff },     // RAM page 2l
    { 0x20800000, 0x208fffff },     // RAM page 3l
    { 0x20900000, 0x209fffff },     // RAM page 4l
};

const uint32_t mem_range1[5][2] = {   // Memory access range
    { 0x60000000, 0x601fffff },     // RAM page 0
    { 0x60200000, 0x603fffff },     // RAM page 1
    { 0x60400000, 0x605fffff },     // RAM page 2
    { 0x60600000, 0x607fffff },     // RAM page 3
    { 0x60800000, 0x609fffff },     // RAM page 4
};

char *const hmsg0 = "Enter Memory Mode d <address> [<count>], s, <ret> or n, q, ?";

char *const rmsg_0 = "RAM page0u:";
char *const rmsg_1 = "RAM page0l:";
char *const rmsg_2 = "RAM page1u:";
char *const rmsg_3 = "RAM page1l:";
char *const rmsg_4 = "RAM page2u:";
char *const rmsg_5 = "RAM page2l:";
char *const rmsg_6 = "RAM page3u:";
char *const rmsg_7 = "RAM page3l:";
char *const rmsg_8 = "RAM page4u:";
char *const rmsg_9 = "RAM page4l:";

char *const rmsg0 = "RAM(Mirror) page0:";
char *const rmsg1 = "RAM(Mirror) page1:";
char *const rmsg2 = "RAM(Mirror) page2:";
char *const rmsg3 = "RAM(Mirror) page3:";
char *const rmsg4 = "RAM(Mirror) page4:";

char *const fmsg0 = "Unknown setting";

char *const msg0 = "m  - Entry Memory Mode";
char *const msg1 = "m>? -> Aditinal functions can see by ?";
char *const msg2 = "r  - Entry Register Mode";
char *const msg3 = "r>? -> Aditinal functions can see by ?";
char *const msg4 = "s  - System Clock -> sf, System / CPU information -> sc";
char *const msg5 = "q  - Quit (back to called routine)";
char *const msg6 = "p  - Entry Port Mode";
char *const msg7 = "p>? -> Aditinal functions can see by ?";

char *const mrmsg0 = "Enter Register Mode u,i,s,t,a,d,l,w,c & ? for help";
char *const mrmsg8 = "Return to All Mode";

//  Function prototypes ---------------------------------------------------------------------------
extern void put_rn ( void );
extern void put_r ( void );
extern void put_lin ( void );
extern void put_spc( uint8_t n);
extern void get_line (char *buff, int len);
extern int xatoi (char **str, int32_t *res);

extern void print_data_address(void);

//-------------------------------------------------------------------------------------------------
//  Control Program
//-------------------------------------------------------------------------------------------------
// No function
static void not_yet_impliment( void )
{
    PRINTF("Not implimented yet");
    put_rn();
}

void not_select( void )
{
    PRINTF("Not select the function");
    put_rn();
}

//  Put \r\n
void put_rn ( void )
{
    PUTC('\r');
    PUTC('\n');
}

//  Put \r
void put_r ( void )
{
    PUTC('\r');
}

// Put ", "
void put_lin ( void )
{
    PRINTF(", ");
}

// Put space n
void put_spc( uint8_t n)
{
    for(; n > 0; n--) {
        PUTC(' ');
    }
}

//  Change string -> integer
int xatoi (char **str, int32_t *res)
{
    unsigned long val;
    unsigned char c, radix, s = 0;

    while ((c = **str) == ' ') (*str)++;
    if (c == '-') {
        s = 1;
        c = *(++(*str));
    }
    if (c == '0') {
        c = *(++(*str));
        if (c <= ' ') {
            *res = 0;
            return 1;
        }
        if (c == 'x') {
            radix = 16;
            c = *(++(*str));
        } else {
            if (c == 'b') {
                radix = 2;
                c = *(++(*str));
            } else {
                if ((c >= '0')&&(c <= '9')) {
                    radix = 8;
                }   else {
                    return 0;
                }
            }
        }
    } else {
        if ((c < '1')||(c > '9')) {
            return 0;
        }
        radix = 10;
    }
    val = 0;
    while (c > ' ') {
        if (c >= 'a') c -= 0x20;
        c -= '0';
        if (c >= 17) {
            c -= 7;
            if (c <= 9) return 0;
        }
        if (c >= radix) return 0;
        val = val * radix + c;
        c = *(++(*str));
    }
    if (s) val = -val;
    *res = val;
    return 1;
}

//  Get key input data
void get_line (char *buff, int len)
{
    char c;
    int idx = 0;

    for (;;) {
        c = GETC();
        if (c == '\r') {
            buff[idx++] = c;
            break;
        }
        if ((c == '\b') && idx) {
            idx--;
            PUTC(c);
            PUTC(' ');
            PUTC(c);
        }
        if (((uint8_t)c >= ' ') && (idx < len - 1)) {
            buff[idx++] = c;
            PUTC(c);
        }
    }
    buff[idx] = 0;
    PUTC('\n');
}


//  Range check for Memory dump
void check_range( MEMO * mem )
{
    uint8_t  i;
    uint32_t m;

    mem->mflg = ERR_NOTHING;
    if (mem->mmode) {
        for (i = 0 ; i < 5; i++) {
            if ( mem->mstr >= mem_range1[i][0]) {
                if ( mem->mstr < mem_range1[i][1] ) {
                    m = mem->mstr + mem->msiz;
                    if ( m < mem_range1[i][1]) {
                        return;            // no modification
                    } else {
                        m = mem_range1[i][1];
                        mem->msiz = m - mem->mstr + 1;
                        mem->mflg = ERR_MODIFY_SIZ;
                        return;            // modified size
                    }
                }
            }
        }
    } else {
        for (i = 0; i < 10; i++) {
            if ( mem->mstr >= mem_range0[i][0]) {
                if ( mem->mstr < mem_range0[i][1] ) {
                    m = mem->mstr + mem->msiz;
                    if ( m < mem_range0[i][1]) {
                        return;            // no modification
                    } else {
                        m = mem_range0[i][1];
                        mem->msiz = m - mem->mstr + 1;
                        mem->mflg = ERR_MODIFY_SIZ;
                        return;            // modified size
                    }
                }
            }
        }
    }
    mem->mflg = ERR_OUT_OF_RANGE;
    mem->mstr = 0;
    mem->msiz = 0;
    return ;
}

//  Memory dump error massage
void error_print ( unsigned char flg )
{
    switch (flg) {
        case ERR_MODIFY_SIZ :
            put_r();
            PRINTF("Reach to out of range");
            put_rn();
            break;
        case ERR_OUT_OF_RANGE :
            put_r();
            PRINTF("Not in a memory area");
            put_rn();
            break;
        case ERR_NOTHING :
        default :
            ;
    }
}

//  Print memory contents
void put_dump (const unsigned char *buff, unsigned long ofs, int cnt)
{
    int n;

    PRINTF("%08lX ", ofs);
    for(n = 0; n < cnt; n++) {      // show hex
        PRINTF(" %02X", buff[n]);
    }
    for(; n < 16; n++) {            // fullfil remained space
        PRINTF("   ");
    }
    PUTC(' ');
    for(n = 0; n < cnt; n++) {      // show char
        if ((buff[n] < 0x20)||(buff[n] >= 0x7F)) {
            PUTC('.');
        } else {
            PUTC(buff[n]);
        }
    }
    put_rn();
}

// dump memory with error check
uint8_t dump_w_err_ckeck ( char **ptr, MEMO * mem )
{
    check_range (mem);
    for (*ptr=(char*)mem->mstr; mem->msiz >= 16; *ptr += 16, mem->msiz -= 16) {
        put_r();
        put_dump((unsigned char*)*ptr, (unsigned int)*ptr, 16);
        if (READABLE()) {
            mem->mstr = (int32_t)*ptr;
            PRINTF("Stop");
            uint8_t c = GETC();
            put_rn();
            return 0;
        }
    }
    if (mem->msiz) {
        put_dump((unsigned char*)*ptr, (unsigned int)*ptr, mem->msiz);
    }
    error_print(mem->mflg);
    return 1;
}

void set_page(char *ptr, uint8_t n)
{
    if (mem.mmode) {
        mem.mstr = mem_range1[n][0];
    } else {
        mem.mstr = mem_range0[n][0]; ;
    }
    mem.msiz =512;
    mem.mold = mem.mstr;
    mem.mtmp = 0;
    mem.mflg = 0;
    if (dump_w_err_ckeck(&ptr, &mem)) {
        mem.mstr += 512;
    }
}

static void mem_inf (char *ptr)
{
    PRINTF(hmsg0);
    put_rn();
    mem.mstr = mem_range0[0][0];     // default start address = Flash
    mem.msiz =512;
    mem.mold = 0;
    mem.mtmp = 0;
    mem.mflg = 0;
    mem.mmode = 0;  // base address mode
    for (; mem.mflg != 0xff;) {
        PRINTF("m>");
        ptr = linebuf;
        get_line(ptr, buf_size);
        put_r();
        switch(*ptr++) {
            case 'c' :
                mem.mstr = mem.mold;
                mem.msiz = 0xfffff;
                mem.mtmp = mem.msiz;
                dump_w_err_ckeck(&ptr, &mem);
                mem.mold = mem.mstr;
                break;
            case 'd' :    // d <address> [<count>] - Dump memory
                mem.mtmp = mem.mstr;
                if (!xatoi(&ptr, &mem.mstr)) {
                    mem.mstr = mem.mtmp;
                }
                if (!xatoi(&ptr, &mem.msiz)) {
                    mem.msiz = 512;
                }
                mem.mtmp = mem.msiz;
                if (dump_w_err_ckeck(&ptr, &mem)) {
                    mem.mstr += mem.mtmp;
                }
                mem.mold = mem.mstr;
                break;
            case 'n' :
            case 0x0d :
                mem.msiz = 512;
                mem.mtmp = mem.msiz;
                if (dump_w_err_ckeck(&ptr, &mem)) {
                    mem.mstr += 512;
                }
                mem.mold = mem.mstr;
                break;
            case 'b' :        // Back to more
                if (mem.mold == 0) {
                    ;
                } else {
                    mem.mold -= 512;
                }
            case '0' :        // start RAM page0 top
                if (mem.mmode) {
                    set_page(ptr, 0);
                } else {
                    if (*ptr == 'l') {
                        set_page(ptr, 5);
                    } else {
                        set_page(ptr, 0);
                    }
                }
                break;
            case '1' :        // start RAM page1 top
                if (mem.mmode) {
                    set_page(ptr, 1);
                } else {
                    if (*ptr == 'l') {
                        set_page(ptr, 6);
                    } else {
                        set_page(ptr, 1);
                    }
                }
                break;
            case '2' :        // start RAM page2 top
                if (mem.mmode) {
                    set_page(ptr, 2);
                } else {
                    if (*ptr == 'l') {
                        set_page(ptr, 7);
                    } else {
                        set_page(ptr, 2);
                    }
                }
                break;
            case '3' :        // start RAM page3 top
                if (mem.mmode) {
                    set_page(ptr, 3);
                } else {
                    if (*ptr == 'l') {
                        set_page(ptr, 8);
                    } else {
                        set_page(ptr, 3);
                    }
                }
                break;
            case '4' :        // start RAM page top
                if (mem.mmode) {
                    set_page(ptr, 4);
                } else {
                    if (*ptr == 'l') {
                        set_page(ptr, 9);
                    } else {
                        set_page(ptr, 4);
                    }
                }
                break;
            case 'm' :        // Change RAM address mode (Base or Mirror)
                PRINTF("Currenr ");
                if (mem.mmode == 0) {
                    PRINTF("BASE mode");
                } else {
                    PRINTF("Mirror mode");
                }
                put_rn();
                PRINTF("Do you want to change the mode? y or n");
                put_rn();
                if (GETC() == 'y') {
                    if (mem.mmode == 0) {
                        mem.mmode = 1;
                    } else {
                        mem.mmode = 0;
                    }
                    PRINTF("Set ");
                    if (mem.mmode == 0) {
                        PRINTF("BASE mode");
                    } else {
                        PRINTF("Mirror mode");
                    }
                    put_rn();
                }
                if (mem.mmode) {
                    mem.mstr = mem_range1[0][0];
                } else {
                    mem.mstr = mem_range0[0][0]; ;
                }
                break;
            case 's' :
                PRINTF("Memory Configuration");
                put_rn();
                if (mem.mmode) {
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg0, mem_range1[0][0], mem_range1[0][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg1, mem_range1[1][0], mem_range1[1][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg2, mem_range1[2][0], mem_range1[2][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg3, mem_range1[3][0], mem_range1[3][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg4, mem_range1[4][0], mem_range1[4][1]);
                } else {
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg_0, mem_range0[0][0], mem_range0[0][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg_1, mem_range0[5][0], mem_range0[5][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg_2, mem_range0[1][0], mem_range0[1][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg_3, mem_range0[6][0], mem_range0[6][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg_4, mem_range0[2][0], mem_range0[2][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg_5, mem_range0[7][0], mem_range0[7][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg_6, mem_range0[3][0], mem_range0[3][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg_7, mem_range0[8][0], mem_range0[8][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg_8, mem_range0[4][0], mem_range0[4][1]);
                    put_rn();
                    PRINTF("%s0x%08lx to 0x%08lx ", rmsg_9, mem_range0[9][0], mem_range0[9][1]);
                }
                put_rn();
                break;
            case 'q' :        // quit
                mem.mflg = 0xff;
                break;
            case '?' :
                PRINTF("d <address> [<count>] - Dump memory");
                put_rn();
                PRINTF("s  - Show RAM structure ");
                put_rn();
                PRINTF("0u,0l,1u,1l,... or 0,1,2,3,4 - Dump RAM data");
                put_rn();
                PRINTF("b  - Dump memory / before 512bytes");
                put_rn();
                PRINTF("c  - Dump memory until hit any key");
                put_rn();
                PRINTF("m  - Change RAM address mode (Base or Mirror)");
                put_rn();
                PRINTF("<RET> or n - Dump memory / next 512bytes");
                put_rn();
                PRINTF("q  - return");
                put_rn();
                break;
            default:
                PUTC('?');
                put_rn();
        }
#if 0
        PRINTF("mstr:0x%08x, msiz:0x%08x, mtmp:0x%08x\r\n",
                                mem.mstr, mem.msiz, mem.mtmp);
        PRINTF("mold:0x%08x, mflg:%d, mbhw:%d, mmode:%d\r\n",
                                mem.mold, mem.mflg, mem.mbhw, mem.mmode);
#endif
    }
}

void system_requency(void)
{
    uint16_t reg0, reg1;

    // Hardware configration (mbed GR-PEACH)= P0_2/MD_CLK -> LOW
    reg0 = CPG.FRQCR;
    reg1 = CPG.FRQCR2;
    PRINTF("FRQCR:0x%04x", reg0);
    put_rn();
    PRINTF("FRQCR:0x%04x", reg1);
    put_rn();
    if (reg1 == 1) {
        PRINTF("EXternal Xtal:%10.6f [MHz]", EXTERNAL_XTAL);
        put_rn();
        PRINTF("PLL Clock    :%10.6f [MHz]", EXTERNAL_XTAL * 30);
        put_rn();
        reg0 &= 0x3ff;
        if (reg0 == 0x035) {
            PRINTF("CPU Clock   I:%10.6f [MHz]", EXTERNAL_XTAL * 30);
        } else if (reg0 == 0x135) {
            PRINTF("CPU Clock   I:%10.6f [MHz]", EXTERNAL_XTAL * 20);
        } else {
            PRINTF(fmsg0);
            return;
        }
        put_rn();
        PRINTF("Graphic     G:%10.6f [MHz]", EXTERNAL_XTAL * 20);
    } else if (reg1 == 3) {
        if (reg0 == 0x035) {
            PRINTF("CPU Clock   I:%10.6f [MHz]", EXTERNAL_XTAL * 30);
        } else if (reg0 == 0x135) {
            PRINTF("CPU Clock   I:%10.6f [MHz]", EXTERNAL_XTAL * 20);
        } else if (reg0 == 0x135) {
            PRINTF("CPU Clock   I:%10.6f [MHz]", EXTERNAL_XTAL * 10);
        } else {
            PRINTF(fmsg0);
            return;
        }
        put_rn();
        PRINTF("Graphic     G:%10.6f [MHz]", EXTERNAL_XTAL * 10);
    } else {
        PRINTF(fmsg0);
        return;
    }
    put_rn();
    PRINTF("Bus Clock   B:%10.6f [MHz]", EXTERNAL_XTAL * 10);
    put_rn();
    PRINTF("Peripheral P1:%10.6f [MHz]", EXTERNAL_XTAL * 5);
    put_rn();
    PRINTF("Peripheral P0:%10.6f [MHz]", EXTERNAL_XTAL * 5 / 2);
    put_rn();
}

//-------------------------------------------------------------------------------------------------
//  Monitor Main Program
//-------------------------------------------------------------------------------------------------
//  Help Massage
void mon_msg_hlp ( void )
{
    PRINTF(mon_msg);
    put_rn();
    PRINTF(msg0);
    put_rn();
    PRINTF(msg1);
    put_rn();
    PRINTF(msg6);
    put_rn();
    PRINTF(msg7);
    put_rn();
    PRINTF(msg2);
    put_rn();
    PRINTF(msg3);
    put_rn();
    PRINTF(msg4);
    put_rn();
    PRINTF(msg5);
    put_rn();
}

int mon_mem(void)
{
    char *ptr;

    put_r();
    PRINTF("%s [Help:'?' key]", mon_msg);
    put_rn();
    for (;;) {
        put_r();
        PUTC('>');
        ptr = linebuf;
        get_line(ptr, buf_size);
        put_r();
        switch (*ptr++) {
                //-----------------------------------------------------------------------------------------
                //    Memory
                //-----------------------------------------------------------------------------------------
            case 'm' :
                mem_inf(ptr);
                break;
                //-----------------------------------------------------------------------------------------
                //    Register
                //-----------------------------------------------------------------------------------------
            case 'r' :
                put_r();
                PRINTF(mrmsg0);
                put_rn();
                quitflag = 0;
                for (; quitflag != 0xff;) {
                    PRINTF("r>");
                    ptr = linebuf;
                    get_line(ptr, buf_size);
                    put_r();
                    switch(*ptr++) {
                        case 'u' :
#if USE_UART

#else
                            not_select();
#endif // USE_UART
                            break;
                        case 'i' :
#if USE_I2C

#else
                            not_select();
#endif // USE_I2C
                            break;
                        case 's' :
#if USE_SPI

#else
                            not_select();
#endif // USE_SPI
                            break;
                        case 't' :      //
                            not_yet_impliment();
                            break;
                        case 'a' :      //
                            not_yet_impliment();
                            break;
                        case 'd' :      //
                            not_yet_impliment();
                            break;
                        case 'w' :      //
                            not_yet_impliment();
                            break;
                        case 'l' :        //
                            not_yet_impliment();
                            break;
                        case 'c' :      //
                            not_yet_impliment();
                            break;
                        case 'x' :      //
                            not_yet_impliment();
                            break;
                        case 'y' :      //
                            not_yet_impliment();
                            break;
                        case '?' :
                            PRINTF("u - USART");
                            put_rn();
                            PRINTF("i - I2C");
                            put_rn();
                            PRINTF("s - SPI");
                            put_rn();
                            PRINTF("t - TIMER");
                            put_rn();
                            PRINTF("a - ADC");
                            put_rn();
                            PRINTF("d - DAC");
                            put_rn();
                            PRINTF("l - LDC");
                            put_rn();
                            PRINTF("w - WWDG");
                            put_rn();
                            PRINTF("c - COMP");
                            put_rn();
                            break;
                        case 'q' :        // quit
                            quitflag = 0xff;
                            break;
                        default:
                            PUTC('?');
                            put_rn();
                    }
                }
                PRINTF(mrmsg8);
                put_rn();
                break;
                //-----------------------------------------------------------------------------------------
                //    Port
                //-----------------------------------------------------------------------------------------
            case 'p' :
#if USE_PORT

#else
                not_select();
#endif // USE_PORT
                break;
                //-----------------------------------------------------------------------------------------
                //    System
                //-----------------------------------------------------------------------------------------
            case 's' :    // System related information
                system_requency();
                break;
                //-----------------------------------------------------------------------------------------
                //    Help
                //-----------------------------------------------------------------------------------------
            case '?' :
                mon_msg_hlp();
                break;
                //-----------------------------------------------------------------------------------------
                //    Return to main routine
                //-----------------------------------------------------------------------------------------
            case 'q' :        // Quit
                put_r();
                PRINTF("Return to monitor");
                put_rn();
                return 0;
                //-----------------------------------------------------------------------------------------
                //    Special command for DEBUG
                //-----------------------------------------------------------------------------------------
            case 'x' :
#if 1
                print_data_address();
#else
                not_yet_impliment();
#endif
                break;
                //-----------------------------------------------------------------------------------------
                //    no support
                //-----------------------------------------------------------------------------------------
            default:
                PUTC('?');
                put_rn();
                break;
        }
    }
}

#endif  // defined(TARGET_RZ_A1H)
