/*
 * mbed Application program / Frequency Counter
 *
 * Copyright (c) 2014,'15,'20 Kenji Arai / JH1PJL
 *      http://www7b.biglobe.ne.jp/~kenjia/
 *      https://os.mbed.com/users/kenjiArai/
 *          Created: October   18th, 2014
 *          Revised: January   13th, 2020
 *
 */

/*
    checked on
        Nucleo-F401RE, F411RE & F446RE
 */

//  Include --------------------------------------------------------------------
#include "mbed.h"
#include "freq_counter.h"

//  Definition -----------------------------------------------------------------
#define GATE_TIME 1.0f
//  Object ---------------------------------------------------------------------
DigitalOut led(LED1);
// PA_8 & PC_9 uses for MCO_1 % MCO_2 -> Clock output for checking
DigitalOut osc1(PA_8);
DigitalOut osc2(PC_9);
Serial pc(USBTX, USBRX);
F_COUNTER fc(PA_0, GATE_TIME);
//F_COUNTER fc(PA_1, GATE_TIME);
//F_COUNTER fc(PB_3, GATE_TIME);


//  RAM ------------------------------------------------------------------------

//  ROM / Constant data --------------------------------------------------------
const float gate_time_select[] = {1.0f, 2.0f, 5.0f, 10.0f, 0.5f, 0.1f};

//  Function prototypes --------------------------------------------------------
void port_mco1_mco2_set(void);

//------------------------------------------------------------------------------
//  Control Program
//------------------------------------------------------------------------------
int main()
{
    pc.printf("\r\nStart Frequency Counter\r\n");
    port_mco1_mco2_set();   // Set Internalclock for reference
    float t_gate = GATE_TIME;
    float freqency = 0;
    uint32_t pin = fc.read_pin();
    pc.printf("Signal input pin ");
    if (pin == PA_0) {
        pc.printf("= PA_0(A0)\r\n");
    } else if (pin == PA_1) {
        pc.printf("= PA_1(A1)\r\n");
    } else if (pin == PB_3) {
        pc.printf("= PB_3(D3)\r\n");
    } else {
        pc.printf("is NOT correct!!\r\n");
    }
    uint32_t size_of_table = sizeof(gate_time_select) / sizeof(float);
    pc.printf("# of parameter %d\r\n", size_of_table);
    while(true) {
        for (uint32_t i = 0; i < size_of_table; i++) {
            t_gate = gate_time_select[i];
            pc.printf("Change gate time : %5.2f [Sec]\r\n", t_gate);
            fc.set_gate_time(t_gate);
            for (uint32_t i = 0; i < 5; i++) {
                led = !led;
                freqency = (float)fc.read_frequency() / t_gate;
                pc.printf("f= %10.0f [Hz], gate= %5.2f [Sec]\r\n",
                          freqency, t_gate);
            }
        }
    }
}

void port_mco1_mco2_set(void)
{
    uint32_t temp = 0x00;

    SystemCoreClockUpdate();
    // PA_8 -> MCO_1
    temp = ((uint32_t)(GPIO_AF0_MCO) << (((uint32_t)8 & (uint32_t)0x07) * 4)) ;
    GPIOA->AFR[8 >> 3] &=
        ~((uint32_t)0xf << ((uint32_t)(8 & (uint32_t)0x07) * 4)) ;
    GPIOA->AFR[8 >> 3] |= temp;
    GPIOA->MODER &= ~(GPIO_MODER_MODER0 << (8 * 2));
    GPIOA->MODER |= (0x2 << (8 * 2));
    GPIOA->OSPEEDR |= (0x03 << (8 * 2)); // High speed
    // PC_9 -> MCO_2
    temp = ((uint32_t)(GPIO_AF0_MCO) << (((uint32_t)9 & (uint32_t)0x07) * 4)) ;
    GPIOC->AFR[9 >> 3] &=
        ~((uint32_t)0xf << ((uint32_t)(9 & (uint32_t)0x07) * 4)) ;
    GPIOC->AFR[9 >> 3] |= temp;
    GPIOC->MODER &= ~(GPIO_MODER_MODER0 << (9 * 2));
    GPIOC->MODER |= (0x2 << (9 * 2));
    GPIOC->OSPEEDR |= (0x03 << (9 * 2)); // High speed
    // Select output clock source
    RCC->CFGR &= 0x009fffff;
    pc.printf("System clock = %d [Hz], HSE = %d [Hz]\r\n",
              SystemCoreClock, HSE_VALUE);
#if 1
    // MCO_1 output HSE 1/4, MCO_2 output SYSCLK 1/4
    //             MCO2          MCO2PRE       MCO1PRE       MCO1
    RCC->CFGR |= (0x0 << 30) + (0x6 << 27) + (0x6 << 24) + (0x3 << 22);
    pc.printf("PA_8(MCO_1) = %d [Hz], PC_9(MCO_2) = %d [Hz]\r\n",
              HSE_VALUE / 4, SystemCoreClock / 4);
#else
    // MCO_1 output HSE 1/1, MCO_2 output SYSCLK 1/2
    //             MCO2          MCO2PRE       MCO1PRE       MCO1
    RCC->CFGR |= (0x0 << 30) + (0x4 << 27) + (0x0 << 24) + (0x3 << 22);
    pc.printf("PA_8(MCO_1) = %d [Hz], PC_9(MCO_2) = %d [Hz]\r\n",
              HSE_VALUE, SystemCoreClock / 2);
#endif
}
